/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.plugin.column;

import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.db.AbstractEditor;
import org.opengion.hayabusa.db.CellEditor;
import org.opengion.hayabusa.db.DBColumn;
import org.opengion.fukurou.util.XHTMLTag;
import org.opengion.fukurou.util.TagBuffer;
import org.opengion.fukurou.util.StringUtil;									// 8.0.2.0 (2021/11/30) Add

import static org.opengion.fukurou.util.StringUtil.isNull;						// 6.1.1.0 (2015/01/17)

/**
 * YMD エディターは、カラムのデータを日付(年/月/日)編集する場合に使用するクラスです。
 * YMD2はカレンダーのポップアップボタンが付属するタイプです。
 *
 * このエディタはeventColumnに対応していません。
 *
 * カラムの表示に必要な属性は, DBColumn オブジェクト より取り出します。
 * このクラスは、DBColumn オブジェクト毎に１つ作成されます。
 *
 * 8.0.2.0 (2021/11/30)
 * カラムのパラメータの情報より、日付送り戻しの機能が使えます。
 * 例：D-1,D0,D+1 or D-1,D0,D+1,true
 *
 * 第一引数は、日付戻しを指定します。
 * 第二引数は、初期化します。
 * 第三引数は、日付送りを指定します。
 * 第四引数は、検索ボタンを押すか押さないか[true/false]を指定します。 (初期値:false)
 *
 * 日付についての加減算処理を行うためのコマンドを指定します。
 *  ・SYXX ：年の最初の日付を指定の分だけ進めます。(SY-1なら先年の1月1日、SY1なら翌年の1月1日)
 *  ・SDXX ：月の最初の日付を指定の分だけ進めます。(SD-1なら先月の1日、SD1なら翌月の1日)
 *  ・SWXX ：週初め(月曜日)を指定の分だけ進めます。(SW-1なら先週の月曜日、SW1なら翌週の月曜日)
 *  ・EYXX ：年の最後の日付を指定の分だけ進めます。(EY-1なら先年の年末、EY1なら翌年の年末)
 *  ・EDXX ：月の最後の日付を指定の分だけ進めます。(ED-1なら先月の月末、ED1なら翌月の月末)
 *  ・EWXX ：週末(日曜日)を指定の分だけ進めます。(EW-1なら先週の日曜日、EW1なら翌週の日曜日)
 *  ・YXX  ：年を指定の分だけ進めます。(Y-1なら1年前、Y1なら1年後)
 *  ・MXX  ：月を指定の分だけ進めます。(M-1なら1月前、M1なら1月後)
 *  ・DXX  ：日を指定の分だけ進めます。(D-1なら1日前、D1なら1日後)
 *  ※ 数字がゼロのコマンドは初期化します。
 *  ※ 数字がないコマンドはサーバー上のシステム日付をセットします。
 *
 * @og.rev 3.5.6.2 (2004/07/05) 新規作成
 * @og.rev 5.4.3.6 (2012/01/19) コメント修正
 * @og.rev 5.6.5.2 (2013/06/21) ポップアップ変更
 * @og.rev 5.9.4.0 (2016/01/08) type=button追加,背景色transparent
 * @og.rev 5.9.4.3 (2016/01/15) buttonのpadding:0
 * @og.rev 7.3.2.3 (2021/04/09) ｼｽﾃﾑ定数のICON_DIRやIMAGE_DIRを使用します。
 * @og.rev 8.0.2.0 (2021/11/30) 日付送り戻し対応
 *
 * @og.group データ編集
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class Editor_YMD2 extends AbstractEditor {
	/** このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "8.0.2.0 (2021/11/30)" ;

	// 5.6.5.2 (2013/06/21) htmlからjspに変更
	// 6.0.2.5 (2014/10/31) height,width に、"px"を追加
	// 5.9.4.0 (2016/01/09) type=button追加
	// 6.5.0.1 (2016/10/21) サイズ変更 '250px','265px' → '265px','260px'
	// 4.3.6.7 (2009/05/22) FireFox対応
	// 5.9.4.0 背景色transparent 5.9.4.3 padding
	// 6.4.2.0 (2016/01/29) alt属性にtitle属性を追記。

	// 7.3.2.3 (2021/04/09) ｼｽﾃﾑ定数のICON_DIRやIMAGE_DIRを使用します。
	private static final String JSP      =  HybsSystem.sys( "JSP" ) ;
	private static final String JSP_ICON =  HybsSystem.sys( "JSP_ICON" ) ;

//	private static final String CAL1 = "<button type=\"button\" onclick=\"ogPopup('../common/calendar.jsp','265px','260px',null,this,new Array('" ;
	private static final String CAL1 = "<button type=\"button\" onclick=\"ogPopup('" + JSP + "/common/calendar.jsp','265px','260px',null,this,new Array('" ;

	// 6.8.5.0 (2018/01/09)
//	private static final String CAL3 = "<button type=\"button\" onclick=\"ogPopup('../common/calendar_3month.jsp','320px','770px',null,this,new Array('" ;
	private static final String CAL3 = "<button type=\"button\" onclick=\"ogPopup('" + JSP + "/common/calendar_3month.jsp','320px','770px',null,this,new Array('" ;

	// 6.8.5.0 (2018/01/09) とりあえず、CAL1 と CAL3 しかないので、決め打ち
	// 6.9.8.0 (2018/05/28) キーワード変更(CAL_ST → CAL_STR)
	private static final String CAL_STR = "CAL1".equals( HybsSystem.sys( "CALENDAR_POPUP_TYPE" ) ) ? CAL1 : CAL3 ;

//	// 6.8.5.0 (2018/01/09) キーワード変更(CAL2 → CAL_END)
//	private static final String CAL_END = "'),event); return false;\" style=\"background-color:transparent;border:0px;padding:0;\">"
//												+ "<img src=\"../image/calendar.gif\" alt=\"Calendar\" title=\"Calendar\"/></button>";
	// 5.9.32.3command,p_appendの対応
	// 7.0.1.0 (2018/10/15) XHTML → HTML5 対応(空要素の、"／＞" 止めを、"＞" に変更します)。
	// 7.4.2.2 (2021/05/28) タブレット用にボタンを大きくしたので、元に戻します。
//	private static final String CAL_END = "'),'NEW',false,event); return false;\" style=\"background-color:transparent;border:0px;padding:0;\">"
	private static final String CAL_END = "'),'NEW',false,event); return false;\" style=\"background-color:transparent;border:0px;padding:0;min-width:0;margin:0;\">"
//												+ "<img src=\"../image/calendar.gif\" alt=\"Calendar\"/></button>";
//												+ "<img src=\"../image/calendar.gif\" alt=\"Calendar\"></button>";
												+ "<img src=\"" + JSP_ICON + "/calendar.gif\" alt=\"Calendar\"></button>";

	// 8.0.2.0 (2021/11/30) 日付送り戻し対応
	private static final int CNT_ARY		= 3;
	private static final String FR_STR		= "<a href=\"#\" onClick=\"dateFeedRtn('%1$s','%2$s','%3$s','%4$s','%5$s');\" style=\"margin:0 5px 0 3px\" >";
	private static final String FR_END		= "<img src=\"" + JSP_ICON + "/%6$s\" alt=\"%4$s\" title=\"%4$s\" /></a>";
	private static final String[] FR_IMG	= { "FR_PREV.png", "FR_CIRCLE.png", "FR_NEXT.png" };

	private String isSubm					= "false";							// 検索ボタンを押すか押さないか
	private String errMsg;														// エラーメッセージ
	private String[] prmAry;													// 編集パラメータ

	// 6.9.8.0 (2018/05/28) disabled 時に、日付ピッカーを出さない対応
	private final boolean hidden;												// 8.0.2.0 (2021/11/30) Modify disabled ⇒ hidden

	/**
	 * デフォルトコンストラクター。
	 * このコンストラクターで、基本オブジェクトを作成します。
	 *
	 * @og.rev 6.9.8.0 (2018/05/28) disabled 時に、日付ピッカーを出さない対応
	 * @og.rev 8.0.2.0 (2021/11/30) 日付送り戻し対応
	 */
//	public Editor_YMD2() { super(); }		// これも、自動的に呼ばれるが、空のメソッドを作成すると警告されるので、明示的にしておきます。
	public Editor_YMD2() {
		super();
		isSubm	= "false";														// 8.0.2.0 (2021/11/30) Add
		errMsg	= null;															// 8.0.2.0 (2021/11/30) Add
		prmAry	= null;															// 8.0.2.0 (2021/11/30) Add
		hidden	= true;															// 8.0.2.0 (2021/11/30) Modify disabled ⇒ hidden
	}

	/**
	 * DBColumnオブジェクトを指定したprivateコンストラクター。
	 *
	 * @og.rev 6.9.8.0 (2018/05/28) disabled 時に、日付ピッカーを出さない対応
	 * @og.rev 8.0.2.0 (2021/11/30) 日付送り戻し対応
	 *
	 * @param	clm	DBColumnオブジェクト
	 */
	private Editor_YMD2( final DBColumn clm ) {
		super( clm );
		tagBuffer.add( XHTMLTag.inputAttri( attributes ) );

		final boolean disabled = "disabled".equalsIgnoreCase( attributes.get( "disabled" ) );	// 6.9.8.0 (2018/05/28)
		final boolean readonly = "readonly".equalsIgnoreCase( attributes.get( "readonly" ) );	// 8.0.2.0 (2021/11/30)

		// 8.0.2.0 (2021/11/30) Add 日付送り戻し対応
		hidden = disabled || readonly ;
		if( !hidden ) {
			// 例：D-1,D0,D+1 or D-1,D0,D+1,true
			final String prmStr = clm.getEditorParam();
			if( prmStr != null ) {
				prmAry = StringUtil.csv2Array( prmStr );
				// パラメータの第四引数がある場合
				if( prmAry.length > CNT_ARY ) {
					isSubm = prmAry[CNT_ARY];
				} else if( prmAry.length < CNT_ARY ) {
					errMsg = "editorParam の設定が不足です。"
										+ " name="  + name
										+ " label=" + clm.getLabel()
										+ " editorParam=" + clm.getEditorParam();
					System.out.println( errMsg );
				}
			}
		}
	}

	/**
	 * 各オブジェクトから自分のインスタンスを返します。
	 * 自分自身をキャッシュするのか、新たに作成するのかは、各サブクラスの実装に
	 * まかされます。
	 *
	 * @param	clm	DBColumnオブジェクト
	 *
	 * @return	CellEditorオブジェクト
	 * @og.rtnNotNull
	 */
	public CellEditor newInstance( final DBColumn clm ) {
		return new Editor_YMD2( clm );
	}

	/**
	 * データの編集用文字列を返します。
	 *
	 * @og.rev 4.3.7.2 (2009/06/15) 属性でidが出力される場合は、idを出力しない
	 * @og.rev 6.8.5.0 (2018/01/09) カレンダポップアップで、３ヶ月ピッカー対応
	 * @og.rev 6.9.8.0 (2018/05/28) disabled 時に、日付ピッカーを出さない対応
	 * @og.rev 8.0.2.0 (2021/11/30) 日付送り戻し対応
	 *
	 * @param	value	入力値
	 *
	 * @return	データの編集用文字列
	 * @og.rtnNotNull
	 */
	@Override
	public String getValue( final String value ) {
		// 8.0.2.0 (2021/11/30) Add
		if( prmAry != null && prmAry.length < CNT_ARY ) {
			return "<span class=\"error\">" + errMsg + "</span>";
		}

		// 6.1.1.0 (2015/01/17) TagBufferの連結記述
		final String tag = new TagBuffer( "input" )
						.add( "name"	, name )
						.add( "id"		, name , isNull( attributes.get( "id" ) ) )		// 4.3.7.2 (2009/06/15)
						.add( "value"	, value )
						.add( "size"	, size1 )
						.add( tagBuffer.makeTag() )
						.makeTag();

//		return tag + CAL_ST + name + CAL_END ;
//		return disabled ? tag									// 6.9.8.0 (2018/05/28) disabled 時に、日付ピッカーを出さない対応
//						: tag + CAL_STR + name + CAL_END ;		// 6.9.8.0 (2018/05/28) 変数名を、CAL_ST → CAL_STR に変更。
		// 8.0.2.0 (2021/11/30) Modify
		if( hidden ) {
			return tag;
		} else {
			final StringBuilder buf = new StringBuilder(BUFFER_MIDDLE);
			buf.append( tag )
				.append( CAL_STR )
				.append( name )
				.append( CAL_END );

			// 8.0.2.0 (2021/11/30) 日付送り戻し対応
			if( prmAry != null && prmAry.length > 0 ) {
				final String nmid = isNull( attributes.get( "id" ) ) ? name : attributes.get( "id" );
				// 第一回目の処理は、日付戻しを指定します。
				// 第二回目の処理は、初期化します。
				// 第三回目の処理は、日付送りを指定します。
				for( int i=0; i<CNT_ARY; i++ ) {
					buf.append( String.format( FR_STR + FR_END, "YMD", nmid, value, prmAry[i], isSubm, FR_IMG[i] ) );
				}
			}
			return buf.toString();
		}
	}

	/**
	 * name属性を変えた、データ表示/編集用のHTML文字列を作成します。
	 * テーブル上の name に 行番号を付加して、名前_行番号 で登録するキーを作成し,
	 * リクエスト情報を１つ毎のフィールドで処理できます。
	 *
	 * @og.rev 4.3.7.2 (2009/06/15) 属性でidが出力される場合は、idを出力しない
	 * @og.rev 6.8.5.0 (2018/01/09) カレンダポップアップで、３ヶ月ピッカー対応
	 * @og.rev 6.9.8.0 (2018/05/28) disabled 時に、日付ピッカーを出さない対応
	 * @og.rev 8.0.2.0 (2021/11/30) 日付送り戻し対応
	 *
	 * @param	row		行番号
	 * @param	value	入力値
	 *
	 * @return	データ表示/編集用の文字列
	 * @og.rtnNotNull
	 */
	@Override
	public String getValue( final int row,final String value ) {
		// 8.0.2.0 (2021/11/30) Add
		if( prmAry != null && prmAry.length < CNT_ARY ) {
			return "<span class=\"error\">" + errMsg + " row=" + row + "</span>";
		}

		final String name2 = name + HybsSystem.JOINT_STRING + row ;

		// 6.1.1.0 (2015/01/17) TagBufferの連結記述
		final String tag = new TagBuffer( "input" )
						.add( "name"	, name2 )
						.add( "id"		, name2 , isNull( attributes.get( "id" ) ) )		// 4.3.7.2 (2009/06/15)
						.add( "value"	, value )
						.add( "size"	, size2 )
						.add( tagBuffer.makeTag() )
						.makeTag( row,value );

//		return tag + CAL_ST + name2 + CAL_END ;
//		return disabled ? tag									// 6.9.8.0 (2018/05/28) disabled 時に、日付ピッカーを出さない対応
//						: tag + CAL_STR + name2 + CAL_END ;		// 6.9.8.0 (2018/05/28) 変数名を、CAL_ST → CAL_STR に変更。
		// 8.0.2.0 (2021/11/30) Modify
		if( hidden ) {
			return tag;
		} else {
			final StringBuilder buf = new StringBuilder(BUFFER_MIDDLE);
			buf.append( tag )
				.append( CAL_STR )
				.append( name2 )
				.append( CAL_END );

			// 8.0.2.0 (2021/11/30) 日付送り戻し対応
			if( prmAry != null && prmAry.length > 0 ) {
				final String nmid = isNull( attributes.get( "id" ) ) ? name2 : attributes.get( "id" );
				// 第一回目の処理は、日付戻しを指定します。
				// 第二回目の処理は、初期化します。
				// 第三回目の処理は、日付送りを指定します。
				for( int i=0; i<CNT_ARY; i++ ) {
					buf.append( String.format( FR_STR + FR_END, "YMD", nmid, value, prmAry[i], isSubm, FR_IMG[i] ) );
				}
			}
			return buf.toString();
		}
	}
}
