/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.taglib;

import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.common.HybsSystemException;
import org.opengion.hayabusa.db.DBTableModel;
import org.opengion.hayabusa.io.TableWriter;
import org.opengion.fukurou.util.FileUtil;
import org.opengion.fukurou.util.HybsEntry;
import org.opengion.fukurou.system.Closer ;
import org.opengion.fukurou.util.StringUtil ;
import org.opengion.fukurou.util.ToString;						// 6.1.1.0 (2015/01/17)
// import org.opengion.fukurou.model.FileOperation;				// 8.0.0.1 (2021/10/08)
import org.opengion.hayabusa.io.HybsFileOperationFactory;		// 8.0.0.1 (2021/10/08)

import static org.opengion.fukurou.util.StringUtil.nval ;

// import jakarta.servlet.ServletResponse ;
// import jakarta.servlet.http.HttpServletResponse ;
// import jakarta.servlet.jsp.JspWriter ;

import java.util.List;
import java.util.ArrayList;
import java.util.Locale;
import java.io.File;
import java.io.OutputStream;
import java.io.PrintWriter;
import java.io.FileOutputStream;
import java.io.BufferedOutputStream;				// 6.0.4.0 (2014/11/28)
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;							// 6.0.4.0 (2014/11/28)
import java.io.FileInputStream;						// 6.0.4.0 (2014/11/28)
import java.io.BufferedInputStream;					// 6.0.4.0 (2014/11/28)
// import java.io.FileNotFoundException;			// 8.0.0.1 (2021/10/08)

import org.apache.commons.compress.archivers.zip.ZipArchiveEntry;			// 6.0.4.0 (2014/11/28)
import org.apache.commons.compress.archivers.zip.ZipArchiveOutputStream;	// 6.0.4.0 (2014/11/28)
import org.apache.commons.compress.utils.IOUtils;							// 6.0.4.0 (2014/11/28)

/**
 * DBTableModelｵﾌﾞｼﾞｪｸﾄを指定のﾌｧｲﾙに出力するﾀｸﾞです。
 *
 * ﾃﾞｰﾀ(DBTableModel)と､ｺﾝﾄﾛｰﾗ(WriteTableTagForm)を与えて､
 * 外部からｺﾝﾄﾛｰﾙすることで､各種形式で ﾃﾞｰﾀ(DBTableModel)を表示できます。
 * writeTableﾀｸﾞ に対して､ｺﾏﾝﾄﾞを与えることにより､内部のｺﾝﾄﾛｰﾗの
 * 実装に対応した､ 形式でﾃﾞｰﾀを作成します。
 *
 * @og.formSample
 * ●形式：
 *     &lt;og:writeTable
 *         fileURL   = "{&#064;USER.ID}"    保存先ﾃﾞｨﾚｸﾄﾘ名
 *         filename  = "{&#064;filename}"   保存ﾌｧｲﾙ名
 *         encode    = "UnicodeLittle"      保存ﾌｧｲﾙｴﾝｺｰﾄﾞ名
 *     /&gt;
 *
 * ●body：あり(EVAL_BODY_BUFFERED:BODYを評価し､{&#064;XXXX} を解析します)
 *
 * ●Tag定義：
 *   &lt;og:writeTable
 *       writerClass        【TAG】実際に書き出すｸﾗｽ名の略称(TableWriter_**** ｸﾗｽの ****)を指定します
 *									(初期値:TABLE_WRITER_DEFAULT_CLASS[={@og.value SystemData#TABLE_WRITER_DEFAULT_CLASS}])
 *       separator          【TAG】可変長ﾌｧｲﾙを作成するときの項目区切り文字をｾｯﾄします(初期値:TableWriter#TAB_SEPARATOR[= ])
 *       headerSequence     【TAG】DBTableModelの出力順(LNSCD など)をｾｯﾄします。(7.2.6.1 (2020/07/17) H追加)
 *       fileURL            【TAG】ﾌｧｲﾙ保存先ﾃﾞｨﾚｸﾄﾘ名を指定します(初期値:FILE_URL[={@og.value SystemData#FILE_URL}])
 *       baseURL            【TAG】ﾌｧｲﾙ保存先ﾃﾞｨﾚｸﾄﾘ名を指定します(初期値:ｺﾝﾃｷｽﾄﾊﾟｽ) 7.2.7.0 (2020/08/07)
 *       filename           【TAG】ﾌｧｲﾙを作成するときのﾌｧｲﾙ名をｾｯﾄします(初期値:FILE_FILENAME[={@og.value SystemData#FILE_FILENAME}])
 *       encode             【TAG】ﾌｧｲﾙを作成するときのﾌｧｲﾙｴﾝｺｰﾃﾞｨﾝｸﾞ名をｾｯﾄします(初期値:FILE_ENCODE[={@og.value SystemData#FILE_ENCODE}])
 *       beforeDelete       【TAG】事前にﾌｧｲﾙを削除するかどうかをｾｯﾄします (初期値:false)
 *       fileAppend         【TAG】追加ﾓｰﾄﾞで書き込むかどうか[true/false]を指定します(初期値:false[新規ﾓｰﾄﾞ])
 *       zip                【TAG】結果をﾌｧｲﾙに出力するときに､ZIPで圧縮するかどうか[true/false]を指定します(初期値:false)
 *       zipFilename        【TAG】ZIPﾌｧｲﾙを作成するときのZIPﾌｧｲﾙ名をｾｯﾄします(初期値:filename + ".zip")
 *       tableId            【TAG】(通常は使いません)sessionから所得する DBTableModelｵﾌﾞｼﾞｪｸﾄの ID
 *       scope              【TAG】ｷｬｯｼｭする場合のｽｺｰﾌﾟ[request/page/session/application]を指定します(初期値:session)
 *       useNumber          【TAG】行番号情報を､出力する/しない[true/false]を指定します(初期値:true)
 *       columns            【TAG】書き込み先ﾌｧｲﾙのｶﾗﾑ列を､外部(ﾀｸﾞ)よりCSV形式で指定します
 *       omitNames          【TAG】書き込み対象外のｶﾗﾑ列を､外部(ﾀｸﾞ)よりCSV形式で指定します
 *       useRenderer        【TAG】書込処理でｺｰﾄﾞﾘｿｰｽのﾗﾍﾞﾙ変換を行うかどうかを指定します
 *                                      (初期値:USE_TABLE_WRITER_RENDERER[={@og.value SystemData#USE_TABLE_WRITER_RENDERER}])
 *       selectedAll        【TAG】ﾁｪｯｸ行のみを対象にするかを指定します(初期値:true)
 *       useLocal           【TAG】ｼｽﾃﾑ定数でｸﾗｳﾄﾞ設定されていても､ｸﾗｳﾄﾞ環境を使用しない場合､trueを指定します(初期値:false) 8.0.1.0 (2021/10/29)
 *   ※  sheetName          【TAG】EXCELﾌｧｲﾙを書き出すときのｼｰﾄ名を設定します
 *   ※  refFileURL         【TAG】EXCEL雛型参照ﾌｧｲﾙのﾃﾞｨﾚｸﾄﾘ名をｾｯﾄします
 *   ※  refFilename        【TAG】EXCEL雛型参考ﾌｧｲﾙ名をｾｯﾄします
 *   ※  refSheetName       【TAG】EXCEL雛型参考ﾌｧｲﾙのｼｰﾄ名を設定します
 *   ※  sheetOverwrite     【TAG】EXCELでsheetNameでの指定ｼｰﾄ名に対して上書きを行うかどうかを指定します。
 *   ※  recalcSheetName    【TAG】EXCELで､出力処理の最後にｾﾙの計算式の再計算をさせるｼｰﾄ名をCSV形式で指定します。
 *   ※  fontName           【TAG】EXCEL出力時のﾃﾞﾌｫﾙﾄﾌｫﾝﾄ名を設定します
 *                                      (初期値:TABLE_WRITER_DEFAULT_FONT_NAME[={@og.value SystemData#TABLE_WRITER_DEFAULT_FONT_NAME}])
 *   ※  fontPoint          【TAG】EXCEL出力時のﾃﾞﾌｫﾙﾄﾌｫﾝﾄﾎﾟｲﾝﾄ数を設定します
 *                                      (初期値:TABLE_WRITER_DEFAULT_FONT_POINTS[={@og.value SystemData#TABLE_WRITER_DEFAULT_FONT_POINTS}])
 *   ※  skipRowCount       【TAG】(通常は使いません)EXCEL出力時のﾃﾞｰﾀの書き込み開始位置を設定します
 *   ※  useCellStyle       【TAG】EXCEL出力時に､ﾃﾞｰﾀを書き込んだ範囲に罫線を入れるかどうかを指定します(初期値:false)
 *   ※  useAutoCellSize    【TAG】EXCEL出力時に､ｾﾙの幅をﾃﾞｰﾀの幅に自動的に合わせるかどうかを指定します(初期値:false)
 *   ※  useActiveWorkbook  【TAG】EXCEL出力時に､ｾﾙの有効範囲を設定するかどうかを指定します(初期値:false)
 *   ※  pageBreakColumn    【TAG】EXCEL出力時に､ｼｰﾄ変更するｷｰとなるｶﾗﾑ名を指定します(このｶﾗﾑの値がｼｰﾄ名になります)
 *   ※  fileBreakColumn    【TAG】EXCEL出力時に､ﾌｧｲﾙ名を変更するｷｰとなるｶﾗﾑ名を指定します(このｶﾗﾑの値がﾌｧｲﾙ名になります)
 *   ※  hyperLinkColumn    【TAG】EXCEL出力時に､Hyperlinkを作成するｷｰとなるｶﾗﾑ名と値となるｶﾗﾑ名を指定します
 *   ※  addTitleSheet      【TAG】EXCEL出力時に､存在するSheet名一覧を作成する場合に､そのSheet名を指定します。
 *       language           【TAG】ﾀｸﾞ内部で使用する言語ｺｰﾄﾞ[ja/en/zh/…]を指定します
 *       caseKey            【TAG】このﾀｸﾞ自体を利用するかどうかの条件ｷｰを指定します(初期値:null) 5.7.7.2 (2014/06/20)
 *       caseVal            【TAG】このﾀｸﾞ自体を利用するかどうかの条件値を指定します(初期値:null) 5.7.7.2 (2014/06/20)
 *       caseNN             【TAG】指定の値が､null/ｾﾞﾛ文字列 でない場合(Not Null=NN)は､このﾀｸﾞは使用されます(初期値:判定しない) 5.7.7.2 (2014/06/20)
 *       caseNull           【TAG】指定の値が､null/ｾﾞﾛ文字列 の場合は､このﾀｸﾞは使用されます(初期値:判定しない) 5.7.7.2 (2014/06/20)
 *       caseIf             【TAG】指定の値が､true/TRUE文字列の場合は､このﾀｸﾞは使用されます(初期値:判定しない)
 *       debug              【TAG】ﾃﾞﾊﾞｯｸﾞ情報を出力するかどうか[true/false]を指定します(初期値:false)
 *   &gt;   ... Body ...
 *   &lt;/og:writeTable&gt;
 *
 * 【廃止】6.9.5.0 (2018/04/23) 物理削除
 *   //  direct             【廃止】結果をﾀﾞｲﾚｸﾄにEXCEL起動するかどうか[true/false]を指定します(初期値:false[ﾌｧｲﾙ])  6.9.5.0 (2018/04/23) 物理削除
 *   //  disposition        【廃止】Content-Disposition (inline:埋め込む/attachment:ﾀﾞｲｱﾛｸﾞを起動)を指定します(初期値:attachment)  6.9.5.0 (2018/04/23) 物理削除
 *
 *
 * ●使用例
 *     &lt;og:writeTable ･････ &gt;
 *         &lt;og:writeTableParam
 *             key  = "Tablename"  value="GE12"
 *         /&gt;
 *         &lt;og:writeTableParam
 *             key  = "First"             First:最初に登録
 *         &gt;
 *                      insert into GE12bk
 *                         select * from GE12
 *                         where SYSTEM_ID='**'
 *         &lt;/og:writeTableParam
 *         &lt;og:writeTableParam
 *             key  = "First"             First:の２番目に登録
 *         &gt;
 *              delete from GE12 where SYSTEM_ID='**' and KBSAKU='0'
 *         &lt;/og:writeTableParam
 *         &lt;og:writeTableParam
 *             key  = "Last"              Last:最後に登録
 *         &gt;
 *              update GE12 set XXXX='YYYY' where SYSTEM_ID='**' and KBSAKU='0'
 *         &lt;/og:writeTableParam
 *     &lt;/og:writeTableParam
 *
 * @og.group ﾌｧｲﾙ出力
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class WriteTableTag extends CommonTagSupport {
	/** このﾌﾟﾛｸﾞﾗﾑのVERSION文字列を設定します	{@value} */
	private static final String VERSION = "8.0.1.0 (2021/10/29)" ;
	private static final long serialVersionUID = 801020211029L ;

	private final String  BASE_URL	= HybsSystem.sys( "FILE_URL" );

	private String	separator		= TableWriter.TAB_SEPARATOR;	 // 項目区切り文字
	private String	headerSequence	;					// ﾍｯﾀﾞｰ項目の並び順
	private String	fileURL 		= BASE_URL;
	private String	filename		= HybsSystem.sys( "FILE_FILENAME" );   // ﾌｧｲﾙ名
	private String	sheetName		= "Sheet1" ;		// 3.5.4.2 (2003/12/15)
	private String	refFileURL		= BASE_URL;			// 3.5.4.3 (2004/01/05)
	private String	refFilename 	;					// 3.5.4.3 (2004/01/05)
	private String	refSheetName	;					// 3.5.4.3 (2004/01/05)
	private String	fontName		= HybsSystem.sys( "TABLE_WRITER_DEFAULT_FONT_NAME" );			// 3.8.5.3 (2006/08/07)
	private String	fontPoint	 	= HybsSystem.sys( "TABLE_WRITER_DEFAULT_FONT_POINTS" );			// 3.8.5.3 (2006/08/07)
	private String	encode			= HybsSystem.sys( "FILE_ENCODE" );	// ﾌｧｲﾙｴﾝｺｰﾃﾞｨﾝｸﾞ  "DEFAULT","JISAutoDetect" ,"JIS", "EUC_JP", "MS932", "SJIS" , "Windows-31J" , "Shift_JIS"
	private String	writerClass 	;														// 6.2.0.0 (2015/02/27) 個人設定可
	private boolean	fileAppend		;					// ﾌｧｲﾙをAPPENDﾓｰﾄﾞで出力するか
//	private boolean	direct			;					// 6.9.5.0 (2018/04/23) 廃止
	private boolean	zip 			;
	private String  zipFilename		;					// 6.0.4.0 (2014/11/28) Zipﾌｧｲﾙ名を外部から与えられるように修正
//	private String  disposition     = "attachment";		// 3.8.0.9 (2005/10/17) Content-Disposition 初期値変更 inline ⇒ attachment 6.9.5.0 (2018/04/23) 廃止

	private transient DBTableModel table ;
	private String	tableId 		= HybsSystem.TBL_MDL_KEY ;
	private String	directory		;					// 3.8.6.0 (2006/08/23)

	private transient List<HybsEntry> param ;			// 4.0.0 (2005/02/28)

	private boolean	useNumber		= true;				// 3.7.0.2 (2005/02/14)
	// 4.0.0 (2005/12/31) 外部よりｶﾗﾑ列(CSV形式)を指定できるようにする。
	private String		columns		;
	// 6.1.0.0 (2014/12/26) 書き込み対象外のｶﾗﾑ列を､外部(ﾀｸﾞ)より指定する。
	private String		omitNames	;					// 6.1.0.0 (2014/12/26)

	// 5.2.1.0 (2010/10/01) 書込処理でｺｰﾄﾞﾘｿｰｽのﾗﾍﾞﾙ変換を行うかどうか
	private boolean	useRenderer		= HybsSystem.sysBool( "USE_TABLE_WRITER_RENDERER" );	// 5.2.1.0 (2010/10/01)

	// 5.5.5.5 (2012/08/24) ﾁｪｯｸ対応
	private boolean selectedAll		= true;				// 5.5.5.5 (2012/08/24)

	private	boolean	useLocal		;					// 8.0.1.0 (2021/10/29) ｸﾗｳﾄﾞ設定を使用しない場合は､true

	// 5.5.9.1 (2012/12/07) 事前にﾌｧｲﾙを削除するかどうか。互換性のため､false に設定。推奨は､true
	private boolean	beforeDelete	;					// 5.5.9.1 (2012/12/07) 事前にﾌｧｲﾙを削除するかどうか。

	private int		skipRowCount	;					// 5.7.9.0 (2014/08/08) ﾃﾞｰﾀの書き込み開始位置設定

	// 5.9.12.1 (2016/09/09)
	private boolean	sheetOverwrite;						// 5.9.12.1 (2016/09/09)
	private String recalcSheetName;						// 5.9.12.1 (2016/09/09)

	// 6.0.2.0 (2014/09/19) EXCEL関係機能追加
	private boolean	useCellStyle		;			// EXCEL出力時に､ﾃﾞｰﾀを書き込んだ範囲に罫線を入れるかどうかを指定します
	private boolean	useAutoCellSize		;			// EXCEL出力時に､ｾﾙの幅をﾃﾞｰﾀの幅に自動的に合わせるかどうかを指定します
	private boolean	useActiveWorkbook	;			// EXCEL出力時に､ｾﾙの有効範囲を設定するかどうかを指定します
	private String	pageBreakColumn		;			// EXCEL出力時に､ｼｰﾄ変更するｷｰとなるｶﾗﾑ名を指定します(このｶﾗﾑの値がｼｰﾄ名になります)
	private String	fileBreakColumn		;			// EXCEL出力時に､ﾌｧｲﾙ名を変更するｷｰとなるｶﾗﾑ名を指定します(このｶﾗﾑの値がﾌｧｲﾙ名になります)
	private String	hyperLinkColumn		;			// EXCEL出力時に､Hyperlinkを作成するｷｰとなるｶﾗﾑ名と値となるｶﾗﾑ名を指定します
	private String	addTitleSheet		;			// EXCEL出力時に､存在するSheet名一覧を作成する場合に､そのSheet名を指定します。

	private File	localFile			;			// 8.0.0.2 (2021/10/15) 一時的に利用するﾛｰｶﾙﾌｧｲﾙ

	/**
	 * ﾃﾞﾌｫﾙﾄｺﾝｽﾄﾗｸﾀｰ
	 *
	 * @og.rev 6.4.2.0 (2016/01/29) PMD refactoring. Each class should declare at least one constructor.
	 */
	public WriteTableTag() { super(); }		// これも､自動的に呼ばれるが､空のﾒｿｯﾄﾞを作成すると警告されるので､明示的にしておきます。

	/**
	 * Taglibの開始ﾀｸﾞが見つかったときに処理する doStartTag() を ｵｰﾊﾞｰﾗｲﾄﾞします。
	 *
	 * @og.rev 5.5.9.1 (2012/12/07) 事前にﾌｧｲﾙを削除するかどうか（beforeDelete）ﾌﾗｸﾞを追加
	 * @og.rev 5.7.7.2 (2014/06/20) caseKey,caseVal,caseNN,caseNull 属性を追加
	 * @og.rev 6.0.4.0 (2014/11/28) Zipﾌｧｲﾙ名を､内部変数化。(将来的に引数化を容易にするため)
	 * @og.rev 8.0.0.1 (2021/10/08) ｸﾗｳﾄﾞ対応
	 * @og.rev 8.0.0.2 (2021/10/15) ﾛｰｶﾙﾌｧｲﾙとｸﾗｳﾄﾞﾌｧｲﾙ間の移動
	 *
	 * @return	後続処理の指示( EVAL_BODY_BUFFERED )
	 */
	@Override
	public int doStartTag() {
		// 5.7.7.2 (2014/06/20) caseKey,caseVal,caseNN,caseNull 属性を追加
		if( !useTag() ) { return SKIP_BODY ; }

		// 8.0.0.2 (2021/10/15) zipやﾃﾞｨﾚｸﾄﾘ作成なども含めた処理を行う。
		localFile = makeLocalFile();

//		directory = HybsSystem.url2dir( fileURL );		// 5.5.9.1 (2012/12/07) doEndTag() から移動
//		if( zip && zipFilename == null ) { zipFilename = filename + ".zip"; }		// 6.0.4.0 (2014/11/28)

		// 5.5.9.1 (2012/12/07) 事前にﾌｧｲﾙを削除するかどうか
		if( beforeDelete && !fileAppend ) {
//			File delFile = null;
//			if( zip ) {
//				delFile = new File( directory,zipFilename );				// 6.0.4.0 (2014/11/28)
	//			// 一旦､ﾛｰｶﾙにﾌｧｲﾙを作成するので､削除はﾛｰｶﾙ側
	//			delFile = HybsFileOperationFactory.create( directory, zipFilename );	// 8.0.0.1 (2021/10/08)
//			}
//			else {
	//			// 一旦､ﾛｰｶﾙにﾌｧｲﾙを作成するので､削除はﾛｰｶﾙ側
//				delFile = new File( directory,filename );
	//			delFile = HybsFileOperationFactory.create( directory, filename );	// 8.0.0.1 (2021/10/08)
//			}
			// 6.0.2.4 (2014/10/17) RV  java.io.File.delete() の例外的戻り値を無視しています。
//			if( delFile.exists() && !delFile.delete() ) {
			if( localFile.exists() && !localFile.delete() ) {
//					final String errMsg = "事前にﾌｧｲﾙを削除するのに失敗しました。filename=[" + delFile +"]" ;
					final String errMsg = "事前にﾌｧｲﾙを削除するのに失敗しました。filename=[" + localFile +"]" ;
					System.err.println( errMsg );
			}
		}

		return EVAL_BODY_BUFFERED ;	// Body を評価する。( extends BodyTagSupport 時)
	}

	/**
	 * Taglibの終了ﾀｸﾞが見つかったときに処理する doEndTag() を ｵｰﾊﾞｰﾗｲﾄﾞします。
	 *
	 * @og.rev 2.1.3.1 (2002/12/13) ContentType を､x-msexcel から vnd.ms-excel に変更した。
	 * @og.rev 3.1.1.2 (2003/04/04) Tomcat4.1 対応。release2() を doEndTag()で呼ぶ。
	 * @og.rev 3.1.3.0 (2003/04/10) FILE_ENCODE から､ｴﾝｺｰﾄﾞ情報を取得する。
	 * @og.rev 3.1.5.0 (2003/04/20) Content-Disposition を外部から指定できるように変更。
	 * @og.rev 3.2.0.0 (2003/05/22) GZIPOutputStream を使用していたが､ZipOutputStream に変更。
	 * @og.rev 3.5.4.1 (2003/12/01) 引数の PrintWriter を､OutputStream に変更。
	 * @og.rev 3.5.4.3 (2004/01/05) 引数を､ PrintWriter に戻す。
	 * @og.rev 3.6.0.2 (2004/10/04) EXCELﾀﾞｲﾚｸﾄ出力時にﾌｧｲﾙｴﾝﾄﾞまで出力されない不具合対策
	 * @og.rev 3.8.0.9 (2005/10/17) disposition属性を使用できるように変更します。
	 * @og.rev 3.8.6.0 (2006/08/23) ﾌｫﾙﾀﾞがない場合は､複数階層分のﾌｫﾙﾀﾞを自動で作成します。
	 * @og.rev 4.3.4.3 (2008/12/22) Excel出力の判別方法を前方一致に変更
	 * @og.rev 5.1.6.0 (2010/05/01) 画面帳票作成機能対応
	 * @og.rev 5.5.2.0 (2012/05/01) ZIP時にもencodeが有効にする
	 * @og.rev 5.5.5.5 (2012/08/24) selectedAllの処理を追加
	 * @og.rev 5.5.9.1 (2012/12/07) beforeDeleteﾌﾗｸﾞの追加に伴う､ﾛｼﾞｯｸ変更
	 * @og.rev 5.7.7.2 (2014/06/20) caseKey,caseVal,caseNN,caseNull 属性を追加
	 * @og.rev 6.0.4.0 (2014/11/28) Zip処理を､ZipOutputStream → ZipArchiveOutputStream に変更
	 * @og.rev 6.2.0.0 (2015/02/27) EXCEL出力のparamLevel初期値変更 3:標準推奨 → 4:個人設定可
	 * @og.rev 6.9.5.0 (2018/04/23) direct 廃止(false固定)
	 * @og.rev 5.10.23.0 (2020/06/01) 書込み件数をﾘｸｴｽﾄ変数WRITE.COUNTに出力
	 * @og.rev 8.0.0.1 (2021/10/08) ｸﾗｳﾄﾞ対応
	 * @og.rev 8.0.0.2 (2021/10/15) ﾛｰｶﾙﾌｧｲﾙとｸﾗｳﾄﾞﾌｧｲﾙ間の移動
	 *
	 * @return	後続処理の指示
	 */
	@Override
	public int doEndTag() {
		debugPrint();		// 4.0.0 (2005/02/28)

		// 5.7.7.2 (2014/06/20) caseKey,caseVal,caseNN,caseNull 属性を追加
		if( !useTag() ) { return EVAL_PAGE ; }

		table = (DBTableModel)getObject( tableId );
		if( ! selectedAll ){		// 5.5.5.5 (2012/08/24)
			table = getSelRowsTable( table );
		}

		int tblcnt = 0;		// 5.10.23.0 (2020/06/01)
		if( table != null && table.getRowCount() > 0 ) {
			// 8.0.0.2 (2021/10/15) localFileは､zipやﾃﾞｨﾚｸﾄﾘ作成なども含めた処理を行う。
//			final File dir = new File( directory );
	//		ｸﾗｳﾄﾞはﾃﾞｨﾚｸﾄﾘ作成不要
	//		final File dir = HybsFileOperationFactory.create( directory );		// 8.0.0.1 (2021/10/08)

//			if( ! dir.exists() && ! dir.mkdirs() ) {
//				final String errMsg = "ﾃﾞｨﾚｸﾄﾘの作成に失敗しました。[" + directory + "]";
//				throw new HybsSystemException( errMsg );
//			}

			// 6.2.0.0 (2015/02/27) EXCEL出力のparamLevel初期値変更 3:標準推奨 → 4:個人設定可
			if( writerClass == null ) {
				writerClass = nval( sys( "TABLE_WRITER_DEFAULT_CLASS" ) , "Default" );		// 6.2.0.0 (2015/02/27) 個人設定可
			}

			PrintWriter pw = null;
			ByteArrayOutputStream bt = null ;
			try {
//				// 6.9.5.0 (2018/04/23) 廃止(false固定)
//				if( direct ) {
//					// 6.0.4.0 (2014/11/28) 共通処理を､if( zip ) の前後でまとめます。
//					bt = new ByteArrayOutputStream();
//					pw = new PrintWriter( bt );
//					create( pw ) ;
//					pw.flush();				// 3.6.0.2 (2004/10/04)
//					final ServletResponse response = pageContext.getResponse() ;
//					if( zip ) {
//						((HttpServletResponse)response).addHeader( "Content-Encoding","gzip" );
//
//						final ZipArchiveOutputStream gzip = new ZipArchiveOutputStream(bt);	// 6.0.4.0 (2014/11/28)
//						gzip.finish() ;
//						response.setContentLength( bt.size() );
//
//						gzip.closeArchiveEntry();	// 6.0.4.0 (2014/11/28)
//						gzip.finish() ;				// 6.0.4.0 (2014/11/28)
//						Closer.ioClose( gzip );		// 6.0.4.0 (2014/11/28)
//
//					}
//					else {
//						response.setContentType( "application/vnd.ms-excel; charset=" + encode );
//	//					response.setContentType( "application/x-msexcel; charset=" + encode );
//	//					response.setContentType( "text/comma-separated-values ; charset=" + encode );
//	//					((HttpServletResponse)response).setHeader( "contentType","application/x-msexcel; charset=Shift_JIS" );
//	//					((HttpServletResponse)response).setHeader( "Content-Disposition","attachment; filename=\"" + filename + "\"" );
//	//					((HttpServletResponse)response).setHeader( "Content-Disposition","inline; filename=\"" + filename + "\"" );
//						// 3.8.0.9 (2005/10/17) disposition属性を使用できるように変更します。
//						((HttpServletResponse)response).setHeader( "Content-Disposition",disposition + "; filename=\"" + filename + "\"" );
//
//					}
//					final JspWriter out = pageContext.getOut();
//					out.clear();
//					out.print( bt );
//					out.flush();			// 3.6.0.2 (2004/10/04)
//				}
//				else {
					final boolean isExcel =  writerClass.toUpperCase(Locale.JAPAN).startsWith("EXCEL")
										  || writerClass.toUpperCase(Locale.JAPAN).startsWith("CALC");	// 6.0.4.0 (2014/11/28)

					// 3.2.0.0 (2003/05/22) GZIPOutputStream を使用していたが､ZipOutputStream に変更。
					if( zip ) {
						InputStream is = null;			// 6.0.4.0 (2014/11/28)
						if( isExcel ) {					// 6.0.4.0 (2014/11/28)
							create( null );				// 6.0.4.0 (2014/11/28)
						}
						else{
							bt = new ByteArrayOutputStream();
							pw = getPrintWriter( bt );	// ﾛｰｶﾙにﾌｧｲﾙ作成
							create( pw ) ;				// 6.0.4.0 (2014/11/28)
							pw.flush();					// 6.0.4.0 (2014/11/28)
						}

						// 6.0.4.0 (2014/11/28) Zip処理を､ZipOutputStream → ZipArchiveOutputStream に変更
						ZipArchiveOutputStream gzip = null;			// 6.0.4.0 (2014/11/28)
						try {
							gzip = new ZipArchiveOutputStream(
										new BufferedOutputStream (
											new FileOutputStream (
												localFile )));							// 8.0.0.2 (2021/10/15)
//												new File( directory,zipFilename ))));	// 6.0.4.0 (2014/11/28)

							gzip.setEncoding( "Windows-31J" );
							gzip.putArchiveEntry( new ZipArchiveEntry( filename ) );
							// 6.0.4.0 (2014/11/28) Excel処理とText処理が異なる。
							// bt.writeTo( gzip );
							if( isExcel ) {
								// 6.0.4.0 (2014/11/28) Excel の場合は､直接書き込むのではなく､ﾌｧｲﾙを読み込んで Zip化する。
//								final File xlsTemp = new File( directory,filename );
								final File xlsTemp = localFile ;		// 8.0.0.2 (2021/10/15) 修正箇所を増やさないだけ。
								is = new BufferedInputStream( new FileInputStream( xlsTemp ) );
								IOUtils.copy( is,gzip );
								Closer.ioClose( is );
								if( xlsTemp.exists() && !xlsTemp.delete() ) {
									final String errMsg = "ExcelをZip化する時のTempﾌｧｲﾙを削除するのに失敗しました。filename=[" + xlsTemp +"]" ;
									System.err.println( errMsg );
								}
							}
							else {
								bt.writeTo( gzip );
							}
							gzip.closeArchiveEntry();				// 6.0.4.0 (2014/11/28)
							gzip.finish() ;
						}
						finally {
							Closer.ioClose( gzip );		// 4.0.0.0 (2006/01/31) close 処理時の IOException を無視
							Closer.ioClose( is );		// 6.0.4.0 (2014/11/28) is は､null でも処理する。
						}
					}
					else {
						if( isExcel ) {					// 6.0.4.0 (2014/11/28)
							create( null );
						}
						else {
							pw = getPrintWriter();		// ﾛｰｶﾙにﾌｧｲﾙ作成
							create( pw );
							pw.flush();					// 3.6.0.2 (2004/10/04)
						}
					}
//				}
				tblcnt = table.getRowCount();			// 5.10.23.0 (2020/06/01)
			}
			catch( final IOException ex ) {
				final String errMsg = "ﾃｰﾌﾞﾙ出力時に､IOｴﾗｰが発生しました。" + toString();
				throw new HybsSystemException( errMsg,ex );		// 3.5.5.4 (2004/04/15) 引数の並び順変更
			}
			finally {
				Closer.ioClose( pw );		// 4.0.0 (2006/01/31) close 処理時の IOException を無視
				Closer.ioClose( bt );		// 4.0.0 (2006/01/31) close 処理時の IOException を無視
			}

			// 8.0.0.2 (2021/10/15) ﾛｰｶﾙに出力したﾌｧｲﾙを､ｸﾗｳﾄﾞに移す。
			HybsFileOperationFactory.local2cloud( useLocal,() -> localFile );		// 8.0.0.2 (2021/10/15)

//			// 8.0.0.1 (2021/10/08) ｸﾗｳﾄﾞ上の場合は､ｽﾄﾚｰｼﾞに登録する
//			FileOperation fileOperation = HybsFileOperationFactory.create( filename );
//			if(!fileOperation.isLocal()){
//			if( HybsFileOperationFactory.useCloud() ){
//				if(zip){
//					entryStorage(zipFilename);
//				}else{
//					entryStorage(filename);
//				}
//			}
		}
		setRequestAttribute( "WRITE.COUNT" , String.valueOf( tblcnt ) ); // 5.10.23.0 (2020/06/01)

		return EVAL_PAGE ;
	}

	/**
	 * ﾛｰｶﾙﾌｧｲﾙを作成します。
	 *
	 * ﾃﾞｨﾚｸﾄﾘの存在ﾁｪｯｸとなければ作成します。
	 * zipﾌｧｲﾙの指定がされて､zipFilenameが指定されていない場合は､filename + ".zip" とします。
	 *
	 * @og.rev 8.0.0.2 (2021/10/15) ﾛｰｶﾙﾌｧｲﾙとｸﾗｳﾄﾞﾌｧｲﾙ間の移動
	 *
	 * @return	ﾛｰｶﾙﾌｧｲﾙｵﾌﾞｼﾞｪｸﾄ
	 */
	private File makeLocalFile() {
		if( filename == null ) {
			final String errMsg = "ﾌｧｲﾙ名がｾｯﾄされていません。";
			throw new HybsSystemException( errMsg );
		}
		directory = HybsSystem.url2dir( fileURL );			// EXCEL作成時に使用するので､ｲﾝｽﾀﾝｽ変数にｾｯﾄ

		// 5.6.1.0 (2013/02/01)
		final File dir = new File(directory);
		if( ! dir.exists() && ! dir.mkdirs() ) {
			final String errMsg = "ﾃﾞｨﾚｸﾄﾘの作成に失敗しました。[" + directory + "]";
			throw new HybsSystemException( errMsg );
		}

		final String newFilename ;
		if( zip ) {
			if( zipFilename == null ) { newFilename = filename + ".zip"; }
			else {						newFilename = zipFilename ; }
		}
		else {
			newFilename = filename;
		}

		return new File( directory,newFilename );
	}

//	/**
//	 * 指定されたﾌｧｲﾙをｸﾗｳﾄﾞ上のｽﾄﾚｰｼﾞに登録します。
//	 * 処理後に登録元のﾌｧｲﾙを削除します。
//	 *
//	 * @og.rev 5.10.9.0 (2019/03/01) ｸﾗｳﾄﾞｽﾄﾚｰｼﾞ対応を追加
//	 * @og.rev 8.0.0.1 (2021/10/08) ｸﾗｳﾄﾞ対応
//	 * @og.rev 8.0.0.2 (2021/10/15) 削除(makeLocalFile() とlocal2cloudで対応)
//	 *
//	 * @param trgFileName 対象のﾌｧｲﾙ名
//	 */
//	private void entryStorage( final String trgFileName ) {
//		FileInputStream fis = null;
//		final String filePath = StringUtil.urlAppend( directory, trgFileName);
//		// ｻｰﾊﾞに作成されたﾌｧｲﾙをｽﾄﾚｰｼﾞに登録する
//		try{
//			fis = new FileInputStream( filePath );
//
//			final FileOperation fileOpp = HybsFileOperationFactory.create( filePath );
//			fileOpp.write(fis);
//		} catch( final FileNotFoundException ex ){
//			String errMsg = "ﾌｧｲﾙが見つかりません。path:" + filePath;
//			throw new HybsSystemException( errMsg,ex );
//		} catch( final IOException ie ) {
//			throw new HybsSystemException(ie.getMessage() );
//		} finally {
//			Closer.ioClose(fis);
//		}
//
//		// 処理後のｻｰﾊﾞのﾌｧｲﾙは削除
//		final File file = new File(filePath);
//		file.delete();
//	}

	/**
	 * ﾀｸﾞﾘﾌﾞｵﾌﾞｼﾞｪｸﾄをﾘﾘｰｽします。
	 * ｷｬｯｼｭされて再利用されるので､ﾌｨｰﾙﾄﾞの初期設定を行います。
	 *
	 * @og.rev 2.0.0.4 (2002/09/27) ｶｽﾀﾑﾀｸﾞの release() ﾒｿｯﾄﾞを､追加
	 * @og.rev 3.1.1.2 (2003/04/04) Tomcat4.1 対応。release2() を doEndTag()で呼ぶ。
	 * @og.rev 3.1.3.0 (2003/04/10) FILE_ENCODE から､ｴﾝｺｰﾄﾞ情報を取得する。
	 * @og.rev 3.1.5.0 (2003/04/20) Content-Disposition を外部から指定できるように変更。
	 * @og.rev 3.5.4.2 (2003/12/15) EXCELのｼｰﾄ名を指定できるように変更。
	 * @og.rev 3.5.4.3 (2004/01/05) EXCEL雛型参照ﾌｧｲﾙのURL,filenameを追加します。
	 * @og.rev 3.8.5.3 (2006/08/07) writerClass 属性の初期値をｼｽﾃﾑﾘｿｰｽより取得します。
	 * @og.rev 3.8.5.3 (2006/08/07) fontName 属性を追加します。
	 * @og.rev 3.8.5.3 (2006/08/07) fontPoint 属性を追加します。
	 * @og.rev 3.8.6.0 (2006/08/23) directory 属性を追加します。
	 * @og.rev 4.0.0.0 (2005/02/28) XMLﾌｧｲﾙにｾｯﾄする､param を追加。
	 * @og.rev 5.2.1.0 (2010/10/01) 書込処理でｺｰﾄﾞﾘｿｰｽのﾗﾍﾞﾙ変換を行うかどうか useRenderer 属性追加
	 * @og.rev 5.5.5.5 (2012/08/24) selectedAll属性を追加
	 * @og.rev 5.5.9.1 (2012/12/07) 事前にﾌｧｲﾙを削除するかどうか。互換性のため､false に設定。推奨は､true
	 * @og.rev 5.7.9.0 (2014/08/08) ﾃﾞｰﾀの書き込み開始位置(skipRowCount)を設定
	 * @og.rev 6.0.2.0 (2014/09/19) EXCEL関係機能追加(useCellStyle,useAutoCellSize,useActiveWorkbook,
	 * 								pageBreakColumn,fileBreakColumn,hyperLinkColumn,addTitleSheet)
	 * @og.rev 6.0.4.0 (2014/11/28) zipFilename属性を追加。
	 * @og.rev 6.1.0.0 (2014/12/26) omitNames 属性を追加
	 * @og.rev 6.2.0.0 (2015/02/27) EXCEL出力のparamLevel初期値変更 3:標準推奨 → 4:個人設定可
	 * @og.rev 5.9.12.1 (2016/09/09) sheetOverwrite で､雛形ｼｰﾄの使用時に､元のｼｰﾄ名を使用します。
	 * @og.rev 6.9.5.0 (2018/04/23) direct 廃止(false固定)
	 * @og.rev 8.0.0.2 (2021/10/15) ﾛｰｶﾙﾌｧｲﾙとｸﾗｳﾄﾞﾌｧｲﾙ間の移動
	 * @og.rev 8.0.1.0 (2021/10/29) useLocal 属性を追加。
	 */
	@Override
	protected void release2() {
		super.release2();
		separator		= TableWriter.TAB_SEPARATOR;   // 項目区切り文字
		headerSequence	= null;				// ﾍｯﾀﾞｰ項目の並び順
		fileURL 		= BASE_URL;
		filename		= HybsSystem.sys( "FILE_FILENAME" );   // ﾌｧｲﾙ名
		sheetName		= "Sheet1";			// 3.5.4.2 (2003/12/15)
		beforeDelete	= false;			// 5.5.9.1 (2012/12/07)
		refFileURL		= BASE_URL;			// 3.5.4.3 (2004/01/05)
		refFilename 	= null;				// 3.5.4.3 (2004/01/05)
		refSheetName	= null;				// 3.5.4.2 (2003/12/15)
		sheetOverwrite	= false;			// 5.9.12.1 (2016/09/09)
		recalcSheetName = null;				// 5.9.12.1 (2016/09/09)
		fontName		= HybsSystem.sys( "TABLE_WRITER_DEFAULT_FONT_NAME" );			// 3.8.5.3 (2006/08/07)
		fontPoint		= HybsSystem.sys( "TABLE_WRITER_DEFAULT_FONT_POINTS" );			// 3.8.5.3 (2006/08/07)
		encode			= HybsSystem.sys( "FILE_ENCODE" );	 // ﾌｧｲﾙｴﾝｺｰﾃﾞｨﾝｸﾞ  "DEFAULT","JISAutoDetect" ,"JIS", "EUC_JP", "MS932", "SJIS" , "Windows-31J" , "Shift_JIS"
		writerClass 	= null;													// 6.2.0.0 (2015/02/27) 個人設定可
		fileAppend		= false;			// ﾌｧｲﾙをAPPENDﾓｰﾄﾞで出力するか
//		direct			= false;			// 6.9.5.0 (2018/04/23) direct 廃止(false固定)
		zip 			= false;
		zipFilename		= null;				// 6.0.4.0 (2014/11/28)
		table			= null;
		tableId 		= HybsSystem.TBL_MDL_KEY ;
//		disposition 	= "attachment";		// 3.8.0.9 (2005/10/17) 初期値変更 inline ⇒ attachment 6.9.5.0 (2018/04/23) 廃止
		param			= null ;			// 4.0.0 (2005/01/31)
		useNumber		= true;				// 3.7.0.2 (2005/02/14)
		skipRowCount	= 0;				// 5.7.9.0 (2014/08/08) ﾃﾞｰﾀの書き込み開始位置を設定
		directory		= null;				// 3.8.6.0 (2006/08/23)
		columns			= null;				// 4.0.0 (2005/12/31)
		omitNames		= null;				// 6.1.0.0 (2014/12/26)
		useRenderer		= HybsSystem.sysBool( "USE_TABLE_WRITER_RENDERER" );	// 5.2.1.0 (2010/10/01)
		selectedAll		= true;				// 5.5.5.5 (2012/08/24)
		useLocal		= false;	// 8.0.1.0 (2021/10/29) ｸﾗｳﾄﾞ設定を使用しない場合は､true

		// 6.0.2.0 (2014/09/19) EXCEL関係機能追加
		useCellStyle		= false;		// EXCEL出力時に､ﾃﾞｰﾀを書き込んだ範囲に罫線を入れるかどうかを指定します
		useAutoCellSize		= false;		// EXCEL出力時に､ｾﾙの幅をﾃﾞｰﾀの幅に自動的に合わせるかどうかを指定します
		useActiveWorkbook	= false;		// EXCEL出力時に､ｾﾙの有効範囲を設定するかどうかを指定します
		pageBreakColumn		= null;			// EXCEL出力時に､ｼｰﾄ変更するｷｰとなるｶﾗﾑ名を指定します(このｶﾗﾑの値がｼｰﾄ名になります)
		fileBreakColumn		= null;			// EXCEL出力時に､ﾌｧｲﾙ名を変更するｷｰとなるｶﾗﾑ名を指定します(このｶﾗﾑの値がﾌｧｲﾙ名になります)
		hyperLinkColumn		= null;			// EXCEL出力時に､Hyperlinkを作成するｷｰとなるｶﾗﾑ名と値となるｶﾗﾑ名を指定します
		addTitleSheet		= null;			// EXCEL出力時に､存在するSheet名一覧を作成する場合に､そのSheet名を指定します。
		localFile			= null;			// 8.0.0.2 (2021/10/15) 一時的に利用するﾛｰｶﾙﾌｧｲﾙ
	}

	/**
	 * TableWriter の実ｵﾌﾞｼﾞｪｸﾄを生成して，引数のPrintWriter に書きだします。
	 *
	 * EXCEL系は､自身で出力先のOutputStreamを作成するので､引数のPrintWriterを
	 * 使わず､directory , filename から､生成します。
	 *
	 * @og.rev 3.5.4.1 (2003/12/01) 引数の PrintWriter を､OutputStream に変更。
	 * @og.rev 3.5.4.2 (2003/12/15) lang 引数も table と同時に設定します。
	 * @og.rev 3.5.4.2 (2003/12/15) TableWriter のｻﾌﾞｸﾗｽ名変更。
	 * @og.rev 3.5.4.3 (2004/01/05) EXCEL雛型参考ﾌｧｲﾙを指定するように変更。
	 * @og.rev 3.5.4.3 (2004/01/05) 引数を､ PrintWriter に戻す。
	 * @og.rev 3.5.4.5 (2004/01/23) 文字ｴﾝｺｰﾄﾞを渡すように変更する。
	 * @og.rev 3.5.6.0 (2004/06/18) 各種ﾌﾟﾗｸﾞｲﾝ関連付け設定を､ｼｽﾃﾑﾊﾟﾗﾒｰﾀ に記述します。
	 * @og.rev 3.8.5.3 (2006/08/07) EXCEL時に fontName と fontPoint 属性をｾｯﾄします。
	 * @og.rev 4.0.0.0 (2005/01/31) ｷｰの指定を､TableWriter. から､TableWriter_ に変更します。
	 * @og.rev 4.0.0.0 (2005/12/31) 外部よりｶﾗﾑ列(CSV形式)を指定できるようにする。
	 * @og.rev 5.2.1.0 (2010/10/01) 書込処理でｺｰﾄﾞﾘｿｰｽのﾗﾍﾞﾙ変換を行うかどうか設定 useRenderer 属性追加
	 * @og.rev 5.7.9.0 (2014/08/08) ﾃﾞｰﾀの書き込み開始位置(skipRowCount)を設定
	 * @og.rev 6.0.2.0 (2014/09/19) EXCEL関係機能追加(useCellStyle,useAutoCellSize,useActiveWorkbook,pageBreakColumn,fileBreakColumn)
	 * @og.rev 6.0.4.0 (2014/11/28) NullPointerException が発生するので､事前にﾁｪｯｸします。
	 * @og.rev 6.1.0.0 (2014/12/26) ﾃﾞﾊﾞｯｸﾞ情報を出力するかどうかを指定
	 * @og.rev 6.1.0.0 (2014/12/26) omitNames 属性を追加
	 * @og.rev 6.4.2.1 (2016/02/05) HybsSystem.url2dir に引数追加。
	 * @og.rev 6.4.3.3 (2016/03/04) HybsSystem.newInstance(String,String) への置き換え。
	 * @og.rev 5.9.12.1 (2016/09/09) sheetOverwrite で､雛形ｼｰﾄの使用時に､元のｼｰﾄ名を使用します。
	 *
	 * @param   outPW 出力先のPrintWriterｵﾌﾞｼﾞｪｸﾄ
	 */
	private void create( final PrintWriter outPW )  {
		// 5.2.1.0 (2010/10/01) writerClass で Renderer系の場合は､useRendererﾌﾗｸﾞに変更する。
		if( "ExcelRenderer".equalsIgnoreCase( writerClass ) ) {
			writerClass = "Excel" ;
			useRenderer = true;
		}
		else if( "Renderer".equalsIgnoreCase( writerClass ) ) {
			writerClass = "Default" ;
			useRenderer = true;
		}

		final TableWriter writer = HybsSystem.newInstance( "TableWriter_" , writerClass );	// 3.5.5.3 (2004/04/09)
		writer.setDBTableModel( table,getLanguage() );
		writer.setSeparator( separator );
		writer.setHeaderSequence( headerSequence );
		writer.setEncode( encode );					// 3.5.4.5 (2004/01/23)
		writer.setAppend( fileAppend );				// 3.5.4.2 (2003/12/15)
		if( param != null ) { writer.setParam( param ); }	// 4.0.0 (2005/01/31)
		writer.setUseNumber( useNumber );			// 3.7.0.2 (2005/02/14)
	//	if( columns == null ) {
	//		columns = (String)getSessionAttribute( HybsSystem.TBL_WRITER_CLMS );
	//		removeSessionAttribute( HybsSystem.TBL_WRITER_CLMS );
	//	}
		writer.setColumns( columns );				// 4.0.0 (2005/12/31)
		writer.setOmitNames( omitNames );			// 6.1.0.0 (2014/12/26)
		writer.setUseRenderer( useRenderer );		// 5.2.1.0 (2010/10/01)
		writer.setDebug( isDebug() );				// 6.1.0.0 (2014/12/26) ﾃﾞﾊﾞｯｸﾞ情報を出力するかどうかを指定
		if( writer.isExcel() ) {					// 3.5.4.3 (2004/01/05)
			writer.setSheetName( sheetName );
			writer.setFilename( directory , filename );					// 6.0.2.0 (2014/09/19) EXCEL関係機能追加
			if( refFileURL != null && refFilename != null ) {
				writer.setRefFilename( HybsSystem.url2dir( refFileURL , refFilename ) );			// 6.4.2.1 (2016/02/05) HybsSystem.url2dir に引数追加
			}
			if( refSheetName != null && refSheetName.length() > 0 ) {	// 5.7.9.0 (2014/08/08) nullﾁｪｯｸ
				writer.setRefSheetName( refSheetName );
			}
			writer.setFontName( fontName )	;							// 5.7.9.0 (2014/08/08) nvalはずし
			if( fontPoint != null && fontPoint.length() > 0 ) {
				writer.setFontPoint( Short.parseShort( fontPoint ) );	// 3.8.5.3 (2006/08/07)
			}
			writer.setSkipRowCount( skipRowCount );						// 5.7.9.0 (2014/08/08)

			writer.setSheetOverwrite( sheetOverwrite );					// 5.9.12.1 (2016/09/09)
			writer.setRecalcSheetName( recalcSheetName );				// 5.9.12.1 (2016/09/09)

			writer.setUseCellStyle( useCellStyle );						// 6.0.2.0 (2014/09/19) EXCEL関係機能追加
			writer.setUseAutoCellSize( useAutoCellSize );				// 6.0.2.0 (2014/09/19) EXCEL関係機能追加
			writer.setUseActiveWorkbook( useActiveWorkbook );			// 6.0.2.0 (2014/09/19) EXCEL関係機能追加
			writer.setPageBreakColumn( pageBreakColumn );				// 6.0.2.0 (2014/09/19) EXCEL関係機能追加
			writer.setFileBreakColumn( fileBreakColumn );				// 6.0.2.0 (2014/09/19) EXCEL関係機能追加
			writer.setHyperLinkColumn( hyperLinkColumn );				// 6.0.2.0 (2014/09/19) EXCEL関係機能追加
			writer.setAddTitleSheet( addTitleSheet );					// 6.0.2.0 (2014/09/19) EXCEL関係機能追加

			writer.writeDBTable();
		}
		else {
			writer.writeDBTable( outPW );
		}
	}

	/**
	 * PrintWriter を取得します。
	 *
	 * ここでは､一般的なﾌｧｲﾙ出力を考慮した PrintWriter を作成します。
	 *
	 * @og.rev 2.2.0.0 (2002/12/17) 中国語(国際化)対応 ｴﾝｺｰﾄﾞの取得方法変更
	 * @og.rev 3.1.3.0 (2003/04/10) FILE_ENCODE から､ｴﾝｺｰﾄﾞ情報を取得する。
	 * @og.rev 3.5.4.1 (2003/12/01) PrintWriter を､OutputStream に変更。
	 * @og.rev 3.5.5.9 (2004/06/07) FileUtil.getBufferedReader を使用
	 * @og.rev 3.7.1.1 (2005/05/23) ﾌｫﾙﾀﾞがない場合は､複数階層分のﾌｫﾙﾀﾞを自動で作成します。
	 * @og.rev 3.8.0.0 (2005/06/07) FileUtil#getPrintWriter を利用。
	 *
	 * @return	ﾌｧｲﾙ書き出し用のPrintWriterｵﾌﾞｼﾞｪｸﾄ
	 */
	private PrintWriter getPrintWriter() {
		if( filename == null ) {
			final String errMsg = "ﾌｧｲﾙ名がｾｯﾄされていません。";
			throw new HybsSystemException( errMsg );
		}

		// 6.4.1.1 (2016/01/16) ﾜﾝｸｯｼｮﾝいれずに return します。
		return FileUtil.getPrintWriter(
//				new File( directory,filename ),
				localFile,							// 8.0.0.2 (2021/10/15) 一時的に利用するﾛｰｶﾙﾌｧｲﾙ
				encode,fileAppend );
	}

	/**
	 * PrintWriter を取得します。
	 *
	 * 引数で指定したOutputStreamを利用して出力します。
	 *
	 * @og.rev 5.5.2.0 (2012/05/01) 新規作成
	 *
	 * @param	os	OutputStreamｵﾌﾞｼﾞｪｸﾄ
	 *
	 * @return	指定したstreamに書き出すPrintWriter
	 */
	private PrintWriter getPrintWriter( final OutputStream os ) {
		// 6.4.1.1 (2016/01/16) ﾜﾝｸｯｼｮﾝいれずに return します。
		return FileUtil.getPrintWriter( os,encode );
	}

	/**
	 * ﾁｪｯｸ分のﾃｰﾌﾞﾙﾓﾃﾞﾙを再構成します。
	 *
	 * @og.rev 5.5.5.5 (2012/08/24) 新規作成
	 * @og.rev 5.5.9.1 (2012/12/07) 内部処理変更
	 *
	 * @param	oldTable	ｺﾋﾟｰ元ﾃｰﾌﾞﾙﾓﾃﾞﾙ
	 *
	 * @return	ﾁｪｯｸした分だけで再構成したﾃｰﾌﾞﾙﾓﾃﾞﾙ
	 */
	private DBTableModel getSelRowsTable( final DBTableModel oldTable ) {
		if( oldTable == null ) { return null; }		// 5.5.9.1 (2012/12/07) 引数のDBTableModelが null の場合の処理。

		final DBTableModel newTable = oldTable.newModel();
		final int[] rowNo = getParameterRows();
		final int rowCount = rowNo.length ;

		for( int i=0; i<rowCount; i++ ) {
			newTable.addValues( oldTable.getValues(rowNo[i]), i );
		}
		return newTable;
	}

	/**
	 * 【TAG】(通常は使いません)結果のDBTableModelを､sessionに登録するときのｷｰを指定します
	 *		(初期値:HybsSystem#TBL_MDL_KEY[={@og.value HybsSystem#TBL_MDL_KEY}])。
	 *
	 * @og.tag
	 * 検索結果より､DBTableModelｵﾌﾞｼﾞｪｸﾄを作成します。これを､下流のviewﾀｸﾞ等に
	 * 渡す場合に､通常は､session を利用します。その場合の登録ｷｰです。
	 * query ﾀｸﾞを同時に実行して､結果を求める場合､同一ﾒﾓﾘに配置される為､
	 * この tableId 属性を利用して､ﾒﾓﾘ空間を分けます。
	 *		(初期値:HybsSystem#TBL_MDL_KEY[={@og.value HybsSystem#TBL_MDL_KEY}])。
	 *
	 * @param	id ﾃｰﾌﾞﾙID (sessionに登録する時のID)
	 */
	public void setTableId( final String id ) {
		tableId = nval( getRequestParameter( id ), tableId );
	}

	/**
	 * 【TAG】可変長ﾌｧｲﾙを作成するときの項目区切り文字をｾｯﾄします
	 *		(初期値:TableWriter#TAB_SEPARATOR[={@og.value org.opengion.hayabusa.io.TableWriter#TAB_SEPARATOR}])。
	 *
	 * @og.tag 可変長ﾌｧｲﾙを作成するときの項目区切り文字をｾｯﾄします。
	 * (初期値:TableWriter#TAB_SEPARATOR[={@og.value org.opengion.hayabusa.io.TableWriter#TAB_SEPARATOR}])。
	 *
	 * @param   separator 項目区切り文字
	 * @see		org.opengion.hayabusa.io.TableWriter#TAB_SEPARATOR
	 */
	public void setSeparator( final String separator ) {
		this.separator = nval( getRequestParameter( separator ),this.separator );
	}

	/**
	 * 【TAG】DBTableModelの出力順(LNSCD など)をｾｯﾄします。
	 *
	 * @og.tag
	 * Label,Name,Size,Class,Data の各ﾌｨｰﾙﾄﾞの頭文字のｱﾙﾌｧﾍﾞｯﾄで
	 * 出力順を設定します。
	 *
	 * N: Name(名前)属性。ﾌｧｲﾙを読み取る(DBTableReader)場合には必要です。
	 * L: Label(ﾗﾍﾞﾙ)属性。指定のｶﾗﾑの名称を表示します。
	 * S: Size(ｻｲｽﾞ)属性。指定のｶﾗﾑの桁数を表示します。
	 * C: Class(ｸﾗｽ)属性。指定のｶﾗﾑのｸﾗｽ(文字､数字など)を表示します。
	 * -: ----(区切り記号)属性。必要な箇所に入れることで､ﾍｯﾀﾞｰ部とﾃﾞｰﾀ部を分離します。
	 * D: Data(ﾃﾞｰﾀ)属性。実際のﾃﾞｰﾀを繰り返し表示します。
	 *
	 * H:#なしのLabel(ﾗﾍﾞﾙ)属性を出力します(EXCELは除く)
	 *
	 * ※ 7.2.6.1 (2020/07/17) H:Header(ﾍｯﾀﾞｰ)属性追加。Label(ﾗﾍﾞﾙ)属性の先頭の#を削除します。
	 *
	 * @param   hs 出力順 (LNSCD など)
	 */
	public void setHeaderSequence( final String hs ) {
		final String[] headers = getRequestParameterValues( hs );
		if( headers != null ) {
			final StringBuilder buf = new StringBuilder( BUFFER_MIDDLE );
			// 7.2.9.4 (2020/11/20) PMD:This for loop can be replaced by a foreach loop
			for( final String head : headers ) {
				if( head != null ) { buf.append( head ); }
			}
//			for( int i=0; i<headers.length; i++ ) {
//				if( headers[i] != null ) { buf.append( headers[i] ); }
//			}
			headerSequence = buf.toString();
		}
	}

	/**
	 * 【TAG】ﾌｧｲﾙ保存先ﾃﾞｨﾚｸﾄﾘ名を指定します
	 *		(初期値:FILE_URL[={@og.value SystemData#FILE_URL}])。
	 *
	 * @og.tag
	 * この属性で指定されるﾃﾞｨﾚｸﾄﾘに､ﾌｧｲﾙをｾｰﾌﾞします。
	 * 指定方法は､通常の fileURL 属性と同様に､先頭が､'/' (UNIX) または､２文字目が､
	 * ":" (Windows)の場合は､指定のURLそのままのﾃﾞｨﾚｸﾄﾘに､そうでない場合は､
	 * ｼｽﾃﾑﾊﾟﾗﾒｰﾀ の FILE_URL 属性で指定のﾌｫﾙﾀﾞの下に､作成されます。
	 * fileURL = "{&#064;USER.ID}" と指定すると､FILE_URL 属性で指定のﾌｫﾙﾀﾞの下に､
	 * さらに､各個人ID別のﾌｫﾙﾀﾞを作成して､そこにｾｰﾌﾞします。
	 *
	 * @og.rev 3.5.4.3 (2004/01/05) 内部処理を､makeFileURL に移動。
	 * @og.rev 4.0.0.0 (2005/01/31) StringUtil.urlAppend ﾒｿｯﾄﾞの利用
	 * @og.rev 4.0.0.0 (2007/11/20) 指定されたﾃﾞｨﾚｸﾄﾘ名の最後が"\"or"/"で終わっていない場合に､"/"を付加する。
	 * @og.rev 6.4.2.1 (2016/02/05) URLの最後に､"/" を追加する処理を廃止。
	 *
	 * @param	url 保存先ﾃﾞｨﾚｸﾄﾘ名
	 */
	public void setFileURL( final String url ) {
		final String furl = nval( getRequestParameter( url ),null );
		if( furl != null ) {
			fileURL = StringUtil.urlAppend( fileURL,furl );
		}
	}

	/**
	 * 【TAG】ﾌｧｲﾙ保存先ﾃﾞｨﾚｸﾄﾘ名を指定します(初期値:ｺﾝﾃｷｽﾄﾊﾟｽ)。
	 *
	 * @og.tag
	 * この属性で指定されるﾃﾞｨﾚｸﾄﾘに､ﾌｧｲﾙをｾｰﾌﾞします。
	 * 絶対ﾊﾟｽに関しては､fileURL の指定と同じですが､相対ﾊﾟｽで指定する場合は､
	 * 基準ﾊﾟｽが､ｺﾝﾃｷｽﾄﾊﾟｽ になります。
	 * この属性は特殊で､互換性のために持っています。
	 * fileUpdate や､fileDownload ｻｰﾌﾞﾚｯﾄなどの基準ﾊﾟｽが､ｺﾝﾃｷｽﾄﾊﾟｽなので､これらの
	 * ﾀｸﾞと組み合わせて使用する場合に､使用します。
	 *
	 * @og.rev 7.2.7.0 (2020/08/07) ｺﾝﾃｷｽﾄﾊﾟｽを基準ﾊﾟｽとするfileURLの設定
	 *
	 * @param	url 保存先ﾃﾞｨﾚｸﾄﾘ名
	 */
	public void setBaseURL( final String url ) {
		fileURL = nval( getRequestParameter( url ),"" );
	}

	/**
	 * 【TAG】ﾌｧｲﾙを作成するときのﾌｧｲﾙ名をｾｯﾄします
	 *		(初期値:FILE_FILENAME[={@og.value SystemData#FILE_FILENAME}])。
	 *
	 * @og.tag ﾌｧｲﾙを作成するときのﾌｧｲﾙ名をｾｯﾄします。
	 * (初期値:ｼｽﾃﾑ定数のFILE_ENCODE[={@og.value SystemData#FILE_FILENAME}])。
	 *
	 * @param   filename ﾌｧｲﾙ名
	 * @see		org.opengion.hayabusa.common.SystemData#FILE_FILENAME
	 */
	public void setFilename( final String filename ) {
		this.filename = nval( getRequestParameter( filename ),this.filename );
	}

	/**
	 * 【TAG】EXCELﾌｧｲﾙを書き出すときのｼｰﾄ名を設定します(初期値:Sheet1)。
	 *
	 * @og.tag
	 * EXCELﾌｧｲﾙを書き出す時に､ｼｰﾄ名を指定します。これにより､複数の形式の
	 * 異なるﾃﾞｰﾀを順次書き出したり(appendﾓｰﾄﾞを併用)することや､ｼｰﾄを指定して
	 * 新規にEXCELを作成することが可能になります。
	 * 初期値は､Sheet1 です。
	 *
	 * @og.rev 3.5.4.2 (2003/12/15) 新規追加
	 *
	 * @param   sheet ｼｰﾄ名
	 */
	public void setSheetName( final String sheet ) {
		sheetName = nval( getRequestParameter( sheet ),sheetName );
	}

	/**
	 * 【TAG】ﾌｧｲﾙを作成するときのﾌｧｲﾙｴﾝｺｰﾃﾞｨﾝｸﾞ名をｾｯﾄします
	 *		(初期値:FILE_ENCODE[={@og.value SystemData#FILE_ENCODE}])。
	 *
	 * @og.tag
	 * "DEFAULT","JISAutoDetect" ,"JIS", "EUC_JP", "MS932", "SJIS" , "Windows-31J" , "Shift_JIS"
	 * (初期値:ｼｽﾃﾑ定数のFILE_ENCODE[={@og.value SystemData#FILE_ENCODE}])。
	 *
	 * @og.rev 2.2.0.0 (2002/12/17) 中国語(国際化)対応 ｴﾝｺｰﾄﾞの取得方法変更
	 * @og.rev 3.1.3.0 (2003/04/10) FILE_ENCODE から､ｴﾝｺｰﾄﾞ情報を取得する。
	 *
	 * @param   enc ｴﾝｺｰﾃﾞｨﾝｸﾞ名 (例:DEFAULT,JISAutoDetect,JIS,EUC_JP,MS932,SJIS,Windows-31J,Shift_JIS)
	 * @see     <a href="http://www.iana.org/assignments/character-sets">IANA Charset Registry</a>
	 * @see		org.opengion.hayabusa.common.SystemData#FILE_ENCODE
	 */
	public void setEncode( final String enc ) {
		encode = nval( getRequestParameter( enc ),encode );
	}

	/**
	 * 【TAG】実際に書き出すｸﾗｽ名の略称({@og.doc03Link writerClass TableWriter_**** ｸﾗｽの ****})を指定します
	 * 		(初期値:TABLE_WRITER_DEFAULT_CLASS[={@og.value SystemData#TABLE_WRITER_DEFAULT_CLASS}])。
	 *
	 * @og.tag
	 * 実際に書き出すｸﾗｽ名(の略称)をｾｯﾄします。
	 * org.opengion.hayabusa.io.TableWriter ｲﾝﾀｰﾌｪｰｽを継承したｻﾌﾞｸﾗｽです。
	 * これは､org.opengion.plugin.io 以下の TableWriter_**** ｸﾗｽの **** を与えます。
	 * 属性ｸﾗｽ定義の {@link org.opengion.hayabusa.io.TableWriter TableWriter} を参照願います。
	 * {@og.doc03Link writerClass TableWriter_**** ｸﾗｽ}
	 *
	 * ExcelRenderer と､Renderer は､特別な名前で､useRenderer 属性に､"true" をｾｯﾄするのと同じ効果があります。
	 * 実際は､ExcelRenderer ⇒ Excel ､Renderer  ⇒ Default のｸﾗｽが呼ばれます。
	 * (初期値:TABLE_WRITER_DEFAULT_CLASS[={@og.value SystemData#TABLE_WRITER_DEFAULT_CLASS}])。
	 *
	 * @param   writerClass ｸﾗｽ名(の略称)
	 * @see		org.opengion.hayabusa.io.TableWriter  TableWriterのｻﾌﾞｸﾗｽ
	 */
	public void setWriterClass( final String writerClass ) {
		this.writerClass = nval( getRequestParameter( writerClass ),this.writerClass );
	}

	/**
	 * 【TAG】追加ﾓｰﾄﾞで書き込むかどうか[true/false]を指定します(初期値:false[新規ﾓｰﾄﾞ])。
	 *
	 * @og.tag DBTableModelのﾌｧｲﾙを､追加ﾓｰﾄﾞで書き込むかどうかをｾｯﾄします。
	 *
	 * @param   flag 追加ﾓｰﾄﾞか [true:追加ﾓｰﾄﾞ/それ以外:新規ﾓｰﾄﾞ]
	 */
	public void setFileAppend( final String flag ) {
		fileAppend = nval( getRequestParameter( flag ),fileAppend );
	}

//	/**
//	 * 【廃止】結果をﾀﾞｲﾚｸﾄにEXCEL起動するかどうか[true/false]を指定します(初期値:false[ﾌｧｲﾙ])。
//	 *
//	 * @og.tag 結果をﾀﾞｲﾚｸﾄに EXCEL ﾌｧｲﾙとして出力するかどうかをｾｯﾄします。
//	 *
//	 * @og.rev 6.9.5.0 (2018/04/23) 物理的に廃止
//	 *
//	 * @param  flag ﾀﾞｲﾚｸﾄ起動か [true:ﾀﾞｲﾚｸﾄ/それ以外:ﾌｧｲﾙ]
//	 * @deprecated 結果をﾀﾞｲﾚｸﾄするとｴﾝｺｰﾄﾞ等でおかしな動きをする可能性がある。
//	 */
//	@Deprecated public void setDirect( final String flag ) {
//		direct = nval( getRequestParameter( flag ),direct );
//	}

	/**
	 * 【TAG】結果をﾌｧｲﾙに出力するときに､ZIPで圧縮するかどうか[true/false]を指定します(初期値:false)。
	 *
	 * @og.tag
	 * 初期値は､圧縮しないです。
	 *
	 * @param  flag ZIPで圧縮する [true:圧縮する/それ以外:しない]
	 */
	public void setZip( final String flag ) {
		zip = nval( getRequestParameter( flag ),zip );
	}

	/**
	 * 【TAG】ZIPﾌｧｲﾙを作成するときのZIPﾌｧｲﾙ名をｾｯﾄします(初期値:filename + ".zip")。
	 *
	 * @og.tag
	 * zip 属性に､true を指定した場合に､ZIPﾌｧｲﾙ化します。その場合のﾌｧｲﾙ名を指定します。
	 * なにも指定しない場合は､filename + ".zip" になります。
	 *
	 * @og.rev 6.0.4.0 (2014/11/28) Zipﾌｧｲﾙ名を外部から与えられるように修正
	 *
	 * @param   zipFile ZIPﾌｧｲﾙ名
	 * @see #setZip( String )
	 */
	public void setZipFilename( final String zipFile ) {
		zipFilename = nval( getRequestParameter( zipFile ),zipFilename );
	}

//	/**
//	 * 【廃止】Content-Disposition [inline:埋め込む/attachment:ﾀﾞｲｱﾛｸﾞを起動]を指定します(初期値:attachment)。
//	 *
//	 * @og.tag
//	 * Content-Disposition を指定することで､ﾌﾞﾗｳｻﾞにｱﾌﾟﾘｹｰｼｮﾝﾃﾞｰﾀを
//	 * 埋め込むのか(inline)､ｾｰﾌﾞ/ｵｰﾌﾟﾝのﾀﾞｲｱﾛｸﾞを起動するのか(attachment)
//	 * 指定します。
//	 * この属性が有効なのは､direct="true" で､zip="false"(初期値)のときのみです。
//	 * 値は､inline ､attachment が指定できます。
//	 * 初期値は､attachment です。
//	 * 注意：動作検証しましたが､inline と attachment で動作に差が現れませんでした。
//	 *
//	 * @og.rev 3.1.5.0 (2003/04/20) Content-Disposition を外部から指定できるように変更。
//	 * @og.rev 3.8.0.9 (2005/10/17) Content-Disposition 初期値変更 inline ⇒ attachment
//	 * @og.rev 6.9.5.0 (2018/04/23) 物理的に廃止
//	 *
//	 * @param  pos 埋め込みするかどうか [inline:埋め込む/attachment:ﾀﾞｲｱﾛｸﾞを起動]
//	 */
//	public void setDisposition( final String pos ) {
//		disposition = nval( getRequestParameter( pos ),disposition );
//		if( !"inline".equals( disposition ) && !"attachment".equals( disposition ) ) {
//			final String errMsg = "disposition には､inline か attachment 以外は指定できません。"
//						+ "disposition=[" + disposition + "]" ;
//			throw new HybsSystemException( errMsg );
//		}
//	}

	/**
	 * 【TAG】EXCEL雛型参照ﾌｧｲﾙのﾃﾞｨﾚｸﾄﾘ名をｾｯﾄします
	 * 		(初期値:FILE_URL[={@og.value SystemData#FILE_URL}])。
	 *
	 * @og.tag
	 * この属性で指定されるﾃﾞｨﾚｸﾄﾘに､ﾌｧｲﾙをｾｰﾌﾞします。
	 * 指定方法は､通常の fileURL 属性と同様に､先頭が､'/' (UNIX) または､２文字目が､
	 * ":" (Windows)の場合は､指定のURLそのままのﾃﾞｨﾚｸﾄﾘに､そうでない場合は､
	 * ｼｽﾃﾑﾊﾟﾗﾒｰﾀ の FILE_URL 属性で指定のﾌｫﾙﾀﾞの下に､作成されます。
	 * fileURL = "{&#064;USER.ID}" と指定すると､FILE_URL 属性で指定のﾌｫﾙﾀﾞの下に､
	 * さらに､各個人ID別のﾌｫﾙﾀﾞを作成して､そこにｾｰﾌﾞします。
	 *
	 * @og.rev 3.5.4.3 (2004/01/05) 新規作成
	 * @og.rev 4.0.0.0 (2005/01/31) StringUtil.urlAppend ﾒｿｯﾄﾞの利用
	 * @og.rev 4.0.0.0 (2007/11/20) 指定されたﾃﾞｨﾚｸﾄﾘ名の最後が"\"or"/"で終わっていない場合に､"/"を付加する。
	 * @og.rev 6.4.2.1 (2016/02/05) URLの最後に､"/" を追加する処理を廃止。
	 *
	 * @param	url 雛型ﾃﾞｨﾚｸﾄﾘ名
	 */
	public void setRefFileURL( final String url ) {
		final String furl = nval( getRequestParameter( url ),null );
		if( furl != null ) {
			refFileURL = StringUtil.urlAppend( refFileURL,furl );
		}
	}

	/**
	 * 【TAG】EXCEL雛型参考ﾌｧｲﾙ名をｾｯﾄします。
	 *
	 * @og.tag
	 * 内部処理的には､雛形ﾌｧｲﾙの雛形ｼｰﾄをﾒﾓﾘ上に構築し､
	 * そのｼｰﾄのｾﾙに､ﾃﾞｰﾀを追加していきます。
	 * 最後には､雛形ｼｰﾄとして存在しない部分は､すべて削除します。
	 * 雛形ｼｰﾄを使わない場合は､新規ｼｰﾄに追加していくことになります。
	 * appendﾓｰﾄﾞの場合は､ｵﾘｼﾞﾅﾙが雛形ﾌｧｲﾙになります。
	 * 雛形ﾌｧｲﾙの指定は､無効になります。
	 * ※ 互換性の為､同じ名前の設定は有効です。
	 *
	 * @og.rev 3.5.4.3 (2004/01/05) 新規作成
	 *
	 * @param   filename 雛型ﾌｧｲﾙ名
	 */
	public void setRefFilename( final String filename ) {
		refFilename = nval( getRequestParameter( filename ),refFilename );
	}

	/**
	 * 【TAG】EXCEL雛型参考ﾌｧｲﾙのｼｰﾄ名を設定します。
	 *
	 * @og.tag
	 * EXCELﾌｧｲﾙを書き出す時に､雛型として参照するｼｰﾄ名を指定します。
	 * これにより､複数の形式の異なるﾃﾞｰﾀを順次書き出したり(appendﾓｰﾄﾞを併用)する
	 * ことや､ｼｰﾄを指定して新規にEXCELを作成する場合にﾌｫｰﾑを設定する事が可能になります。
	 * 初期値は､null(第一ｼｰﾄ) です。
	 *
	 * @og.rev 3.5.4.3 (2004/01/05) 新規追加
	 *
	 * @param   sheet 雛型ｼｰﾄ名
	 */
	public void setRefSheetName( final String sheet ) {
		refSheetName = nval( getRequestParameter( sheet ),refSheetName );
	}

	/**
	 * 【TAG】雛形のｼｰﾄ名を､そのまま使用する(true)か､新規､または､外部指定のｼｰﾄ名を使用する(false)を指定します。(初期値:false[外部指定のｼｰﾄ名を使用])。
	 *
	 * @og.tag
	 * ※ Ver5では､追記ﾓｰﾄﾞ時に､指定ｼｰﾄが存在した場合は上書きします(初期値:false[上書きしない])。5.9.12.1 (2016/09/09)
	 *    Ver6では､追記ﾓｰﾄﾞ時に､雛形を指定できないため､雛形のｼｰﾄ名を､そのまま使用する(true)か､
	 *    新規､または､外部指定のｼｰﾄ名を使用する(false)を指定する属性になります。
	 *
	 * @og.rev 6.5.0.0 (2016/09/30) sheetOverwrite で､雛形ｼｰﾄの使用時に､元のｼｰﾄ名を使用します。
	 *
	 * @param   flag 元のｼｰﾄ名を使用するかどうか [true:使用する/false:新規､または､外部指定のｼｰﾄ名を使用]
	 */
	public void setSheetOverwrite( final String flag ) {
		sheetOverwrite = nval( getRequestParameter( flag ),sheetOverwrite );
	}

	/**
	 * 【TAG】EXCELで､出力処理の最後に､ｾﾙ内の計算式を再計算させるｼｰﾄ名をCSV形式で指定します。
	 *
	 * @og.tag
	 * 通常は利用する必要はありません。
	 * 一覧結果を利用した集計用ｼｰﾄを別に持っているような雛形を利用した出力で利用します。
	 *
	 * @og.rev 6.5.0.0 (2016/09/30) recalcSheetName で､ｾﾙ内の計算式を再計算させるｼｰﾄ名を指定。5.9.12.1 (2016/09/09)
	 *
	 * @param   sheet 対象ｼｰﾄ名(CSV形式文字)
	 */
	public void setRecalcSheetName( final String sheet ) {
		recalcSheetName = nval( getRequestParameter( sheet ),recalcSheetName );
	}

	/**
	 * 【TAG】EXCEL出力時のﾃﾞﾌｫﾙﾄﾌｫﾝﾄ名を設定します
	 *		(初期値:TABLE_WRITER_DEFAULT_FONT_NAME[={@og.value SystemData#TABLE_WRITER_DEFAULT_FONT_NAME}])。
	 *
	 * @og.tag
	 * これは､EXCEL追加機能として実装されています。
	 *
	 * EXCELﾌｧｲﾙを書き出す時に､ﾃﾞﾌｫﾙﾄﾌｫﾝﾄ名を指定します。
	 * ﾌｫﾝﾄ名は､EXCELのﾌｫﾝﾄ名をそのまま使用してください。
	 * 内部的に､POI の org.apache.poi.hssf.usermodel.HSSFFont#setFontName( String )
	 * に設定されます。
	 * (初期値:ｼｽﾃﾑ定数のTABLE_WRITER_DEFAULT_FONT_NAME[={@og.value SystemData#TABLE_WRITER_DEFAULT_FONT_NAME}])。
	 *
	 * @og.rev 3.8.5.3 (2006/08/07) 新規追加
	 *
	 * @param   name ﾃﾞﾌｫﾙﾄﾌｫﾝﾄ名
	 * @see		org.opengion.hayabusa.common.SystemData#TABLE_WRITER_DEFAULT_FONT_NAME
	 */
	public void setFontName( final String name ) {
		fontName = nval( getRequestParameter( name ),fontName );
	}

	/**
	 * 【TAG】EXCEL出力時のﾃﾞﾌｫﾙﾄﾌｫﾝﾄﾎﾟｲﾝﾄ数を設定します
	 *		(初期値:TABLE_WRITER_DEFAULT_FONT_POINTS[={@og.value SystemData#TABLE_WRITER_DEFAULT_FONT_POINTS}])。
	 *
	 * @og.tag
	 * これは､EXCEL追加機能として実装されています。
	 *
	 * EXCELﾌｧｲﾙを書き出す時に､ﾃﾞﾌｫﾙﾄﾎﾟｲﾝﾄ数を指定します。
	 * 内部的に､POI の org.apache.poi.hssf.usermodel.HSSFFont#setFontHeightInPoints( short )
	 * に設定されます。
	 * (初期値:ｼｽﾃﾑ定数のTABLE_WRITER_DEFAULT_FONT_POINTS[={@og.value SystemData#TABLE_WRITER_DEFAULT_FONT_POINTS}])。
	 *
	 * @og.rev 3.8.5.3 (2006/08/07) 新規追加
	 *
	 * @param   point ﾌｫﾝﾄﾎﾟｲﾝﾄ数
	 * @see		org.opengion.hayabusa.common.SystemData#TABLE_WRITER_DEFAULT_FONT_POINTS
	 */
	public void setFontPoint( final String point ) {
		fontPoint = nval( getRequestParameter( point ),fontPoint );
	}

	/**
	 * 【TAG】ﾌｧｲﾙ出力時に､行番号情報を､出力する/しない[true/false]を指定します(初期値:true)。
	 *
	 * @og.tag
	 * 通常のﾌｫｰﾏｯﾄでは､各行の先頭に行番号を出力します。
	 * これは､#NAME 属性を使用する場合には､必ず出力する必要があります。
	 * (#NAME 属性は､読み取り時にあれば､自動的にｶﾗﾑに割り当てられます。)
	 * この､先頭の行番号が不要な場合(つまり､他のｼｽﾃﾑへのﾃﾞｰﾀ出力等)
	 * の為に出力する場合に､false を設定することで､行番号列を出力しない
	 * ようにできます。
	 * 初期値は､true(出力する) です。
	 *
	 * @og.rev 3.7.0.2 (2005/02/14) 新規追加
	 *
	 * @param   useNo 行番号情報 [true:出力する/false:しない]
	 */
	public void setUseNumber( final String useNo ) {
		useNumber = nval( getRequestParameter( useNo ),useNumber );
	}

	/**
	 * 【TAG】書き込み先ﾌｧｲﾙのｶﾗﾑ列を､外部(ﾀｸﾞ)よりCSV形式で指定します。
	 *
	 * @og.tag
	 * この指定により､書き込むｶﾗﾑを限定的に使用することが可能です。
	 * ｶﾗﾑ名は､検索したDBTableModel上に含まれる名前で指定する必要があります。
	 * 別名で書き込みたい場合は､予めその名前で検索しておく必要があります。
	 * ｶﾗﾑ名を指定しない場合(例えば ,,, など)は､ｶﾗﾑを書き飛ばします。
	 *
	 * @og.rev 4.0.0.0 (2005/12/31) 新規作成
	 *
	 * @param   clms 書込ｶﾗﾑ列 (CSV形式)
	 */
	public void setColumns( final String clms ) {
		columns = nval( getRequestParameter( clms ),columns );
	}

	/**
	 * 【TAG】書き込み対象外のｶﾗﾑ列を､外部(ﾀｸﾞ)よりCSV形式で指定します。
	 *
	 * @og.tag
	 * 指定するｶﾗﾑ名に対して､書き込み処理を行いません。
	 * ここで指定するｶﾗﾑ名は､検索したDBTableModel上に含まれる必要はありません。
	 * その場合は､ここでの指定は無視されます。
	 *
	 * @og.rev 6.1.0.0 (2014/12/26) omitNames 属性を追加
	 *
	 * @param   clms 書込対象外ｶﾗﾑ列 (CSV形式)
	 */
	public void setOmitNames( final String clms ) {
		omitNames = nval( getRequestParameter( clms ),omitNames );
	}

	/**
	 * 【TAG】書込処理でｺｰﾄﾞﾘｿｰｽのﾗﾍﾞﾙ変換を行うかどうかを指定します
	 *		(初期値:USE_TABLE_WRITER_RENDERER[={@og.value SystemData#USE_TABLE_WRITER_RENDERER}])。
	 *
	 * @og.tag
	 * ｺｰﾄﾞﾘｿｰｽをそのままの値で出力すると､数字や記号になり何が書かれているのか
	 * 不明になります。
	 * これは､ｺｰﾄﾞﾘｿｰｽをﾗﾍﾞﾙに変換して出力するかどうかを指定します。
	 * 当然､ｺｰﾄﾞはﾕﾆｰｸですが､ﾗﾍﾞﾙはﾕﾆｰｸになるかどうか保障はされていませんので
	 * TableReader 系で読み込む場合には､ﾘｽｸが発生します。
	 * また､TableReader 系で読み込む場合にも､ﾗﾍﾞﾙからｺｰﾄﾞを求める逆変換を行うように､
	 * setUseRenderer ﾒｿｯﾄﾞで指定する必要があります。
	 *
	 * 従来は､TableWriter 系に､TableWriter_Renderer 系のｸﾗｽを作って対応していましたが､
	 * このﾒｿｯﾄﾞの属性値のﾌﾗｸﾞで､制御します。
	 * (初期値:ｼｽﾃﾑ定数のUSE_TABLE_WRITER_RENDERER[={@og.value SystemData#USE_TABLE_WRITER_RENDERER}])。
	 *
	 * @og.rev 5.2.1.0 (2010/10/01) 新規作成
	 *
	 * @param  flag ﾗﾍﾞﾙ変換可否 [true:する/false:しない]
	 * @see		org.opengion.hayabusa.common.SystemData#USE_TABLE_WRITER_RENDERER
	 */
	public void setUseRenderer( final String flag ) {
		useRenderer = nval( getRequestParameter( flag ),useRenderer );
	}

	/**
	 * 【TAG】(通常は使いません)EXCEL出力時のﾃﾞｰﾀの書き込み開始行番号を設定します(初期値:0)。
	 *
	 * @og.tag
	 * EXCELのﾃﾞｰﾀの書き込みの開始行番号を指定します。
	 * 属性名は､行を飛ばす処理ということで､readTable ﾀｸﾞと同じ名称です。
	 * ﾌｧｲﾙの先頭行が､０行としてｶｳﾝﾄしますので､設定値は､読み飛ばす
	 * 件数になります。(１と指定すると､１行読み飛ばし､２行目から読み込みます。)
	 * 行の読み飛ばしと､ｶﾗﾑの読み飛ばし(columns)､refFileURL､refFilename､
	 * refSheetName とともに使用すれば､ある程度のﾚｲｱｳﾄ設定が可能です。
	 * 現時点では､Excel の場合のみ有効です。
	 *
	 * @og.rev 5.7.9.0 (2014/08/08) 新規作成
	 *
	 * @param	count 書き込み開始行番号
	 */
	public void setSkipRowCount( final String count ) {
		skipRowCount = nval( getRequestParameter( count ),skipRowCount );
	}

	/**
	 * 【TAG】EXCEL出力時に､ﾃﾞｰﾀを書き込んだ範囲に罫線を入れるかどうかを指定します(初期値:false)。
	 *
	 * @og.tag
	 * ﾃﾞｰﾀを書き込んでEXCELを作成しても､ﾉｰﾏﾙのｾﾙに値がｾｯﾄされている
	 * だけなので､ある程度加工が必要です。
	 * そこで､ﾃﾞｰﾀのｾｯﾄされたｾﾙに罫線を入れることで､それなりのﾃﾞｰﾀが
	 * 出力された感じになります。
	 * この設定と､useAutoCellSize="true" で､ｾﾙの幅を自動調整すれば､見栄えが良くなります。
	 * なお､この機能は､TableWriter_Excel のみに実装します。
	 *
	 * @og.rev 6.0.2.0 (2014/09/19) 新規作成
	 *
	 * @param	useStyle 罫線追記 [true:する/false:しない]
	 * @see		#setUseAutoCellSize( String )
	 */
	public void setUseCellStyle( final String useStyle ) {
		useCellStyle = nval( getRequestParameter( useStyle ),useCellStyle );
	}

	/**
	 * 【TAG】EXCEL出力時に､ｾﾙの幅をﾃﾞｰﾀの幅に自動的に合わせるかどうかを指定します(初期値:false)。
	 *
	 * @og.tag
	 * ﾃﾞｰﾀを書き込んでEXCELを作成しても､ﾉｰﾏﾙのｾﾙに値がｾｯﾄされている
	 * だけなので､ある程度加工が必要です。
	 * そこで､ﾃﾞｰﾀのｾｯﾄされたｾﾙの幅を自動調整することで､それなりのﾃﾞｰﾀが
	 * 出力された感じになります。
	 * この設定と､useCellStyle="true" で､ｾﾙの罫線を自動設定すれば､見栄えが良くなります。
	 * なお､この機能は､TableWriter_Excel のみに実装します。
	 *
	 * @og.rev 6.0.2.0 (2014/09/19) 新規作成
	 *
	 * @param	useAutoSize 幅の自動的調整 [true:する/false:しない]
	 * @see		#setUseCellStyle( String )
	 */
	public void setUseAutoCellSize( final String useAutoSize ) {
		useAutoCellSize = nval( getRequestParameter( useAutoSize ),useAutoCellSize );
	}

	/**
	 * 【TAG】EXCEL出力時に､ｾﾙの有効範囲を設定するかどうかを指定します(初期値:false)。
	 *
	 * @og.tag
	 * ｾﾙの有効範囲というのは､EXCELでの 空行､空列の存在しない範囲を指します(初期値:false)。
	 * 通常､空行でも､ﾃﾞｰﾀとして残っている場合は､EXCELのｾﾙｵﾌﾞｼﾞｪｸﾄは存在します。
	 * ここで､useActiveWorkbook="true" とすると､空行､空列を削除します。
	 *
	 * 雛形を使用した場合は､ﾃﾞｰﾀより多めに設定した計算などは､この処理で
	 * 削除されますので､ﾃﾞｰﾀｻｲｽﾞにﾌｨｯﾄさせることができます。
	 * なお､この機能は､TableWriter_Excel のみに実装します。
	 *
	 * @og.rev 6.0.2.0 (2014/09/19) 新規作成
	 *
	 * @param	useActWB 有効範囲の設定 [true:する/false:しない]
	 */
	public void setUseActiveWorkbook( final String useActWB ) {
		useActiveWorkbook = nval( getRequestParameter( useActWB ),useActiveWorkbook );
	}

	/**
	 * 【TAG】EXCEL出力時に､ｼｰﾄ変更するｷｰとなるｶﾗﾑ名を指定します(このｶﾗﾑの値がｼｰﾄ名になります)。
	 *
	 * @og.tag
	 * EXCEL帳票では､帳票雛形に､PAGE_BRAKE ｷｰを設定しましたが､TableWriterでは､
	 * ﾒﾓﾘ上のｶﾗﾑの値が変更したときに､ｼｰﾄ変更させることができます。
	 * このｶﾗﾑの値がｷｰﾌﾞﾚｲｸすると､新しいｼｰﾄに書き出し始めます。
	 * ｼｰﾄ名は､このｶﾗﾑの値（ｷｰﾌﾞﾚｲｸする値）です。
	 *
	 * 雛形ﾌｧｲﾙを使用する場合､雛形ｼｰﾄもｷｰﾌﾞﾚｲｸに伴って､＋１されます。
	 * つまり､雛形ｼｰﾄとﾃﾞｰﾀｼｰﾄは同時に変更されます。
	 * ただし､雛形ｼｰﾄは､最後の雛形ｼｰﾄで止まります。
	 * これは､雛形ｼｰﾄにﾍｯﾀﾞｰ雛形とﾎﾞﾃﾞｨ雛形を用意しておき､最初のｷｰﾌﾞﾚｲｸで
	 * ﾍｯﾀﾞｰからﾎﾞﾃﾞｨの書き込みにﾁｪﾝｼﾞするｲﾒｰｼﾞで使用できます。
	 * なお､この機能は､TableWriter_Excel のみに実装します。
	 *
	 * @og.rev 6.0.2.0 (2014/09/19) 新規作成
	 *
	 * @param	pbColumn ｼｰﾄ変更ｶﾗﾑ名
	 * @see		#setFileBreakColumn( String )
	 */
	public void setPageBreakColumn( final String pbColumn ) {
		pageBreakColumn = nval( getRequestParameter( pbColumn ),pageBreakColumn );
	}

	/**
	 * 【TAG】EXCEL出力時に､ﾌｧｲﾙ名を変更するｷｰとなるｶﾗﾑ名を指定します(このｶﾗﾑの値がﾌｧｲﾙ名になります)。
	 *
	 * @og.tag
	 * EXCEL帳票では､ﾒﾓﾘ上のｶﾗﾑの値が変更したときに､ﾌｧｲﾙ名を変更することができます。
	 * このｶﾗﾑの値がｷｰﾌﾞﾚｲｸすると､新しいﾌｧｲﾙに書き出し始めます。
	 * ﾌｧｲﾙ名は､このｶﾗﾑの値（ｷｰﾌﾞﾚｲｸする値）＋ 元の出力ﾌｧｲﾙ名の拡張子(.xlsなど)です。
	 * この設定を使用する場合は､出力ﾌｧｲﾙ名は無視されますが､拡張子だけは使用されます。
	 *
	 * 雛形ﾌｧｲﾙを使用する場合､雛形ﾌｧｲﾙもｷｰﾌﾞﾚｲｸに伴って､再利用されます。
	 * 例えば､pageBreakColumn と併用する場合､ｷｰﾌﾞﾚｲｸで雛形ｼｰﾄも最初から適用になります。
	 * なお､この機能は､TableWriter_Excel のみに実装します。
	 *
	 * @og.rev 6.0.2.0 (2014/09/19) 新規作成
	 *
	 * @param	fbColumn ﾌｧｲﾙﾌﾞﾚｲｸｶﾗﾑ
	 * @see		#setPageBreakColumn( String )
	 */
	public void setFileBreakColumn( final String fbColumn ) {
		fileBreakColumn = nval( getRequestParameter( fbColumn ),fileBreakColumn );
	}

	/**
	 * 【TAG】EXCEL出力時に､Hyperlinkを作成するｷｰとなる｢ｶﾗﾑ名:値｣形式のｶﾗﾑ名を指定します。
	 *
	 * @og.tag
	 * ここで､作成するﾊｲﾊﾟｰﾘﾝｸは､EXCELのｼｰﾄに対するﾊｲﾊﾟｰﾘﾝｸです。
	 * それ以外のﾘﾝｸ(本当のURLやﾌｧｲﾙ等)のﾘﾝｸは(今は)作成できません。
	 * ﾊｲﾊﾟｰﾘﾝｸを作成するには､①作成するｶﾗﾑ と ②作成する値 が必要です。
	 * このﾒｿｯﾄﾞで設定するのは､｢①:②｣という形式でｶﾗﾑ名を指定します。
	 * ②がなければ､①と同じとします。
	 * ②の値のｼｰﾄの存在有無は､無視します。ﾊｲﾊﾟｰﾘﾝｸを作成するｼｰﾄを作成する前に
	 * ﾊｲﾊﾟｰﾘﾝｸを作成するｹｰｽが存在します。
	 * (例えば､各ｼｰﾄへのﾘﾝｸを持った一覧を作成してから､明細の各ｼｰﾄを作成する様なｹｰｽ)
	 * なお､この機能は､TableWriter_Excel のみに実装します。
	 *
	 * @og.rev 6.0.2.0 (2014/09/19) 新規作成
	 *
	 * @param	hlColumn ﾘﾝｸ作成ｶﾗﾑ名 (｢ｶﾗﾑ名:値｣形式)
	 */
	public void setHyperLinkColumn( final String hlColumn ) {
		hyperLinkColumn = nval( getRequestParameter( hlColumn ),hyperLinkColumn );
	}

	/**
	 * 【TAG】EXCEL出力時に､Sheet一覧を作成する場合のSheet名を指定します。
	 *
	 * @og.tag
	 * これは､Workbook に含まれる Sheet 一覧を作成する場合に､利用可能です。
	 * なお､この機能は､TableWriter_Excel のみに実装します。
	 *
	 * @og.rev 6.0.2.0 (2014/09/19) 新規作成
	 *
	 * @param	sheetName ﾀｲﾄﾙｼｰﾄ名
	 */
	public void setAddTitleSheet( final String sheetName ) {
		addTitleSheet = nval( getRequestParameter( sheetName ),addTitleSheet );
	}

	/**
	 * 【TAG】ﾁｪｯｸ行のみを対象にするかを指定します(初期値:true)。
	 *
	 * @og.tag
	 * falseを指定する事で､前画面でﾁｪｯｸした行のみを出力します。
	 * 初期値は全ての行を出力(true)です。
	 * 出力されるﾌｧｲﾙの行番号は､1から振りなおされます。
	 *
	 * @og.rev 5.5.5.5 (2012/08/24) 新規作成
	 *
	 * @param   flag 対象にするか [true:全件/false:対象のみ]
	 */
	public void setSelectedAll( final String flag ) {
		selectedAll = nval( getRequestParameter( flag ),selectedAll );		// 5.5.9.1 (2012/12/07) ﾊﾞｸﾞ対応
	}

	/**
	 * 【TAG】ｼｽﾃﾑ定数でｸﾗｳﾄﾞ設定されていても､ｸﾗｳﾄﾞ環境を使用しない場合､trueを指定します(初期値:false)。
	 *
	 * @og.tag
	 * ｸﾗｳﾄﾞ設定は､ｼｽﾃﾑ定数の『CLOUD_TARGET』と『CLOUD_BUCKET』の設定で自動的に使用しますが､
	 * どうしてもﾛｰｶﾙでのみ使いたい場合は､この属性を true に設定します。
	 * 標準はfalse:設定どおりとなります。
	 * true/false以外を指定した場合はfalse扱いとします。
	 *
	 * @og.rev 8.0.1.0 (2021/10/29) useLocal 属性を追加。
	 *
	 * @param flag ﾛｰｶﾙ環境のみ [true:ﾛｰｶﾙのみ/false:設定どおり]
	 */
	public void setUseLocal( final String flag ) {
		useLocal = nval( getRequestParameter( flag ),useLocal );
	}

	/**
	 * 【TAG】事前にﾌｧｲﾙを削除するかどうかを指定します(初期値:false)。
	 *
	 * @og.tag
	 * 出力元の DBTableModel が､存在しないか､０件の場合､過去に作成したﾌｧｲﾙが
	 * そのまま､残っていると､不要なﾄﾗﾌﾞﾙのもととなる為､実行前に削除するか
	 * どうかを指定できます。
	 * なお､fileAppend="true"（追加ﾓｰﾄﾞで書き込むを指定)時には､ﾃﾞｰﾀが０件でも
	 * 事前削除は行いません。
	 * 初期値は互換性のため削除しない(false)です。
	 *
	 * @og.rev 5.5.9.1 (2012/12/07) 新規作成
	 *
	 * @param   flag 事前削除するかどうか [true:削除する/false:削除しない]
	 */
	public void setBeforeDelete( final String flag ) {
		beforeDelete = nval( getRequestParameter( flag ),beforeDelete );
	}

	/**
	 * WriteTable ｵﾌﾞｼﾞｪｸﾄに渡すﾊﾟﾗﾒｰﾀｵﾌﾞｼﾞｪｸﾄをｾｯﾄします。
	 *
	 * WriteTableParamTag ｸﾗｽよりｾｯﾄされます。
	 *
	 * @og.rev 4.0.0.0 (2005/02/28) 新規追加
	 *
	 * @param   entry HybsEntryｵﾌﾞｼﾞｪｸﾄ
	 */
	protected void addParam( final HybsEntry entry ) {
		if( param == null ) { param = new ArrayList<>(); }
		param.add( entry );
	}

	/**
	 * このｵﾌﾞｼﾞｪｸﾄの文字列表現を返します。
	 * 基本的にﾃﾞﾊﾞｯｸﾞ目的に使用します。
	 *
	 * @return このｸﾗｽの文字列表現
	 * @og.rtnNotNull
	 */
	@Override
	public String toString() {
		return ToString.title( this.getClass().getName() )
				.println( "VERSION"			,VERSION		)
				.println( "BASE_URL"		,BASE_URL		)
				.println( "separator"		,separator		)
				.println( "headerSequence"	,headerSequence	)
				.println( "fileURL"			,fileURL 		)
				.println( "filename"		,filename		)
				.println( "refFileURL"		,refFileURL		)
				.println( "refFilename"		,refFilename 	)
				.println( "encode"			,encode			)
				.println( "writerClass"		,writerClass 	)
				.println( "fileAppend"		,fileAppend		)
//				.println( "direct"			,direct			)		// 6.9.5.0 (2018/04/23) 物理的に廃止
				.println( "zip"				,zip 			)
				.println( "zipFilename"		,zipFilename	)		// 6.0.4.0 (2014/11/28)
//				.println( "disposition"		,disposition	)		// 6.9.5.0 (2018/04/23) 物理的に廃止
				.println( "tableId"			,tableId 		)
				.println( "sheetName"		,sheetName		)
				.println( "refSheetName"	,refSheetName	)
				.println( "useNumber"		,useNumber		)
				.println( "Other..."	,getAttributes().getAttribute() )
				.fixForm().toString() ;
	}
}
