/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.plugin.column;

import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.db.AbstractEditor;
import org.opengion.hayabusa.db.CellEditor;
import org.opengion.hayabusa.db.DBColumn;
import org.opengion.fukurou.util.XHTMLTag;
import org.opengion.fukurou.util.TagBuffer;

import static org.opengion.fukurou.util.StringUtil.isNull;				// 6.1.1.0 (2015/01/17)

/**
 * HM エディターは、カラムのデータを時分編集する場合に使用するクラスです。
 * 選択用ピッカーが付属します。
 *
 * 編集パラメータは、starthour , endhour , minutedivisions , showminutes の順に、
 * CSV形式で与えます。
 * 変数の数が少ない場合は、初期値が設定されます。
 * 初期値は、starthour="6" endhour="20" minutedivisions="4" showminutes="true" です。
 *
 * ex) 6,19,4,true
 * starthour		：ピッカーの開始時間(0-23)
 * endhour			：ピッカーの終了時間(0-23)
 * minutedivisions	：単位「分」の分割数。4なら15分単位。(2-60)
 * showminutes		：分ピッカーの表示(true/false)
 * usesecond		：秒付き(6桁、秒は00固定)で値を返すかどうか(true/false)
 * layout			：表示の縦横(vertical/horizon)
 *
 * このエディタはeventColumnに対応していません。
 *
 *  カラムの表示に必要な属性は, DBColumn オブジェクト より取り出します。
 * このクラスは、DBColumn オブジェクト毎に１つ作成されます。
 *
 * @og.rev 5.6.5.2 (2013/06/21) 新規作成
 * @og.rev 5.9.12.3 (2016/09/23) usesecond追加
 * @og.rev 5.9.17.2 (2017/02/17) layout追加
 * @og.rev 7.3.2.3 (2021/04/09) ｼｽﾃﾑ定数のICON_DIRやIMAGE_DIRを使用します。
 * @og.group データ編集
 *
 * @version  5.0
 * @author   Takahashi Masakazu
 * @since    JDK6.0,
 */
public class Editor_HM extends AbstractEditor {
	/** このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "8.0.3.0 (2021/12/17)" ;

	// 7.3.2.3 (2021/04/09) ｼｽﾃﾑ定数のICON_DIRやIMAGE_DIRを使用します。
	private static final String JSP_ICON =  HybsSystem.sys( "JSP_ICON" ) ;

//	private static final String PIC1 = "<img src=\"../image/clock.png\" class=\"clockpick\"  valuefield=\""; // altは付けない
	private static final String PIC1 = "<img src=\"" + JSP_ICON + "/clock.png\" class=\"clockpick\"  valuefield=\""; // altは付けない

	// 7.0.1.0 (2018/10/15) XHTML → HTML5 対応(空要素の、"／＞" 止めを、"＞" に変更します)。
//	private static final String PIC2 = " />";
	private static final String PIC2 = " >";		// 7.0.1.0 (2018/10/15)

	// 6.9.8.0 (2018/05/28) 賛否両論:前のタグの終了属性を、入れておく事で、文字列連結数を減らす。(属性の飛ばしがないので、良しとします。)
	private static final String OPT1 = "starthour=\"";
//	private static final String OPT2 = "endhour=\"";
//	private static final String OPT3 = "minutedivisions=\"";
//	private static final String OPT4 = "showminutes=\"";
//	private static final String OPT5 = "usesecond=\"";			// 5.9.12.3 (2016/09/23)
//	private static final String OPT6 = "layout=\"";				// 5.9.17.2 (2017/02/17)
	private static final String OPT2 = "\" endhour=\"";
	private static final String OPT3 = "\" minutedivisions=\"";
	private static final String OPT4 = "\" showminutes=\"";
	private static final String OPT5 = "\" usesecond=\"";		// 5.9.12.3 (2016/09/23)
	private static final String OPT6 = "\" layout=\"";			// 5.9.17.2 (2017/02/17)
	private static final String END_ATTR = "\" ";

//	private final String options;
	private final String endTag;		// 6.9.8.0 (2018/05/28) END_ATTR + options + PIC2 を、コンストラクターで行う。

	// 6.9.8.0 (2018/05/28) disabled 時に、時間ピッカーを出さない対応
	private final boolean disabled ;

	/**
	 * デフォルトコンストラクター。
	 * このコンストラクターで、基本オブジェクトを作成します。
	 *
	 * @og.rev 6.9.8.0 (2018/05/28) disabled 時に、時間ピッカーを出さない対応
	 */
	public Editor_HM() {
		super();		// 6.4.1.1 (2016/01/16) PMD refactoring. It is a good practice to call super() in a constructor
		// 4.3.4.4 (2009/01/01)
//		options  = null;		// 6.9.8.0 (2018/05/28) options は、endTag に、入れておく。
		disabled = false;		// 6.9.8.0 (2018/05/28) disabled 時に、時間ピッカーを出さない対応
		endTag   = null;		// 6.9.8.0 (2018/05/28) endTag は、disabled="false" なので、何でも良い。
	}

	/**
	 * DBColumnオブジェクトを指定したprivateコンストラクター。
	 *
	 * @og.rev 6.9.8.0 (2018/05/28) disabled 時に、時間ピッカーを出さない対応
	 *
	 * @param	clm	DBColumnオブジェクト
	 */
	private Editor_HM( final DBColumn clm ) {
		super( clm );
		// 6.9.8.0 (2018/05/28) END_ATTR + options + PIC2 を、コンストラクターで行う。
//		options = createOptions( clm.getEditorParam() );
		tagBuffer.add( XHTMLTag.inputAttri( attributes ) );

		disabled = "disabled".equalsIgnoreCase( attributes.get( "disabled" ) );			// 6.9.8.0 (2018/05/28) disabled対応
		endTag   = disabled ? null
							: END_ATTR + createOptions( clm.getEditorParam() ) + PIC2 ;	// 6.9.8.0 (2018/05/28) options対応
	}

	/**
	 * 各オブジェクトから自分のインスタンスを返します。
	 * 自分自身をキャッシュするのか、新たに作成するのかは、各サブクラスの実装に
	 * まかされます。
	 *
	 * @param	clm	DBColumnオブジェクト
	 *
	 * @return	CellEditorオブジェクト
	 * @og.rtnNotNull
	 */
	public CellEditor newInstance( final DBColumn clm ) {
		return new Editor_HM( clm );
	}

	/**
	 * データの編集用文字列を返します。
	 *
	 * @og.rev 6.9.8.0 (2018/05/28) disabled 時に、時間ピッカーを出さない対応
	 * @og.rev 8.0.3.0 (2021/12/17) ｲﾒｰｼﾞにname属性を付けるため､spanﾀｸﾞで囲います｡
	 *
	 * @param	value 入力値
	 *
	 * @return	データの編集用文字列
	 * @og.rtnNotNull
	 */
	@Override
	public String getValue( final String value ) {
		// 6.1.1.0 (2015/01/17) TagBufferの連結記述
		final String tag = new TagBuffer( "input" )
						.add( "name"	, name )
						.add( "id"		, name , isNull( attributes.get( "id" ) ) )		// 4.3.7.2 (2009/06/15)
						.add( "value"	, value )
						.add( "size"	, size1 )
						.add( tagBuffer.makeTag() )
						.makeTag();

//		return tag + PIC1 + name + END_ATTR + options + PIC2 ;
//		return disabled ? tag										// 6.9.8.0 (2018/05/28) disabled 時に、時間ピッカーを出さない対応
//						: tag + PIC1 + name + endTag ;				// 6.9.8.0 (2018/05/28) END_ATTR + options + PIC2 を、コンストラクターで行う。

		// 8.0.3.0 (2021/12/17)
		if( disabled ) {
			return tag;
		} else {
			final StringBuilder buf = new StringBuilder(BUFFER_MIDDLE)
				.append( tag )
				.append( "<span name=\"img" ).append( name ).append( "\">" )	// 8.0.3.0 (2021/12/17)
				.append( PIC1 )
				.append( name )
				.append( endTag )
				.append( "</span>" );	// 8.0.3.0 (2021/12/17)

			return buf.toString();
		}
	}

	/**
	 * name属性を変えた、データ表示/編集用のHTML文字列を作成します。
	 * テーブル上の name に 行番号を付加して、名前_行番号 で登録するキーを作成し,
	 * リクエスト情報を１つ毎のフィールドで処理できます。
	 *
	 * @og.rev 6.9.8.0 (2018/05/28) disabled 時に、時間ピッカーを出さない対応
	 * @og.rev 8.0.3.0 (2021/12/17) ｲﾒｰｼﾞにname属性を付けるため､spanﾀｸﾞで囲います｡
	 *
	 * @param	row   行番号
	 * @param	value 入力値
	 *
	 * @return	データ表示/編集用の文字列
	 * @og.rtnNotNull
	 */
	@Override
	public String getValue( final int row,final String value ) {
		final String name2 =  name + HybsSystem.JOINT_STRING + row ;

		// 6.1.1.0 (2015/01/17) TagBufferの連結記述
		final String tag = new TagBuffer( "input" )
						.add( "name"	, name2 )
						.add( "id"		, name2 , isNull( attributes.get( "id" ) ) )		// 4.3.7.2 (2009/06/15)
						.add( "value"	, value )
						.add( "size"	, size2 )
						.add( tagBuffer.makeTag() )
						.makeTag( row,value );

//		return tag + PIC1 + name2 + END_ATTR + options + PIC2 ;
//		return disabled ? tag										// 6.9.8.0 (2018/05/28) disabled 時に、時間ピッカーを出さない対応
//						: tag + PIC1 + name2 + endTag ;				// 6.9.8.0 (2018/05/28) END_ATTR + options + PIC2 を、コンストラクターで行う。

		// 8.0.3.0 (2021/12/17)
		if( disabled ) {
			return tag;
		} else {
			final StringBuilder buf = new StringBuilder(BUFFER_MIDDLE)
				.append( tag )
				.append( "<span name=\"img" ).append( name2 ).append( "\">" )	// 8.0.3.0 (2021/12/17)
				.append( PIC1 )
				.append( name2 )
				.append( endTag )
				.append( "</span>" );	// 8.0.3.0 (2021/12/17)

			return buf.toString();
		}
	}

	/**
	 * 編集パラメータから、オプション文字列を作成します。
	 * 編集パラメータは、starthour , endhour , minutedivisions , showminutes の順に、
	 * CSV形式で与えます。
	 * 変数の数が少ない場合は、初期値が設定されます。
	 * 初期値は、starthour="6" endhour="20" minutedivisions="4" showminutes="true" です。
	 *
	 * @og.rev 5.9.12.3 (2016/09/23) usesecond対応
	 * @og.rfv 5.9.17.2 (2017/02/15) layout対応
	 * @og.rfv 6.9.8.0 (2018/05/28) 賛否両論:前のタグの終了属性を、入れておく事で、文字列連結数を減らす。(属性の飛ばしがないので、良しとします。)
	 *
	 * @param	editPrm  編集パラメータ
	 *
	 * @return	パラメータのオプション文字列
	 * @og.rtnNotNull
	 */
	private String createOptions( final String editPrm ){
		if( editPrm == null ) { return "" ;}
		// 6.0.2.5 (2014/10/31) null でないことがわかっている値の冗長な null チェックがあります。
		final String[] param = editPrm.split( "," ) ;
		final String start  = param.length > 0 ? param[0].trim() : "6" ;
		final String end    = param.length > 1 ? param[1].trim() : "20" ;
		final String step   = param.length > 2 ? param[2].trim() : "4" ;
		final String min    = param.length > 3 ? param[3].trim() : "true" ;
		final String sec    = param.length > 4 ? param[4].trim() : "false" ;			// 5.9.12.3 (2016/09/23)
		final String layout = param.length > 5 ? param[5].trim() : "vertical" ;			// 5.9.17.2 (2017/02/15)

//		return OPT1 + start + END_ATTR + OPT2 + end + END_ATTR + OPT3 + step + END_ATTR + OPT4 + min + END_ATTR + OPT5 + sec + END_ATTR + OPT6 + layout + END_ATTR; // 5.9.12.3 (2016/09/23)
		return OPT1 + start + OPT2 + end + OPT3 + step + OPT4 + min + OPT5 + sec + OPT6 + layout + END_ATTR;		// 6.9.8.0 (2018/05/28)
	}
}
