/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.plugin.column;

import java.util.Locale;
import java.util.Map;
import java.util.HashMap;

import org.opengion.hayabusa.common.HybsSystem;						// 7.3.2.3 (2021/04/09)
import org.opengion.hayabusa.db.AbstractRenderer;
import org.opengion.hayabusa.db.CellRenderer;
import org.opengion.hayabusa.db.DBColumn;
import org.opengion.fukurou.util.StringUtil ;						// 6.2.2.0 (2015/03/27)

/**
 * ICON レンデラーは、カラムのファイル名の拡張子からアイコンファイルのイメージタグを作成します。
 * イメージデータは、jsp/image/thumb を使用します。
 *
 * 実質的には、アイコンではなく、サムネイルとして利用します。
 *
 * 縦横比をそのままに、縦か横の最大値に画像サイズを合わせるには、
 * style="max-width:100; max-height:100;" をセットすることで対応できます。
 * class="ICON" 属性を出力しておきますので、CSSファイルで記述してください。
 *
 * (例：)
 *<pre>
 *  &lt;style&gt;
 *      img.ICON { max-width:100px; max-height:100px; }
 *  &lt;/style&gt;
 *</pre>
 *
 * このクラスは、不変オブジェクトとして、共有されます。
 *
 * @og.rev 5.6.5.1 (2013/06/14) 新規作成
 * @og.rev 7.3.2.3 (2021/04/09) ｼｽﾃﾑ定数のJSP_IMGを使用します。(※ SYS.JSP + SYS.IMAGE_DIR)
 *
 * @og.group データ表示
 *
 * @version  4.0
 * @author	 Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class Renderer_ICON extends AbstractRenderer {
	/** このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "7.0.1.0 (2018/10/15)" ;

	private static final CellRenderer DB_CELL = new Renderer_ICON() ;

	// 7.3.2.3 (2021/04/09) ｼｽﾃﾑ定数のJSP_IMGを使用します。(※ SYS.JSP + SYS.IMAGE_DIR)
	private static final String JSP_IMG =  HybsSystem.sys( "JSP_IMG" ) ;

//	private static final String DOC_VIEW = "../image/thumb/docview.png" ;		// その他のアイコン
	private static final String DOC_VIEW = JSP_IMG + "/thumb/docview.png" ;		// その他のアイコン

	// アイコンファイルに割り当てられる拡張子とファイルの関連(MAP)情報
	/** staticイニシャライザ後、読み取り専用にするので、ConcurrentHashMap を使用しません。 */
	private static final Map<String,String> ICON_MAP ;
	static {
		ICON_MAP = new HashMap<>();

//		ICON_MAP.put( "doc"		,	"../image/thumb/doc.png" );
//		ICON_MAP.put( "docx"	,	"../image/thumb/doc.png" );
//		ICON_MAP.put( "xls"		,	"../image/thumb/xls.png" );
//		ICON_MAP.put( "xlsx"	,	"../image/thumb/xls.png" );
//		ICON_MAP.put( "xlsm"	,	"../image/thumb/xls.png" );	 // 6.2.2.0 (2015/03/27) マクロ付Excel(.xlsm)対応
//		ICON_MAP.put( "ppt"		,	"../image/thumb/ppt.png" );
//		ICON_MAP.put( "pptx"	,	"../image/thumb/ppt.png" );
//		ICON_MAP.put( "pdf"		,	"../image/thumb/pdf.png" );
//		ICON_MAP.put( "txt"		,	"../image/thumb/text.png" );
//		ICON_MAP.put( "zip"		,	"../image/thumb/zip.png" );

		ICON_MAP.put( "doc"		,	JSP_IMG + "/thumb/doc.png" );
		ICON_MAP.put( "docx"	,	JSP_IMG + "/thumb/doc.png" );
		ICON_MAP.put( "xls"		,	JSP_IMG + "/thumb/xls.png" );
		ICON_MAP.put( "xlsx"	,	JSP_IMG + "/thumb/xls.png" );
		ICON_MAP.put( "xlsm"	,	JSP_IMG + "/thumb/xls.png" );	 // 6.2.2.0 (2015/03/27) マクロ付Excel(.xlsm)対応
		ICON_MAP.put( "ppt"		,	JSP_IMG + "/thumb/ppt.png" );
		ICON_MAP.put( "pptx"	,	JSP_IMG + "/thumb/ppt.png" );
		ICON_MAP.put( "pdf"		,	JSP_IMG + "/thumb/pdf.png" );
		ICON_MAP.put( "txt"		,	JSP_IMG + "/thumb/text.png" );
		ICON_MAP.put( "zip"		,	JSP_IMG + "/thumb/zip.png" );
	}

	/**
	 * デフォルトコンストラクター
	 *
	 * @og.rev 6.4.2.0 (2016/01/29) PMD refactoring. Each class should declare at least one constructor.
	 */
	public Renderer_ICON() { super(); }		// これも、自動的に呼ばれるが、空のメソッドを作成すると警告されるので、明示的にしておきます。

	/**
	 * 各オブジェクトから自分のインスタンスを返します。
	 * 自分自身をキャッシュするのか、新たに作成するのかは、各サブクラスの実装に
	 * まかされます。
	 *
	 * @param	clm	DBColumnオブジェクト
	 *
	 * @return	CellRendererオブジェクト
	 * @og.rtnNotNull
	 */
	public CellRenderer newInstance( final DBColumn clm ) {
		return DB_CELL;
	}

	/**
	 * データの表示用文字列を返します。
	 *
	 * @og.rev 6.0.2.4 (2014/10/17) img タグに、title 属性追記
	 * @og.rev 6.2.2.0 (2015/03/27) BRと\nを相互に変換する処理を追加
	 * @og.rev 6.2.2.3 (2015/04/10) htmlフィルターに、BR→改行処理機能を追加。
	 * @og.rev 7.0.1.0 (2018/10/15) XHTML → HTML5 対応(空要素の、"／＞" 止めを、"＞" に変更します)。
	 *
	 * @param	value 入力値
	 *
	 * @return	データの表示用文字列
	 * @og.rtnNotNull
	 */
	@Override
	public String getValue( final String value ) {
		String icon = null;

		if( value != null ) {
			String sufix = null;
			final int idx = value.lastIndexOf('.');		// 6.0.2.5 (2014/10/31) refactoring
			if( idx >= 0 ) {
				sufix = value.substring( idx+1 ).toLowerCase( Locale.JAPAN );
				icon = ICON_MAP.get( sufix );
			}
		}

		if( icon == null ) { icon = DOC_VIEW; }

		final String title = StringUtil.htmlFilter( value,true );
//		return "<img class=\"ICON\" src=\"" + icon + "\" alt=\"" + title + "\" title=\"" + title + "\" />" ;
		return "<img class=\"ICON\" src=\"" + icon + "\" alt=\"" + title + "\" title=\"" + title + "\" >" ;
	}

	/**
	 * データ出力用の文字列を作成します。
	 * ファイル等に出力する形式を想定しますので、HTMLタグを含まない
	 * データを返します。
	 * 基本は、#getValue( String ) をそのまま返します。
	 *
	 * @og.rev 6.0.4.0 (2014/11/28) データ出力用のレンデラー
	 *
	 * @param   value 入力値
	 *
	 * @return  データ出力用の文字列
	 * @og.rtnNotNull
	 * @see		#getValue( String )
	 */
	@Override
	public String getWriteValue( final String value ) {
		return ( value == null ) ? "" : value;
	}
}
