/*
 * $Header: /cvshome/build/org.osgi.service.dmt/src/org/osgi/service/dmt/RemoteAlertSender.java,v 1.4 2005/11/11 15:43:24 tszeredi Exp $
 *
 * Copyright (c) OSGi Alliance (2004, 2005). All Rights Reserved.
 * 
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 which accompanies this 
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html.
 */

package org.osgi.service.dmt;


/**
 * The RemoteAlertSender can be used to send notifications to (remote) entities
 * identified by principal names. This service is provided by Protocol Adapters,
 * and is used by the Dmt Admin when sending alerts. Implementations of this
 * interface have to be able to connect and send alerts to one or more
 * management servers in a protocol specific way.
 * <p>
 * The properties of the service registration must specify a list of
 * destinations (principals) where the service is capable of sending alerts.
 * This must be done by providing a <code>String</code> array of principal
 * names in the <code>principals</code> registration property.
 * <p>
 * The <code>principals</code> registration property is used when the
 * {@link DmtAdmin#sendAlert DmtAdmin.sendAlert} method is called, to find the
 * proper <code>RemoteAlertSender</code> for the given destination. If the
 * caller does not specify a principal, the alert is only sent if the Dmt Admin
 * finds an appropriate default alert sender (for example if there is only one
 * registered).
 */
public interface RemoteAlertSender {
    /**
     * Sends an alert to a server identified by its principal name. In case the
     * alert is sent in response to a previous
     * {@link DmtSession#execute(String, String, String) execute} command, a
     * correlation identifier can be specified to provide the association
     * between the execute and the alert.
     * <p>
     * The <code>principal</code> parameter specifies which server the alert
     * should be sent to. This parameter can be <code>null</code> if the
     * client does not know the name of the destination. The alert should still
     * be delivered if possible; for example if the alert sender is only
     * connected to one destination.
     * <p>
     * Any exception thrown on this method will be propagated to the original
     * sender of the event, wrapped in a <code>DmtException</code> with the
     * code <code>REMOTE_ERROR</code>.
     * <p>
     * Since sending the alert and receiving acknowledgment for it is 
     * potentially a very time-consuming operation, alerts are sent
     * asynchronously.  This method should attempt to ensure that the alert can
     * be sent successfully, and should throw an exception if it detects any 
     * problems.  If the method returns without error, the alert is accepted for 
     * sending and the implementation must make a best-effort attempt to 
     * deliver it.
     * 
     * @param principal the name identifying the server where the alert should
     *        be sent, can be <code>null</code>
     * @param code the alert code, can be 0 if not needed
     * @param correlator the correlation identifier of an associated EXEC
     *        command, or <code>null</code> if there is no associated EXEC
     * @param items the data of the alert items carried in this alert, can be
     *        empty or <code>null</code> if no alert items are needed
     * @throws Exception if the alert can not be sent to the server
     */
    void sendAlert(String principal, int code, String correlator, 
            AlertItem[] items) throws Exception;
}
