#include <pekoe.h>

#include "kernel.h"
#include "kprintf.h"
#include "idt.h"

extern struct __descriptor g_IDT64[256];

void loadIDT(uint16 size, uint64 address)
{
	__idt idt;
	idt.size = size;
	idt.address = address;
	asm volatile("lidt %0\n" : /* No Output */ : "m"(idt));
}

void loadIDTR(uint16 selector)
{
	asm volatile("ltr %0\n"  : /* No Output */ : "m"(selector));
}

void set_handler(int index, uint16 selector, void (*address)(), int type, int dpl, int ist)
{
	//asm volatile("cli\n");
	g_IDT64[index].offset_low = ( (uint64) address & 0xFFFF );
	g_IDT64[index].offset_high = ( (uint64) address >> 16);
	g_IDT64[index].selector = selector;
	g_IDT64[index].ist = ist;
	g_IDT64[index].type = type;
	g_IDT64[index].dpl = dpl;
	g_IDT64[index].p = 1; /* present */
	g_IDT64[index].s = 0; /* system segment */
	g_IDT64[index].res0 = g_IDT64[index].res1 = 0;
	//asm volatile("sti\n");
}

void add_int_gate(uint16 index, void (*address)())
{
	set_handler(index, AMD64_KCS, address, 0xe, 0, 0);
}

void add_syscall_gate(uint16 index, void (*address)())
{
	set_handler(index, AMD64_KCS, address, 0xe, 3, 0);
}

void add_trap_gate(uint16 index, void (*address)())
{
	set_handler(index, AMD64_KCS, address, 0xf, 0, 0);
}

//extern "C" void error_handler(void);
//extern "C" void syscall_handler(void);

void error_handler(void)
{
	for (;;);
	//printf("error,");
}

void syscall_handler(void)
{
	for (;;);
	//printf("syscall,");
}

void init_handler()
{
	//kprintf("IDT: START\n");
	loadIDT(sizeof(g_IDT64), (uint64) g_IDT64);
	add_int_gate(AMD64_EXC_DIVIDE_ERROR, error_handler);
	add_int_gate(AMD64_EXC_DEBUG, error_handler);
	add_int_gate(AMD64_EXC_NMI, error_handler);
	add_syscall_gate(AMD64_EXC_BREAKPOINT, error_handler);
	add_int_gate(AMD64_EXC_OVERFLOW, error_handler);
	add_int_gate(AMD64_EXC_BOUNDRANGE, error_handler);
	add_int_gate(AMD64_EXC_INVALIDOPCODE, error_handler);
	add_int_gate(AMD64_EXC_NOMATH_COPROC, error_handler);
	add_int_gate(AMD64_EXC_DOUBLEFAULT, error_handler);
	add_int_gate(AMD64_EXC_COPSEG_OVERRUN, error_handler);
	add_int_gate(AMD64_EXC_INVALID_TSS, error_handler);
	add_int_gate(AMD64_EXC_SEGMENT_NOT_PRESENT, error_handler);
	add_int_gate(AMD64_EXC_STACKSEG_FAULT, error_handler);
	add_int_gate(AMD64_EXC_GENERAL_PROTECTION, error_handler);
	add_int_gate(AMD64_EXC_PAGEFAULT, error_handler);
	/* 15 reserved */
	add_int_gate(AMD64_EXC_FPU_FAULT, error_handler);
	add_int_gate(AMD64_EXC_ALIGNEMENT_CHECK, error_handler);
	add_int_gate(AMD64_EXC_MACHINE_CHECK, error_handler);
	add_int_gate(AMD64_EXC_SIMD_FAULT, error_handler);
	/* syscall */
	for (int i = AMD64_EXC_SIMD_FAULT + 1; i < 0xFF; i++) {
		add_syscall_gate(i, syscall_handler);
	}
	//kprintf("IDT: END\n");
	//add_syscall_gate(0x21, syscall_handler);
	//add_syscall_gate(0x80, syscall_handler);
}
