/*
 * Copyright 2000-2004 The Apache Software Foundation.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package jp.sf.pal.vfs;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import java.text.DateFormat;
import java.util.Date;
import java.util.Locale;

import javax.faces.context.ExternalContext;
import javax.faces.context.FacesContext;
import javax.portlet.PortletRequest;

import org.apache.commons.vfs.FileObject;
import org.apache.commons.vfs.FileSystemException;
import org.apache.commons.vfs.FileType;

/**
 * @author <a href="mailto:shinsuke@users.sourceforge.jp">Shinsuke Sugaya</a>
 */
public class FileParameter
{
    /**
     * Logger for this class
     */
    private static final Log log = LogFactory.getLog(FileParameter.class);

    private static final String NULL_VALUE = "---";

    private FileObject fileObject;

    public FileParameter(FileObject fObject)
    {
        setFileObject(fObject);
    }

    /**
     * @return
     */
    public FileObject getFileObject()
    {
        return fileObject;
    }

    /**
     * @param object
     */
    public void setFileObject(FileObject object)
    {
        fileObject = object;
    }

    public String getBaseName()
    {
        return getFileObject().getName().getBaseName();
    }

    public String getURI()
    {
        return getFileObject().getName().getURI();
    }

    public String getParentURI()
    {
        try
        {
            return getFileObject().getParent().getName().getURI();
        }
        catch (FileSystemException e)
        {
            return "";
        }
    }

    public long getSize()
    {
        try
        {
            return getFileObject().getContent().getSize();
        }
        catch (FileSystemException e)
        {
            return -1;
        }
        catch (Exception e)
        {
            return -1;
        }
    }

    public boolean isFolder()
    {
        try
        {
            if (fileObject.getType() == FileType.FOLDER)
            {
                return true;
            }
        }
        catch (FileSystemException e)
        {
            // TODO Auto-generated catch block
            log.error("isFolder()", e);
        }
        return false;
    }

    public boolean isFile()
    {
        try
        {
            if (fileObject.getType() == FileType.FILE)
            {
                return true;
            }
        }
        catch (FileSystemException e)
        {
            // TODO Auto-generated catch block
            log.error("isFile()", e);
        }
        return false;
    }

    public String getFormattedSize()
    {
        // TODO I18N
        long size = getSize();
        if (size == -1)
        {
            return NULL_VALUE;
        }
        else if (size < 10 * 3)
        {
            return new String(size + "byte");
        }
        else if (size < 10 * 6)
        {
            long value = (long) (size / (10 * 3));
            return new String(value + "K byte");
        }
        else if (size < 10 * 9)
        {
            long value = (long) (size / (10 * 6));
            return new String(value + "M byte");
        }
        else if (size < 10 * 12)
        {
            long value = (long) (size / (10 * 9));
            return new String(value + "G byte");
        }
        return new String(size + "byte");
    }

    public String getContentType()
    {
        String mimeType = null;
        try
        {
            if (fileObject.getContent() == null || fileObject.getType() == null)
            {
                if (log.isDebugEnabled())
                {
                    log.debug("getContentType() - fileObject.getContent() = null or fileObject.getType() == null");
                }
                return NULL_VALUE;
            }

            mimeType = fileObject.getContent().getContentInfo().getContentType();
        }
        catch (FileSystemException e)
        {
            // TODO Auto-generated catch block
            log.error("getContentType()", e);
        }
        catch (Exception e)
        {
            // TODO Auto-generated catch block
            log.error("getContentType()", e);
        }

        if (mimeType == null)
        {
            return NULL_VALUE;
        }
        return mimeType;
    }

    public String getFormattedLastModifiedTime()
    {
        try
        {
            if (fileObject.getContent() == null || fileObject.getType() == null)
            {
                if (log.isDebugEnabled())
                {
                    log.debug("getFormattedLastModifiedTime() - fileObject.getContent() = null or fileObject.getType() == null");
                }
                return NULL_VALUE;
            }
            long time = fileObject.getContent().getLastModifiedTime();

            FacesContext context = FacesContext.getCurrentInstance();
            ExternalContext exContext = context.getExternalContext();
            PortletRequest request = (PortletRequest) exContext.getRequest();
            Locale locale = request.getLocale();
            DateFormat df = DateFormat.getDateTimeInstance(DateFormat.DEFAULT, DateFormat.DEFAULT, locale);
            return df.format(new Date(time));
        }
        catch (FileSystemException e)
        {
            // TODO Auto-generated catch block
            log.error("getFormattedLastModifiedTime()", e);
        }
        catch (Exception e)
        {
            // TODO Auto-generated catch block
            log.error("getFormattedLastModifiedTime()", e);
        }

        return NULL_VALUE;
    }
}
