/*
 * Copyright 2005-2006 The Portal Application Laboratory Team.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package jp.sf.pal.notepad.bean;

import java.util.ArrayList;
import java.util.Calendar;
import java.util.List;

import javax.faces.context.FacesContext;
import javax.faces.model.SelectItem;

import jp.sf.pal.notepad.NotepadConstants;
import jp.sf.pal.notepad.NotepadRuntimeException;
import jp.sf.pal.notepad.model.Category;
import jp.sf.pal.notepad.model.Note;
import jp.sf.pal.notepad.util.NotepadDaoUtil;

import org.apache.commons.lang.StringEscapeUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.marevol.utils.faces.application.FacesMessageUtil;
import com.marevol.utils.portlet.faces.PortletResourceBundleUtil;

public class NotepadEditPageBean
{
    /**
     * Logger for this class
     */
    private static final Log log = LogFactory.getLog(NotepadEditPageBean.class);

    private Long id;

    private String name;

    private String content;

    private String permission;

    private Long categoryId;

    private String categoryName;

    private String owner;

    private NotepadSessionBean notepadSession;

    public String update()
    {
        if (id != null)
        {
            try
            {
                Note note = NotepadDaoUtil.getNoteDao().getNoteById(id.longValue());
                note.setContent(getContent());
                note.setName(getName());
                note.setPermission(getPermission());
                note.setUpdatedtime(Calendar.getInstance().getTime());
                if (getCategoryId().longValue() != -1)
                {
                    Category c = NotepadDaoUtil.getCategoryDao().getCategoryById(getCategoryId().longValue());
                    note.setCategory(c);
                }
                else
                {
                    note.setCategory(null);
                }

                NotepadDaoUtil.getNoteDao().saveOrUpdate(note);

                // i18n
                FacesMessageUtil.addInfoMessage(PortletResourceBundleUtil.getString("notepadedit.UpdatedTheNote"));
            }
            catch (NotepadRuntimeException e)
            {
                // i18n
                FacesMessageUtil.addInfoMessage(PortletResourceBundleUtil
                        .getString("notepadedit.FailedToUpdateTheNote"), e.toString());
                log.error("Failed to update the note.", e);
            }
            catch (RuntimeException e)
            {
                // i18n
                FacesMessageUtil.addInfoMessage(PortletResourceBundleUtil
                        .getString("notepadedit.FailedToUpdateTheNote"), e.toString());
                log.error("Failed to update the note.", e);
            }

            return NotepadConstants.NOTEPAD_VIEW_VIEW_ID;
        }
        else
        {
            // i18n
            FacesMessageUtil.addErrorMessage(PortletResourceBundleUtil.getString("notepadedit.TheSpecifiedItemIsNull"));
            log.warn("The specified item is null.");

            return NotepadConstants.NOTEPAD_VIEW_VIEW_ID;
        }
    }

    public String delete()
    {

        if (id != null)
        {
            try
            {
                Note note = NotepadDaoUtil.getNoteDao().getNoteById(id.longValue());
                NotepadDaoUtil.getNoteDao().delete(note);

                // i18n
                FacesMessageUtil.addInfoMessage(PortletResourceBundleUtil.getString("notepadedit.DeletedTheNote"));
            }
            catch (NotepadRuntimeException e)
            {
                // i18n
                FacesMessageUtil.addInfoMessage(PortletResourceBundleUtil
                        .getString("notepadedit.FailedToDeleteTheNote"), e.toString());
                log.error("Failed to delete the note.", e);
            }
            catch (RuntimeException e)
            {
                // i18n
                FacesMessageUtil.addInfoMessage(PortletResourceBundleUtil
                        .getString("notepadedit.FailedToDeleteTheNote"), e.toString());
                log.error("Failed to delete the note.", e);
            }

            return NotepadConstants.NOTEPAD_VIEW_VIEW_ID;
        }
        else
        {
            // i18n
            FacesMessageUtil.addErrorMessage(PortletResourceBundleUtil.getString("notepadedit.TheSpecifiedItemIsNull"));
            log.warn("The specified item is null.");

            return NotepadConstants.NOTEPAD_VIEW_VIEW_ID;
        }
    }

    public String back()
    {
        return NotepadConstants.NOTEPAD_VIEW_VIEW_ID;
    }

    public SelectItem[] getPermissionItems()
    {
        // i18n
        return new SelectItem[] {
                new SelectItem(NotepadConstants.PUBLIC_PERMISSION, PortletResourceBundleUtil
                        .getString("notepadedit.Public")),
                new SelectItem(NotepadConstants.PRIVATE_PERMISSION, PortletResourceBundleUtil
                        .getString("notepadedit.Private")) };
    }

    public String getPermissionLabel()
    {
        if (getPermission().equals(NotepadConstants.PUBLIC_PERMISSION))
        {
            // i18n
            return PortletResourceBundleUtil.getString("notepadedit.Public");
        }
        else if (getPermission().equals(NotepadConstants.PRIVATE_PERMISSION))
        {
            // i18n
            return PortletResourceBundleUtil.getString("notepadedit.Private");
        }
        return "";
    }

    public SelectItem[] getCategoryItems()
    {
        ArrayList list = new ArrayList();
        // i18n
        list.add(new SelectItem(new Long(-1), PortletResourceBundleUtil.getString("notepadedit.All")));

        String username = FacesContext.getCurrentInstance().getExternalContext().getRemoteUser();
        if (username != null)
        {
            List categories = NotepadDaoUtil.getCategoryDao().getCategoryByScopeOwner(getNotepadSession().getScope(),
                    username);
            if (categories.size() > 0)
            {
                for (int i = 0; i < categories.size(); i++)
                {
                    Category cat = (Category) categories.get(i);
                    list.add(new SelectItem(new Long(cat.getId()), cat.getName()));
                }
            }
        }

        SelectItem[] items = new SelectItem[list.size()];
        for (int i = 0; i < list.size(); i++)
        {
            items[i] = (SelectItem) list.get(i);
        }
        return items;
    }

    /**
     * @return Returns the content.
     */
    public String getContent()
    {
        return content;
    }

    /**
     * @param content The content to set.
     */
    public void setContent(String content)
    {
        this.content = content;
    }

    /**
     * @return Returns the id.
     */
    public Long getId()
    {
        return id;
    }

    /**
     * @param id The id to set.
     */
    public void setId(Long id)
    {
        this.id = id;
    }

    /**
     * @return Returns the name.
     */
    public String getName()
    {
        return name;
    }

    /**
     * @param name The name to set.
     */
    public void setName(String name)
    {
        this.name = name;
    }

    /**
     * @return Returns the notepadSession.
     */
    public NotepadSessionBean getNotepadSession()
    {
        return notepadSession;
    }

    /**
     * @param notepadSession The notepadSession to set.
     */
    public void setNotepadSession(NotepadSessionBean notepadSession)
    {
        this.notepadSession = notepadSession;
        if (notepadSession != null)
        {
            Note note = notepadSession.getNote();
            setContent(note.getContent());
            setId(new Long(note.getId()));
            setName(note.getName());
            setPermission(note.getPermission());
            setOwner(note.getOwner());
            Category c = note.getCategory();
            if (c != null)
            {
                setCategoryId(new Long(c.getId()));
                setCategoryName(c.getName());
            }
            else
            {
                setCategoryId(new Long(-1));
                // i18n
                setCategoryName(PortletResourceBundleUtil.getString("notepadedit.All"));
            }

        }
    }

    /**
     * @return Returns the permission.
     */
    public String getPermission()
    {
        return permission;
    }

    /**
     * @param permission The permission to set.
     */
    public void setPermission(String permission)
    {
        this.permission = permission;
    }

    /**
     * @return Returns the categoryId.
     */
    public Long getCategoryId()
    {
        return categoryId;
    }

    /**
     * @param categoryId The categoryId to set.
     */
    public void setCategoryId(Long categoryId)
    {
        this.categoryId = categoryId;
    }

    /**
     * @return Returns the categoryName.
     */
    public String getCategoryName()
    {
        return categoryName;
    }

    /**
     * @param categoryName The categoryName to set.
     */
    public void setCategoryName(String categoryName)
    {
        this.categoryName = categoryName;
    }

    public String getFormattedContent()
    {
        return StringEscapeUtils.escapeHtml(getContent()).replaceAll("\\n", "<br/>");
    }

    /**
     * @return Returns the owner.
     */
    public String getOwner()
    {
        return owner;
    }

    /**
     * @param owner The owner to set.
     */
    public void setOwner(String owner)
    {
        this.owner = owner;
    }

}
