/*
 * Copyright 2004-2006 The Apache Software Foundation.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package jp.sf.pal.tomahawk.resourcehandler;

import java.util.Enumeration;
import java.util.ResourceBundle;

import javax.portlet.PortletContext;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

/**
 * <p>
 * ResourceHandlerFactory provides the proper ResourceHandler instance by checking portal server info.
 * </p>
 * <p>
 * The portal server info and ResourceHandler class are defined in ContainerInfo.properites. If you want to add a new
 * ResourceHandler class, you need to add information to ContainerInfo.properties.
 * </p>
 * 
 * @author <a href="mailto:shinsuke@yahoo.co.jp">Shinsuke Sugaya</a>
 */
public class ResourceHandlerFactory
{
    private static final Log log = LogFactory
            .getLog(ResourceHandlerFactory.class);

    private static String RESOURCE_BUNDLE_NAME = "jp.sf.pal.tomahawk.resourcehandler.resources.ContainerInfo";

    private static ResourceBundle bundle;
    static
    {
        try
        {
            bundle = ResourceBundle.getBundle(RESOURCE_BUNDLE_NAME);
        }
        catch (Exception e)
        {
            log.warn("Could not find the resource bundole: "
                    + RESOURCE_BUNDLE_NAME, e);
        }
    }

    private static final String PORTAL_NAME_PREFIX = "headerresource.portal.";

    private static final String CLASS_NAME_PREFIX = "headerresource.class.";

    /**
     * Returns a ResourceHandler instance by checking PortletContext#getServerInfo().
     * 
     * @param context portlet's context
     * @return        ResourceHandler instance for each portal server
     */
    public static ResourceHandler getHeaderResource(PortletContext context)
    {
        if (bundle != null)
        {

            String serverInfo = context.getServerInfo();
            for (Enumeration e = bundle.getKeys(); e.hasMoreElements();)
            {
                String key = (String) e.nextElement();
                if (key.startsWith(PORTAL_NAME_PREFIX)
                        && serverInfo.matches(bundle.getString(key)))
                {
                    String classNameKey = new String(CLASS_NAME_PREFIX
                            + key.substring(PORTAL_NAME_PREFIX.length()));
                    try
                    {
                        String className = bundle.getString(classNameKey);
                        if (className != null)
                        {
                            Class clazz = Class.forName(bundle
                                    .getString(classNameKey));
                            if (clazz != null)
                            {
                                Object obj = clazz.newInstance();
                                if (obj instanceof ResourceHandler)
                                {
                                    ResourceHandler resourceHandler = (ResourceHandler) obj;
                                    resourceHandler.setPortletContext(context);
                                    resourceHandler.init();
                                    return resourceHandler;
                                }
                            }
                        }
                        else
                        {
                            log.warn("Could not find " + className);
                        }
                    }
                    catch (InstantiationException ie)
                    {
                        log
                                .warn("Instantiation Exception: "
                                        + classNameKey, ie);
                    }
                    catch (IllegalAccessException iae)
                    {
                        log.warn("Illegal Access Exception: " + classNameKey,
                                iae);
                    }
                    catch (ClassNotFoundException cnfe)
                    {
                        log.warn("Class Not Found: " + classNameKey, cnfe);
                    }
                }
            }
        }

        if (log.isInfoEnabled())
        {
            log.info("Load a default resource handler.");
        }

        return getDefaultHeaderResource(context);
    }

    public static ResourceHandler getDefaultHeaderResource(
            PortletContext context)
    {
        ResourceHandler resourceHandler = new DefaultResourceHandler();
        resourceHandler.setPortletContext(context);
        resourceHandler.init();
        return resourceHandler;
    }
}
