/*
 * Copyright 2004-2006 The Portal Application Laboratory Team.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, 
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package jp.sf.pal.vfs.helper.impl;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;

import javax.mail.internet.MimeUtility;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import jp.sf.pal.vfs.helper.DownloadHelper;

/**
 * @author shinsuke
 * 
 */
public class DownloadHelperImpl implements DownloadHelper {
    /**
     * Logger for this class
     */
    private static final Log log = LogFactory.getLog(DownloadHelperImpl.class);

    private static final int BLOCK_SIZE = 4096;

    private HttpServletRequest request;

    private HttpServletResponse response;

    /*
     * (non-Javadoc)
     * 
     * @see jp.sf.pal.vfs.helper.impl.DownloadHelper#setContentType(javax.servlet.http.HttpServletResponse,
     *      java.lang.String)
     */
    public void setContentType(String type) {
        response.setContentType(type);
    }

    /*
     * (non-Javadoc)
     * 
     * @see jp.sf.pal.vfs.helper.impl.DownloadHelper#setFilename(javax.servlet.http.HttpServletResponse,
     *      java.lang.String)
     */
    public void setFilename(String filename) {
        if (isMSIE()) {
            // for MSIE
            try {
                filename = URLEncoder.encode(filename, "UTF-8");
            } catch (UnsupportedEncodingException e) {
                log.error("Unsupported Encoding.", e);
            }
        } else {
            // for Others
            try {
                filename = MimeUtility.encodeWord(filename, "UTF-8", "B");
            } catch (UnsupportedEncodingException e) {
                log.error("Unsupported Encoding.", e);
            }
        }
        response.setHeader("Content-Disposition", "inline; filename=\""
                + filename + "\"");
    }

    private boolean isMSIE() {
        String agent = request.getHeader("user-agent");
        if (log.isDebugEnabled()) {
            log.debug("isMSIE() - agent=" + agent);
        }
        if (agent.indexOf("MSIE") > 0) {
            return true;
        }
        return false;
    }

    /*
     * (non-Javadoc)
     * 
     * @see jp.sf.pal.vfs.helper.impl.DownloadHelper#writeResponse(javax.servlet.http.HttpServletResponse,
     *      java.io.InputStream)
     */
    public void writeResponse(InputStream in) throws IOException {
        drain(in, response.getOutputStream());
    }

    private void drain(InputStream r, OutputStream w) throws IOException {
        byte[] bytes = new byte[BLOCK_SIZE];
        try {
            int length = r.read(bytes);
            while (length != -1) {
                if (length != 0) {
                    w.write(bytes, 0, length);
                }
                length = r.read(bytes);
            }
        } finally {
            bytes = null;
        }

    }

    /**
     * @return Returns the response.
     */
    public HttpServletResponse getResponse() {
        return response;
    }

    /**
     * @param response
     *            The response to set.
     */
    public void setResponse(HttpServletResponse response) {
        this.response = response;
    }

    /**
     * @return Returns the request.
     */
    public HttpServletRequest getRequest() {
        return request;
    }

    /**
     * @param request
     *            The request to set.
     */
    public void setRequest(HttpServletRequest request) {
        this.request = request;
    }
}
