/*
 * Copyright 2005-2006 The Portal Application Laboratory Team.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package jp.sf.pal.vfs.util;

import java.util.Map;

import javax.faces.context.FacesContext;
import javax.portlet.PortletConfig;
import javax.portlet.PortletContext;
import javax.servlet.ServletContext;

import jp.sf.pal.vfs.VFSConstants;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.commons.vfs.FileSystemManager;
import org.seasar.framework.container.factory.SingletonS2ContainerFactory;

public class VFSUtil {
    /**
     * Logger for this class
     */
    private static final Log log = LogFactory.getLog(VFSUtil.class);

    private static String loadConfig(String key) {
        return getPortletConfig().getInitParameter(key);
    }

    public static boolean checkPath(String path) {
        String acceptPaths[] = getAcceptPaths();
        String denyPaths[] = getDenyPaths();
        if (acceptPaths.length > 0) {
            for (int i = 0; i < acceptPaths.length; i++) {
                if (path.matches(acceptPaths[i])) {
                    return true;
                }
            }
            return false;
        } else {
            if (denyPaths.length > 0) {
                for (int i = 0; i < denyPaths.length; i++) {
                    if (path.matches(denyPaths[i])) {
                        return false;
                    }
                }
                return true;
            } else {
                // accept-paths and deny-paths are not defined
                return true;
            }
        }
    }

    /**
     * @return Returns the acceptPath.
     */
    public static String[] getAcceptPaths() {
        String acceptPathString = loadConfig(VFSConstants.ACCEPT_PATHS);
        if (acceptPathString == null) {
            return new String[0];
        } else {
            return acceptPathString.split(",");
        }
    }

    /**
     * @return Returns the defualtUrl.
     */
    public static String getDefaultUri() {
        String defaultUri = loadConfig(VFSConstants.DEFAULT_URI);
        if (defaultUri == null) {
            defaultUri = "file:///";
        }
        return defaultUri;
    }

    /**
     * @return Returns the denyPaths.
     */
    public static String[] getDenyPaths() {
        String denyPathString = loadConfig(VFSConstants.DENY_PATHS);
        if (denyPathString == null) {
            return new String[0];
        } else {
            return denyPathString.split(",");
        }
    }

    public static PortletConfig getPortletConfig() {
        String name = "javax.portlet.PortletConfig";
        return (PortletConfig) FacesContext.getCurrentInstance()
                .getExternalContext().getRequestMap().get(name);
    }

    public static void publishOutputHtml(String output) {
        FacesContext.getCurrentInstance().getExternalContext().getSessionMap()
                .put(VFSConstants.OUTPUT_HTML, output);
    }

    public static String consumeOutputHtml() {
        Map sessionMap = FacesContext.getCurrentInstance().getExternalContext()
                .getSessionMap();
        String outputHtml = (String) sessionMap.get(VFSConstants.OUTPUT_HTML);
        sessionMap.remove(VFSConstants.OUTPUT_HTML);
        if (outputHtml == null) {
            outputHtml = "";
        }
        return outputHtml;
    }

    public static FileSystemManager getFileSystemManager() {
        return (FileSystemManager) SingletonS2ContainerFactory.getContainer()
                .getComponent(FileSystemManager.class);
    }

    public static String getMimeType(Object context, String filename) {
        if (context instanceof ServletContext) {
            ServletContext servletContext = (ServletContext) context;
            return servletContext.getMimeType(filename);
        } else if (context instanceof PortletContext) {
            PortletContext portletContext = (PortletContext) context;
            return portletContext.getMimeType(filename);
        }
        return null;
    }
}
