/*
 * Copyright 2005-2006 The Portal Application Laboratory Team.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, 
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package jp.sf.pal.cms.dao.impl;

import java.io.ByteArrayInputStream;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.List;

import javax.jcr.AccessDeniedException;
import javax.jcr.InvalidItemStateException;
import javax.jcr.ItemExistsException;
import javax.jcr.Node;
import javax.jcr.NodeIterator;
import javax.jcr.PathNotFoundException;
import javax.jcr.RepositoryException;
import javax.jcr.Session;
import javax.jcr.UnsupportedRepositoryOperationException;
import javax.jcr.ValueFormatException;
import javax.jcr.lock.LockException;
import javax.jcr.nodetype.ConstraintViolationException;
import javax.jcr.nodetype.NoSuchNodeTypeException;
import javax.jcr.query.InvalidQueryException;
import javax.jcr.query.Query;
import javax.jcr.query.QueryResult;
import javax.jcr.version.VersionException;

import jp.sf.pal.cms.CMSConstants;
import jp.sf.pal.cms.CMSException;
import jp.sf.pal.cms.dao.FileNodeDao;
import jp.sf.pal.cms.dto.FileNodeDto;
import jp.sf.pal.cms.util.CMSUtil;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.seasar.framework.container.S2Container;
import org.seasar.jcr.SessionFactory;

/**
 * @author shinsuke
 * 
 */
public class FileNodeDaoImpl implements FileNodeDao {
    /**
     * Logger for this class
     */
    private static final Log log = LogFactory.getLog(FileNodeDaoImpl.class);

    private SessionFactory sessionFactory_;

    private S2Container container;

    public FileNodeDaoImpl(SessionFactory sessionFactory) {
        sessionFactory_ = sessionFactory;
    }

    public List getFileNodes(String scopeName, String path) throws CMSException {
        if (log.isDebugEnabled()) {
            log.debug("getFileNodes(String, String) - scopeName=" + scopeName
                    + ", path=" + path);
        }

        StringBuffer q = new StringBuffer(CMSConstants.PATH_SEPARATOR
                + CMSConstants.PATH_SEPARATOR + CMSConstants.SCOPE_NODE + "[@"
                + CMSConstants.SCOPE_NODE_NAME_ATTR + " = '"
                + CMSUtil.escapeAttributeValue(scopeName) + "']");

        String[] nodes = path.split(CMSConstants.PATH_SEPARATOR);
        for (int i = 1; i < nodes.length; i++) {
            q.append(CMSConstants.PATH_SEPARATOR + CMSConstants.FOLDER_NODE
                    + "[@" + CMSConstants.FOLDER_NODE_NAME_ATTR + "='"
                    + CMSUtil.escapeAttributeValue(nodes[i]) + "']");
        }
        q.append(CMSConstants.PATH_SEPARATOR + "*");
        if (log.isDebugEnabled()) {
            log.debug("getFileNodes(String, String) - q=" + q);
        }

        try {
            Query query = sessionFactory_.getSession().getWorkspace()
                    .getQueryManager().createQuery(q.toString(), Query.XPATH);
            QueryResult result = query.execute();
            NodeIterator ni = result.getNodes();
            List list = new ArrayList();
            while (ni.hasNext()) {
                Node node = ni.nextNode();
                if (log.isDebugEnabled()) {
                    // TODO remove this log
                    log.debug("getFileNodes(String, String) - node.getName()="
                            + node.getName());
                }
                if (node.getName().equals(CMSConstants.FILE_NODE)) {
                    FileNodeDto fileNodeDto = (FileNodeDto) container
                            .getComponent(FileNodeDto.class);
                    // FileNodeDto objectNode = new FileNodeDto();
                    fileNodeDto.setNodeType(CMSConstants.FILE_NODE);
                    fileNodeDto.setScope(scopeName);
                    fileNodeDto.setParentPath(path);
                    fileNodeDto.setName(node.getProperty(
                            CMSConstants.FILE_NODE_NAME_ATTR).getString());

                    // TODO check node type
                    // Node contentNode =
                    // node.getNode(CMSConstants.CONTENT_NODE);
                    Node contentNode = node
                            .getNode(CMSConstants.AUTHORING_CONTENT_NODE);
                    fileNodeDto.setTitle(contentNode.getProperty(
                            CMSConstants.CONTENT_NODE_TITLE_ATTR).getString());
                    fileNodeDto.setDescription(contentNode.getProperty(
                            CMSConstants.CONTENT_NODE_DESCRIPTION_ATTR)
                            .getString());
                    fileNodeDto.setMimeType(contentNode.getProperty(
                            CMSConstants.CONTENT_NODE_MIMETYPE_ATTR)
                            .getString());
                    fileNodeDto.setEncoding(contentNode.getProperty(
                            CMSConstants.CONTENT_NODE_ENCODING_ATTR)
                            .getString());
                    // TODO LANG
                    fileNodeDto.setInputStream(contentNode.getProperty(
                            CMSConstants.CONTENT_NODE_DATA_ATTR).getStream());
                    fileNodeDto.setCreatedTime(node.getProperty(
                            CMSConstants.FILE_NODE_CREATED_ATTR).getDate());
                    fileNodeDto.setLastModifiedTime(contentNode.getProperty(
                            CMSConstants.CONTENT_NODE_LASTMODIFIED_ATTR)
                            .getDate());

                    list.add(fileNodeDto);
                    // } else if
                    // (node.getName().equals(CMSConstants.FOLDER_NODE)) {
                    // ObjectNodeDto objectNode = new FileNodeDto();
                    // objectNode.setNodeType(CMSConstants.FOLDER_NODE);
                    // objectNode.setScope(scopeName);
                    // objectNode.setPath(path);
                    // objectNode.setName(node.getProperty(
                    // CMSConstants.FOLDER_NODE_NAME_ATTR).getString());
                    // list.add(objectNode);
                } else {
                    log.warn("Undefined object:" + node);
                }
            }
            return list;
        } catch (InvalidQueryException e) {
            throw new CMSException("Exception: Query=" + q, e);
        } catch (ItemExistsException e) {
            throw new CMSException("Exception: Query=" + q, e);
        } catch (PathNotFoundException e) {
            throw new CMSException("Exception: Query=" + q, e);
        } catch (VersionException e) {
            throw new CMSException("Exception: Query=" + q, e);
        } catch (ConstraintViolationException e) {
            throw new CMSException("Exception: Query=" + q, e);
        } catch (LockException e) {
            throw new CMSException("Exception: Query=" + q, e);
        } catch (ValueFormatException e) {
            throw new CMSException("Exception: Query=" + q, e);
        } catch (AccessDeniedException e) {
            throw new CMSException("Exception: Query=" + q, e);
        } catch (InvalidItemStateException e) {
            throw new CMSException("Exception: Query=" + q, e);
        } catch (NoSuchNodeTypeException e) {
            throw new CMSException("Exception: Query=" + q, e);
        } catch (RepositoryException e) {
            throw new CMSException("Exception: Query=" + q, e);
        }
    }

    public FileNodeDto getFileNode(String scopeName, String path)
            throws CMSException {
        if (log.isDebugEnabled()) {
            log.debug("getFileNode(String, String) - scopeName=" + scopeName
                    + ", path=" + path);
        }

        StringBuffer q = new StringBuffer(CMSConstants.PATH_SEPARATOR
                + CMSConstants.PATH_SEPARATOR + CMSConstants.SCOPE_NODE + "[@"
                + CMSConstants.SCOPE_NODE_NAME_ATTR + " = '"
                + CMSUtil.escapeAttributeValue(scopeName) + "']");

        String[] nodes = path.split(CMSConstants.PATH_SEPARATOR);
        int i;
        for (i = 1; i < nodes.length - 1; i++) {
            q.append(CMSConstants.PATH_SEPARATOR + CMSConstants.FOLDER_NODE
                    + "[@" + CMSConstants.FOLDER_NODE_NAME_ATTR + "='"
                    + CMSUtil.escapeAttributeValue(nodes[i]) + "']");
        }
        q.append(CMSConstants.PATH_SEPARATOR + CMSConstants.FILE_NODE + "[@"
                + CMSConstants.FILE_NODE_NAME_ATTR + "='"
                + CMSUtil.escapeAttributeValue(nodes[i]) + "']");
        if (log.isDebugEnabled()) {
            log.debug("getFileNode(String, String) - q=" + q);
        }

        try {
            Query query = sessionFactory_.getSession().getWorkspace()
                    .getQueryManager().createQuery(q.toString(), Query.XPATH);
            QueryResult result = query.execute();
            NodeIterator ni = result.getNodes();
            if (ni.hasNext()) {
                Node node = ni.nextNode();
                if (node.getName().equals(CMSConstants.FILE_NODE)) {
                    FileNodeDto fileNode = (FileNodeDto) container
                            .getComponent(FileNodeDto.class);
                    // FileNodeDto fileNode = new FileNodeDto();
                    fileNode.setScope(scopeName);
                    fileNode.setParentPath(CMSUtil.getParentPath(path));
                    fileNode.setName(node.getProperty(
                            CMSConstants.FILE_NODE_NAME_ATTR).getString());

                    // TODO check node type
                    // Node resNode = node.getNode(CMSConstants.CONTENT_NODE);
                    Node resNode = node
                            .getNode(CMSConstants.AUTHORING_CONTENT_NODE);

                    fileNode.setTitle(resNode.getProperty(
                            CMSConstants.CONTENT_NODE_TITLE_ATTR).getString());
                    fileNode.setDescription(resNode.getProperty(
                            CMSConstants.CONTENT_NODE_DESCRIPTION_ATTR)
                            .getString());
                    fileNode.setMimeType(resNode.getProperty(
                            CMSConstants.CONTENT_NODE_MIMETYPE_ATTR)
                            .getString());
                    fileNode.setEncoding(resNode.getProperty(
                            CMSConstants.CONTENT_NODE_ENCODING_ATTR)
                            .getString());
                    // TODO LANG
                    fileNode.setInputStream(resNode.getProperty(
                            CMSConstants.CONTENT_NODE_DATA_ATTR).getStream());
                    fileNode.setCreatedTime(node.getProperty(
                            CMSConstants.FILE_NODE_CREATED_ATTR).getDate());
                    fileNode.setLastModifiedTime(resNode.getProperty(
                            CMSConstants.CONTENT_NODE_LASTMODIFIED_ATTR)
                            .getDate());

                    return fileNode;
                    // } else if
                    // (node.getName().equals(CMSConstants.FOLDER_NODE)) {
                    // ObjectNodeDto objectNode = new FileNodeDto();
                    // objectNode.setNodeType(CMSConstants.FOLDER_NODE);
                    // objectNode.setScope(scopeName);
                    // objectNode.setPath(path);
                    // objectNode.setName(node.getProperty(
                    // CMSConstants.FOLDER_NODE_NAME_ATTR).getString());
                    // list.add(objectNode);
                } else {
                    log.warn("Undefined object:" + node);
                }
            }
            return null;
        } catch (InvalidQueryException e) {
            throw new CMSException("Exception: Query=" + q, e);
        } catch (ItemExistsException e) {
            throw new CMSException("Exception: Query=" + q, e);
        } catch (PathNotFoundException e) {
            throw new CMSException("Exception: Query=" + q, e);
        } catch (VersionException e) {
            throw new CMSException("Exception: Query=" + q, e);
        } catch (ConstraintViolationException e) {
            throw new CMSException("Exception: Query=" + q, e);
        } catch (LockException e) {
            throw new CMSException("Exception: Query=" + q, e);
        } catch (ValueFormatException e) {
            throw new CMSException("Exception: Query=" + q, e);
        } catch (AccessDeniedException e) {
            throw new CMSException("Exception: Query=" + q, e);
        } catch (InvalidItemStateException e) {
            throw new CMSException("Exception: Query=" + q, e);
        } catch (NoSuchNodeTypeException e) {
            throw new CMSException("Exception: Query=" + q, e);
        } catch (RepositoryException e) {
            throw new CMSException("Exception: Query=" + q, e);
        }
    }

    public void insert(FileNodeDto fileNode) throws CMSException {
        if (fileNode.getName() == null || fileNode.getName().equals("")) {
            throw new CMSException("Invalid file name.");
        }
        String scopeName = fileNode.getScope();
        StringBuffer q = new StringBuffer(CMSConstants.PATH_SEPARATOR
                + CMSConstants.PATH_SEPARATOR + CMSConstants.SCOPE_NODE + "[@"
                + CMSConstants.SCOPE_NODE_NAME_ATTR + " = '"
                + CMSUtil.escapeAttributeValue(scopeName) + "']");

        String[] nodes = fileNode.getParentPath().split(
                CMSConstants.PATH_SEPARATOR);
        for (int i = 1; i < nodes.length; i++) {
            q.append(CMSConstants.PATH_SEPARATOR + CMSConstants.FOLDER_NODE
                    + "[@" + CMSConstants.FOLDER_NODE_NAME_ATTR + "='"
                    + CMSUtil.escapeAttributeValue(nodes[i]) + "']");
        }
        if (log.isDebugEnabled()) {
            log.debug("insert(FileNodeDto) - q=" + q);
        }
        try {
            Session session = sessionFactory_.getSession();
            Query query = session.getWorkspace().getQueryManager().createQuery(
                    q.toString(), Query.XPATH);
            QueryResult result = query.execute();
            NodeIterator ni = result.getNodes();
            if (ni.hasNext()) {
                Calendar now = Calendar.getInstance();
                Node folder = ni.nextNode();

                // cms:file
                Node newFileNode = folder.addNode(CMSConstants.FILE_NODE);
                CMSUtil.setProperty(newFileNode,
                        CMSConstants.FILE_NODE_NAME_ATTR, fileNode.getName());
                CMSUtil.setProperty(newFileNode,
                        CMSConstants.FILE_NODE_CREATED_ATTR, fileNode
                                .getCreatedTime(), now);
                // TODO file name update time
                CMSUtil.setProperty(newFileNode,
                        CMSConstants.FILE_NODE_LASTMODIFIED_ATTR, now);

                // jcr:authoring-content
                Node authoringContentNode = newFileNode
                        .addNode(CMSConstants.AUTHORING_CONTENT_NODE);
                CMSUtil.setProperty(authoringContentNode,
                        CMSConstants.CONTENT_NODE_TITLE_ATTR, fileNode
                                .getTitle(), "");
                CMSUtil.setProperty(authoringContentNode,
                        CMSConstants.CONTENT_NODE_DESCRIPTION_ATTR, fileNode
                                .getDescription(), "");
                // TODO default mime type
                CMSUtil.setProperty(authoringContentNode,
                        CMSConstants.CONTENT_NODE_MIMETYPE_ATTR, fileNode
                                .getMimeType(),
                        CMSConstants.DEFAULT_TEXT_MIMETYPE);
                CMSUtil.setProperty(authoringContentNode,
                        CMSConstants.CONTENT_NODE_ENCODING_ATTR, fileNode
                                .getEncoding(), CMSConstants.DEFAULT_ENCODING);
                CMSUtil.setProperty(authoringContentNode,
                        CMSConstants.CONTENT_NODE_LANG_ATTR, null,
                        CMSConstants.DEFAULT_LANG);
                CMSUtil.setProperty(authoringContentNode,
                        CMSConstants.CONTENT_NODE_DATA_ATTR, fileNode
                                .getInputStream(), new ByteArrayInputStream(""
                                .getBytes()));
                CMSUtil.setProperty(authoringContentNode,
                        CMSConstants.CONTENT_NODE_LASTMODIFIED_ATTR, fileNode
                                .getLastModifiedTime(), now);

                // jcr:content
                // Node contentNode = newFileNode
                // .addNode(CMSConstants.CONTENT_NODE);
                // contentNode.addMixin("mix:versionable");
                // CMSUtil.setProperty(contentNode,
                // CMSConstants.CONTENT_NODE_TITLE_ATTR, fileNode
                // .getTitle(), "");
                // CMSUtil.setProperty(contentNode,
                // CMSConstants.CONTENT_NODE_DESCRIPTION_ATTR, fileNode
                // .getDescription(), "");
                // // TODO default mime type
                // CMSUtil.setProperty(contentNode,
                // CMSConstants.CONTENT_NODE_MIMETYPE_ATTR, fileNode
                // .getMimeType(),
                // CMSConstants.DEFAULT_TEXT_MIMETYPE);
                // CMSUtil.setProperty(contentNode,
                // CMSConstants.CONTENT_NODE_ENCODING_ATTR, fileNode
                // .getEncoding(), CMSConstants.DEFAULT_ENCODING);
                // CMSUtil.setProperty(contentNode,
                // CMSConstants.CONTENT_NODE_LANG_ATTR, null,
                // CMSConstants.DEFAULT_LANG);
                // CMSUtil.setProperty(contentNode,
                // CMSConstants.CONTENT_NODE_DATA_ATTR, fileNode
                // .getInputStream(), new ByteArrayInputStream(""
                // .getBytes()));
                // CMSUtil.setProperty(contentNode,
                // CMSConstants.CONTENT_NODE_LASTMODIFIED_ATTR, fileNode
                // .getLastModifiedTime(), now);

                session.save();
                // contentNode.checkin();
            } else {
                throw new CMSException(
                        "Cannot find the target folder: scopeName=" + scopeName
                                + ", path=" + fileNode.getParentPath()
                                + ", name=" + fileNode.getName());
            }
        } catch (InvalidQueryException e) {
            throw new CMSException("Exception: Query=" + q, e);
        } catch (UnsupportedRepositoryOperationException e) {
            throw new CMSException("Exception: Query=" + q, e);
        } catch (ItemExistsException e) {
            throw new CMSException("Exception: Query=" + q, e);
        } catch (PathNotFoundException e) {
            throw new CMSException("Exception: Query=" + q, e);
        } catch (VersionException e) {
            throw new CMSException("Exception: Query=" + q, e);
        } catch (ConstraintViolationException e) {
            throw new CMSException("Exception: Query=" + q, e);
        } catch (LockException e) {
            throw new CMSException("Exception: Query=" + q, e);
        } catch (NoSuchNodeTypeException e) {
            throw new CMSException("Exception: Query=" + q, e);
        } catch (ValueFormatException e) {
            throw new CMSException("Exception: Query=" + q, e);
        } catch (AccessDeniedException e) {
            throw new CMSException("Exception: Query=" + q, e);
        } catch (InvalidItemStateException e) {
            throw new CMSException("Exception: Query=" + q, e);
        } catch (RepositoryException e) {
            throw new CMSException("Exception: Query=" + q, e);
        }
    }

    public void update(FileNodeDto fileNode) throws CMSException {
        if (fileNode.getName() == null || fileNode.getName().equals("")) {
            throw new CMSException("Invalid file name.");
        }
        if (fileNode.getOldName() == null || fileNode.getOldName().equals("")) {
            throw new CMSException("Invalid previous file name.");
        }
        String scopeName = fileNode.getScope();
        StringBuffer q = new StringBuffer(CMSConstants.PATH_SEPARATOR
                + CMSConstants.PATH_SEPARATOR + CMSConstants.SCOPE_NODE + "[@"
                + CMSConstants.SCOPE_NODE_NAME_ATTR + " = '"
                + CMSUtil.escapeAttributeValue(scopeName) + "']");

        String[] nodes = fileNode.getParentPath().split(
                CMSConstants.PATH_SEPARATOR);
        for (int i = 1; i < nodes.length; i++) {
            q.append(CMSConstants.PATH_SEPARATOR + CMSConstants.FOLDER_NODE
                    + "[@" + CMSConstants.FOLDER_NODE_NAME_ATTR + "='"
                    + CMSUtil.escapeAttributeValue(nodes[i]) + "']");
        }
        q.append(CMSConstants.PATH_SEPARATOR + CMSConstants.FILE_NODE + "[@"
                + CMSConstants.FILE_NODE_NAME_ATTR + "='"
                + CMSUtil.escapeAttributeValue(fileNode.getOldName()) + "']");
        // q.append(CMSConstants.PATH_SEPARATOR + CMSConstants.FILE_NODE + "[@"
        // + CMSConstants.FILE_NODE_NAME_ATTR + "='"
        // + CMSUtil.escapeAttributeValue(objectNode.getOldName()) + "']");

        if (log.isDebugEnabled()) {
            log.debug("update(FileNodeDto) - q=" + q);
        }
        try {
            Session session = sessionFactory_.getSession();
            Query query = session.getWorkspace().getQueryManager().createQuery(
                    q.toString(), Query.XPATH);
            QueryResult result = query.execute();
            NodeIterator ni = result.getNodes();
            if (ni.hasNext()) {
                Calendar now = Calendar.getInstance();
                Node updatedFileNode = ni.nextNode();
                CMSUtil.setProperty(updatedFileNode,
                        CMSConstants.FILE_NODE_NAME_ATTR, fileNode.getName());
                if (fileNode.getCreatedTime() != null) {
                    CMSUtil.setProperty(updatedFileNode,
                            CMSConstants.FILE_NODE_CREATED_ATTR, fileNode
                                    .getCreatedTime());
                }
                // else {
                // updatedFileNode.setProperty(
                // CMSConstants.FILE_NODE_CREATED_ATTR, now);
                // }
                // TODO file name update time
                CMSUtil.setProperty(updatedFileNode,
                        CMSConstants.FILE_NODE_LASTMODIFIED_ATTR, now);

                session.save();
            } else {
                throw new CMSException(
                        "Cannot find the target file: scopeName=" + scopeName
                                + ", path=" + fileNode.getParentPath()
                                + ", name=" + fileNode.getName());
            }
        } catch (InvalidQueryException e) {
            throw new CMSException("Exception: Query=" + q, e);
        } catch (UnsupportedRepositoryOperationException e) {
            throw new CMSException("Exception: Query=" + q, e);
        } catch (ItemExistsException e) {
            throw new CMSException("Exception: Query=" + q, e);
        } catch (PathNotFoundException e) {
            throw new CMSException("Exception: Query=" + q, e);
        } catch (VersionException e) {
            throw new CMSException("Exception: Query=" + q, e);
        } catch (ConstraintViolationException e) {
            throw new CMSException("Exception: Query=" + q, e);
        } catch (LockException e) {
            throw new CMSException("Exception: Query=" + q, e);
        } catch (NoSuchNodeTypeException e) {
            throw new CMSException("Exception: Query=" + q, e);
        } catch (ValueFormatException e) {
            throw new CMSException("Exception: Query=" + q, e);
        } catch (AccessDeniedException e) {
            throw new CMSException("Exception: Query=" + q, e);
        } catch (InvalidItemStateException e) {
            throw new CMSException("Exception: Query=" + q, e);
        } catch (RepositoryException e) {
            throw new CMSException("Exception: Query=" + q, e);
        }
    }

    public void updateContent(FileNodeDto fileNode) throws CMSException {
        if (fileNode.getName() == null || fileNode.getName().equals("")) {
            throw new CMSException("Invalid file name.");
        }
        String scopeName = fileNode.getScope();
        StringBuffer q = new StringBuffer(CMSConstants.PATH_SEPARATOR
                + CMSConstants.PATH_SEPARATOR + CMSConstants.SCOPE_NODE + "[@"
                + CMSConstants.SCOPE_NODE_NAME_ATTR + " = '"
                + CMSUtil.escapeAttributeValue(scopeName) + "']");

        String[] nodes = fileNode.getParentPath().split(
                CMSConstants.PATH_SEPARATOR);
        for (int i = 1; i < nodes.length; i++) {
            q.append(CMSConstants.PATH_SEPARATOR + CMSConstants.FOLDER_NODE
                    + "[@" + CMSConstants.FOLDER_NODE_NAME_ATTR + "='"
                    + CMSUtil.escapeAttributeValue(nodes[i]) + "']");
        }
        q.append(CMSConstants.PATH_SEPARATOR + CMSConstants.FILE_NODE + "[@"
                + CMSConstants.FILE_NODE_NAME_ATTR + "='"
                + CMSUtil.escapeAttributeValue(fileNode.getName()) + "']");
        // q.append(CMSConstants.PATH_SEPARATOR + CMSConstants.FILE_NODE + "[@"
        // + CMSConstants.FILE_NODE_NAME_ATTR + "='"
        // + CMSUtil.escapeAttributeValue(objectNode.getName()) + "']");

        if (log.isDebugEnabled()) {
            log.debug("updateContent(FileNodeDto) - q=" + q);
        }
        try {
            Session session = sessionFactory_.getSession();
            Query query = session.getWorkspace().getQueryManager().createQuery(
                    q.toString(), Query.XPATH);
            QueryResult result = query.execute();
            NodeIterator ni = result.getNodes();
            if (ni.hasNext()) {
                Calendar now = Calendar.getInstance();
                Node updatedFileNode = ni.nextNode();
                Node authoringContentNode = updatedFileNode
                        .getNode(CMSConstants.AUTHORING_CONTENT_NODE);
                CMSUtil.setProperty(authoringContentNode,
                        CMSConstants.CONTENT_NODE_TITLE_ATTR, fileNode
                                .getTitle());
                CMSUtil.setProperty(authoringContentNode,
                        CMSConstants.CONTENT_NODE_DESCRIPTION_ATTR, fileNode
                                .getDescription());
                CMSUtil.setProperty(authoringContentNode,
                        CMSConstants.CONTENT_NODE_MIMETYPE_ATTR, fileNode
                                .getMimeType());
                CMSUtil.setProperty(authoringContentNode,
                        CMSConstants.CONTENT_NODE_ENCODING_ATTR, fileNode
                                .getEncoding());
                CMSUtil.setProperty(authoringContentNode,
                        CMSConstants.CONTENT_NODE_LANG_ATTR, (String) null);
                CMSUtil.setProperty(authoringContentNode,
                        CMSConstants.CONTENT_NODE_DATA_ATTR, fileNode
                                .getInputStream());
                CMSUtil.setProperty(authoringContentNode,
                        CMSConstants.CONTENT_NODE_LASTMODIFIED_ATTR, fileNode
                                .getLastModifiedTime(), now);

                // Node contentNode = updatedFileNode
                // .getNode(CMSConstants.CONTENT_NODE);
                // contentNode.checkout();
                // CMSUtil.setProperty(contentNode,
                // CMSConstants.CONTENT_NODE_TITLE_ATTR, fileNode
                // .getTitle());
                // CMSUtil.setProperty(contentNode,
                // CMSConstants.CONTENT_NODE_DESCRIPTION_ATTR, fileNode
                // .getDescription());
                // CMSUtil.setProperty(contentNode,
                // CMSConstants.CONTENT_NODE_MIMETYPE_ATTR, fileNode
                // .getMimeType());
                // CMSUtil.setProperty(contentNode,
                // CMSConstants.CONTENT_NODE_ENCODING_ATTR, fileNode
                // .getEncoding());
                // CMSUtil.setProperty(contentNode,
                // CMSConstants.CONTENT_NODE_LANG_ATTR,(String)null);
                // CMSUtil.setProperty(contentNode,
                // CMSConstants.CONTENT_NODE_DATA_ATTR, fileNode
                // .getInputStream());
                // CMSUtil.setProperty(contentNode,
                // CMSConstants.CONTENT_NODE_LASTMODIFIED_ATTR, fileNode
                // .getLastModifiedTime(), now);

                session.save();
                // contentNode.checkin();
            } else {
                throw new CMSException(
                        "Cannot find the target file: scopeName=" + scopeName
                                + ", path=" + fileNode.getParentPath()
                                + ", name=" + fileNode.getName());
            }
        } catch (InvalidQueryException e) {
            throw new CMSException("Exception: Query=" + q, e);
        } catch (UnsupportedRepositoryOperationException e) {
            throw new CMSException("Exception: Query=" + q, e);
        } catch (ItemExistsException e) {
            throw new CMSException("Exception: Query=" + q, e);
        } catch (PathNotFoundException e) {
            throw new CMSException("Exception: Query=" + q, e);
        } catch (VersionException e) {
            throw new CMSException("Exception: Query=" + q, e);
        } catch (ConstraintViolationException e) {
            throw new CMSException("Exception: Query=" + q, e);
        } catch (LockException e) {
            throw new CMSException("Exception: Query=" + q, e);
        } catch (NoSuchNodeTypeException e) {
            throw new CMSException("Exception: Query=" + q, e);
        } catch (ValueFormatException e) {
            throw new CMSException("Exception: Query=" + q, e);
        } catch (AccessDeniedException e) {
            throw new CMSException("Exception: Query=" + q, e);
        } catch (InvalidItemStateException e) {
            throw new CMSException("Exception: Query=" + q, e);
        } catch (RepositoryException e) {
            throw new CMSException("Exception: Query=" + q, e);
        }
    }

    public void delete(FileNodeDto fileNode) throws CMSException {
        String scopeName = fileNode.getScope();
        StringBuffer q = new StringBuffer(CMSConstants.PATH_SEPARATOR
                + CMSConstants.PATH_SEPARATOR + CMSConstants.SCOPE_NODE + "[@"
                + CMSConstants.SCOPE_NODE_NAME_ATTR + " = '"
                + CMSUtil.escapeAttributeValue(scopeName) + "']");

        String[] nodes = fileNode.getParentPath().split(
                CMSConstants.PATH_SEPARATOR);
        for (int i = 1; i < nodes.length; i++) {
            q.append(CMSConstants.PATH_SEPARATOR + CMSConstants.FOLDER_NODE
                    + "[@" + CMSConstants.FOLDER_NODE_NAME_ATTR + "='"
                    + CMSUtil.escapeAttributeValue(nodes[i]) + "']");
        }
        q.append(CMSConstants.PATH_SEPARATOR + CMSConstants.FILE_NODE + "[@"
                + CMSConstants.FILE_NODE_NAME_ATTR + "='"
                + CMSUtil.escapeAttributeValue(fileNode.getName()) + "']");
        // q.append(CMSConstants.PATH_SEPARATOR + CMSConstants.FILE_NODE + "[@"
        // + CMSConstants.FILE_NODE_NAME_ATTR + "='"
        // + CMSUtil.escapeAttributeValue(objectNode.getName()) + "']");

        if (log.isDebugEnabled()) {
            log.debug("delete(FileNodeDto) - q=" + q);
        }
        try {
            Session session = sessionFactory_.getSession();
            Query query = session.getWorkspace().getQueryManager().createQuery(
                    q.toString(), Query.XPATH);
            QueryResult result = query.execute();
            NodeIterator ni = result.getNodes();
            if (ni.hasNext()) {
                Node deletedFileNode = ni.nextNode();
                deletedFileNode.remove();
                session.save();
            } else {
                throw new CMSException(
                        "Cannot find the target file: scopeName=" + scopeName
                                + ", path=" + fileNode.getParentPath()
                                + ", name=" + fileNode.getName());
            }
        } catch (InvalidQueryException e) {
            throw new CMSException("Exception: Query=" + q, e);
        } catch (VersionException e) {
            throw new CMSException("Exception: Query=" + q, e);
        } catch (LockException e) {
            throw new CMSException("Exception: Query=" + q, e);
        } catch (ConstraintViolationException e) {
            throw new CMSException("Exception: Query=" + q, e);
        } catch (AccessDeniedException e) {
            throw new CMSException("Exception: Query=" + q, e);
        } catch (ItemExistsException e) {
            throw new CMSException("Exception: Query=" + q, e);
        } catch (InvalidItemStateException e) {
            throw new CMSException("Exception: Query=" + q, e);
        } catch (NoSuchNodeTypeException e) {
            throw new CMSException("Exception: Query=" + q, e);
        } catch (RepositoryException e) {
            throw new CMSException("Exception: Query=" + q, e);
        }
    }

    /**
     * @return Returns the container.
     */
    public S2Container getContainer() {
        return container;
    }

    /**
     * @param container
     *            The container to set.
     */
    public void setContainer(S2Container container) {
        this.container = container;
    }

}
