/*
 * Copyright 2005-2006 The Portal Application Laboratory Team.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, 
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package jp.sf.pal.cms.dto;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.UnsupportedEncodingException;

import jp.sf.pal.cms.CMSConstants;
import jp.sf.pal.cms.CMSException;
import jp.sf.pal.cms.dao.FileNodeDao;
import jp.sf.pal.cms.util.CMSUtil;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.seasar.framework.container.S2Container;

public class FileNodeDto extends ObjectNodeDto {
    /**
     * Logger for this class
     */
    private static final Log log = LogFactory.getLog(FileNodeDto.class);

    private String title;

    private String description;

    private String mimeType;

    private String encoding;

    private InputStream inputStream;

    private S2Container container;

    public FileNodeDto() {
        super();
        setNodeType(CMSConstants.FILE_NODE);
        setTitle("");
        setDescription("");
        setMimeType("");
        setEncoding("");
        setInputStream(null);
    }

    public String toString() {
        return super.toString() + //
                "title=" + title + "\n" + //
                "description=" + description + "\n" + //
                "mimeType=" + mimeType + "\n" + //
                "encoding=" + encoding + "\n" + //
                "inputStream=" + inputStream + "\n" + //
                "";
    }

    public boolean isTextEditable() {
        // TODO
        return true;
    }

    public boolean isHtmlEditable() {
        // TODO
        return true;
    }

    /**
     * @return the mimeType
     */
    public String getMimeType() {
        return mimeType;
    }

    /**
     * @param mimeType
     *            the mimeType to set
     */
    public void setMimeType(String mimeType) {
        this.mimeType = mimeType;
    }

    /**
     * @return Returns the description.
     */
    public String getDescription() {
        return description;
    }

    /**
     * @param description
     *            The description to set.
     */
    public void setDescription(String description) {
        this.description = description;
    }

    /**
     * @return Returns the title.
     */
    public String getTitle() {
        return title;
    }

    /**
     * @param title
     *            The title to set.
     */
    public void setTitle(String title) {
        this.title = title;
    }

    /**
     * @return Returns the encoding.
     */
    public String getEncoding() {
        return encoding;
    }

    /**
     * @param encoding
     *            The encoding to set.
     */
    public void setEncoding(String encoding) {
        this.encoding = encoding;
    }

    /**
     * @return Returns the inputStream.
     */
    public InputStream getInputStream() {
        return inputStream;
    }

    /**
     * @param inputStream
     *            The inputStream to set.
     */
    public void setInputStream(InputStream content) {
        this.inputStream = content;
    }

    private String cachedContent;

    public String getContent() {
        if (cachedContent != null) {
            return cachedContent;
        }

        if (getInputStream() != null) {
            cachedContent = getContent(getInputStream(), getEncoding());
            setInputStream(null);
            return cachedContent;
        }

        // S2Container container = SingletonS2ContainerFactory.getContainer();
        FileNodeDao fileNodeDao = (FileNodeDao) container
                .getComponent(FileNodeDao.class);
        FileNodeDto fileNode = null;
        try {
            fileNode = fileNodeDao.getFileNode(getScope(), CMSUtil
                    .normalizedPath(getParentPath()
                            + CMSConstants.PATH_SEPARATOR + getName()));
            if (fileNode != null) {
                cachedContent = getContent(fileNode.getInputStream(), fileNode
                        .getEncoding());
            } else {
                cachedContent = "";
            }
        } catch (CMSException e) {
            log.error(
                    "Could not load the content from the persistent storage.",
                    e);
            cachedContent = "";
        }
        return cachedContent;
    }

    private String getContent(InputStream inputStream, String encoding) {
        ByteArrayOutputStream os = new ByteArrayOutputStream();
        try {
            drain(inputStream, os);
        } catch (IOException e) {
            log.error("Could not load the content from the input stream.", e);
            return "";
        }
        try {
            return os.toString(encoding);
        } catch (UnsupportedEncodingException e) {
            log.warn("Unsupported encoding: " + encoding, e);
            return os.toString();
        } finally {
            try {
                inputStream.close();
            } catch (IOException e) {
                log.error("Could not close the input stream.", e);
            }
        }
    }

    private static int BLOCK_SIZE = 4096;

    private void drain(InputStream r, OutputStream w) throws IOException {
        byte[] bytes = new byte[BLOCK_SIZE];
        try {
            int length = r.read(bytes);
            while (length != -1) {
                if (length != 0) {
                    w.write(bytes, 0, length);
                }
                length = r.read(bytes);
            }
        } finally {
            bytes = null;
        }

    }

    /**
     * @return Returns the container.
     */
    public S2Container getContainer() {
        return container;
    }

    /**
     * @param container
     *            The container to set.
     */
    public void setContainer(S2Container container) {
        this.container = container;
    }
}
