/*
 * Copyright 2005-2006 The Portal Application Laboratory Team.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, 
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package jp.sf.pal.cms.web.editor;

import java.util.Calendar;

import javax.faces.context.FacesContext;

import jp.sf.pal.cms.CMSConstants;
import jp.sf.pal.cms.CMSException;
import jp.sf.pal.cms.dao.FileNodeDao;
import jp.sf.pal.cms.dao.PageNodeDao;
import jp.sf.pal.cms.dto.FileNodeDto;
import jp.sf.pal.cms.dto.PageNodeDto;
import jp.sf.pal.cms.util.CMSUtil;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.seasar.framework.container.S2Container;

import com.marevol.utils.faces.application.FacesMessageUtil;

/**
 * @author shinsuke
 * 
 */
public class DisplayFileDetailPage {
    /**
     * Logger for this class
     */
    private static final Log log = LogFactory
            .getLog(DisplayFileDetailPage.class);

    private String path;

    private String name;

    private S2Container container;

    private boolean textFile;

    private boolean htmlFile;

    public DisplayFileDetailPage(S2Container container) {
        setContainer(container);

        Object obj = FacesContext.getCurrentInstance().getExternalContext()
                .getSessionMap().get(CMSConstants.PATH);
        if (obj != null) {
            setPath((String) obj);
        } else {
            setPath(CMSConstants.ROOT_FOLDER);
        }

        String name = (String) FacesContext.getCurrentInstance()
                .getExternalContext().getSessionMap().get(
                        CMSConstants.SELECTED_OBJECT_NAME);
        if (name != null) {
            setName(name);

            FileNodeDao fileNodeDao = (FileNodeDao) container
                    .getComponent(FileNodeDao.class);
            try {
                FileNodeDto fileNode = fileNodeDao.getFileNode(CMSUtil
                        .getScopeFromRequest(),
                        CMSUtil.normalizedPath(getPath()
                                + CMSConstants.PATH_SEPARATOR)
                                + name);
                String mimeType = fileNode.getMimeType();
                setHtmlFile(CMSUtil.isHtmlMimeType(mimeType));
                setTextFile(CMSUtil.isTextMimeType(mimeType));
            } catch (CMSException e) {
                log.error("Could not open: " + name, e);
                // TODO i18n
                FacesMessageUtil.addErrorMessage("Could not open: " + name, e);
            }
        } else {
            setName("");
            // TODO i18n
            FacesMessageUtil.addErrorMessage("The selected object is null.");
        }
    }

    public String backToList() {
        CMSUtil.clearSession();
        return CMSConstants.DISPLAY_FOLDER_LIST_PAGE;
    }

    public String update() {
        String fileName = (String) FacesContext.getCurrentInstance()
                .getExternalContext().getSessionMap().get(
                        CMSConstants.SELECTED_OBJECT_NAME);
        if (fileName == null) {
            // TODO i18n
            FacesMessageUtil.addErrorMessage("The selected object is null.");
            return null;
        }
        if (log.isDebugEnabled()) {
            log.debug("update() - getPath()=" + getPath());
            log.debug("update() - getName()=" + getName());
            log.debug("update() - name=" + fileName);
        }
        if (!fileName.equals(getName())) {
            // S2Container container =
            // SingletonS2ContainerFactory.getContainer();
            FileNodeDao fileNodeDao = (FileNodeDao) container
                    .getComponent(FileNodeDao.class);
            try {
                FileNodeDto fileNode = fileNodeDao.getFileNode(CMSUtil
                        .getScopeFromRequest(),
                        CMSUtil.normalizedPath(getPath()
                                + CMSConstants.PATH_SEPARATOR)
                                + fileName);
                fileNode.setOldName(fileName);
                fileNode.setName(getName());
                fileNode.setLastModifiedTime(Calendar.getInstance());

                fileNodeDao.update(fileNode);
                FacesContext.getCurrentInstance().getExternalContext()
                        .getSessionMap().put(CMSConstants.SELECTED_OBJECT_NAME,
                                getName());
                // TODO i18n
                FacesMessageUtil.addInfoMessage("Updated this file.");
            } catch (CMSException e) {
                log.error("Could not update this file.", e);
                // TODO i18n
                FacesMessageUtil.addErrorMessage("Could not update this file.",
                        e);
            }
        } else {
            // TODO i18n
            FacesMessageUtil.addInfoMessage("The filename is not changed.");
        }
        return null;
    }

    public String updateTextContent() {
        return CMSConstants.WRITE_TEXT_CONTENT_PAGE;
    }

    public String updateHtmlContent() {
        return CMSConstants.WRITE_HTML_CONTENT_PAGE;
    }

    public String uploadContent() {
        return CMSConstants.UPLOAD_FILE_PAGE;
    }

    public String delete() {
        String fileName = (String) FacesContext.getCurrentInstance()
                .getExternalContext().getSessionMap().get(
                        CMSConstants.SELECTED_OBJECT_NAME);
        if (fileName == null) {
            // TODO i18n
            FacesMessageUtil.addErrorMessage("The selected object is null.");
            return null;
        }
        if (log.isDebugEnabled()) {
            log.debug("delete() - getPath()=" + getPath());
            log.debug("delete() - getName()=" + getName());
            log.debug("delete() - name=" + fileName);
        }
        // S2Container container = SingletonS2ContainerFactory.getContainer();
        FileNodeDao fileNodeDao = (FileNodeDao) container
                .getComponent(FileNodeDao.class);
        try {
            FileNodeDto fileNode = fileNodeDao.getFileNode(CMSUtil
                    .getScopeFromRequest(), CMSUtil.normalizedPath(getPath()
                    + CMSConstants.PATH_SEPARATOR)
                    + fileName);
            fileNodeDao.delete(fileNode);
            // TODO i18n
            FacesMessageUtil.addInfoMessage("Deleted the file.");
            return CMSConstants.DISPLAY_FOLDER_LIST_PAGE;
        } catch (CMSException e) {
            log.error("Could not delete: " + fileName, e);
            // TODO i18n
            FacesMessageUtil
                    .addErrorMessage("Could not delete: " + fileName, e);
        }

        return null;
    }

    public String setAsDefaultPage() {
        String fileName = (String) FacesContext.getCurrentInstance()
                .getExternalContext().getSessionMap().get(
                        CMSConstants.SELECTED_OBJECT_NAME);
        if (fileName == null) {
            // TODO i18n
            FacesMessageUtil.addErrorMessage("The selected object is null.");
            return null;
        }

        // check MIME
        FileNodeDao fileNodeDao = (FileNodeDao) container
                .getComponent(FileNodeDao.class);
        try {
            FileNodeDto fileNode = fileNodeDao.getFileNode(CMSUtil
                    .getScopeFromRequest(), CMSUtil.normalizedPath(getPath()
                    + CMSConstants.PATH_SEPARATOR)
                    + fileName);
            if (fileNode.getMimeType() == null
                    || !CMSUtil.isTextMimeType(fileNode.getMimeType())) {
                // TODO i18n
                FacesMessageUtil
                        .addWarnMessage("This file cannot be set as a default page.");
                return CMSConstants.DISPLAY_FOLDER_LIST_PAGE;
            }
        } catch (CMSException e) {
            log.error("Could not open: " + fileName, e);
            // TODO i18n
            FacesMessageUtil.addErrorMessage("Could not open: " + fileName, e);
        }

        String uri = CMSUtil.normalizedPath(getPath()
                + CMSConstants.PATH_SEPARATOR)
                + fileName;
        PageNodeDao pageNodeDao = (PageNodeDao) getContainer().getComponent(
                PageNodeDao.class);
        String namespace = CMSUtil.getNamespace();
        try {
            PageNodeDto pageNodeDto = pageNodeDao.getPageNode(namespace);
            pageNodeDto.setPath(uri);
            pageNodeDao.update(pageNodeDto);
            // TODO i18n
            FacesMessageUtil
                    .addInfoMessage("Updated the default page for this portlet.");
        } catch (CMSException e) {
            if (log.isDebugEnabled()) {
                log.debug("Could not update an existing page node: "
                        + namespace + " path: " + uri
                        + " Inserting a new page node...");
            }
            PageNodeDto newPageNodeDto = (PageNodeDto) container
                    .getComponent(PageNodeDto.class);
            // PageNodeDto newPageNodeDto = new PageNodeDto();
            newPageNodeDto.setName(namespace);
            newPageNodeDto.setPath(uri);
            try {
                pageNodeDao.insert(newPageNodeDto);
                // TODO i18n
                FacesMessageUtil
                        .addInfoMessage("Added the default page for this portlet.");
            } catch (CMSException e1) {
                log.error("Could not insert a new page node.", e1);
            }
        }
        return null;
    }

    /**
     * @return Returns the htmlFile.
     */
    public boolean isHtmlFile() {
        return htmlFile;
    }

    /**
     * @param htmlFile
     *            The htmlFile to set.
     */
    public void setHtmlFile(boolean htmlFile) {
        this.htmlFile = htmlFile;
    }

    /**
     * @return Returns the textFile.
     */
    public boolean isTextFile() {
        return textFile;
    }

    /**
     * @param textFile
     *            The textFile to set.
     */
    public void setTextFile(boolean textFile) {
        this.textFile = textFile;
    }

    /**
     * @return Returns the name.
     */
    public String getName() {
        return name;
    }

    /**
     * @param name
     *            The name to set.
     */
    public void setName(String name) {
        this.name = name;
    }

    /**
     * @return Returns the path.
     */
    public String getPath() {
        return path;
    }

    /**
     * @param path
     *            The path to set.
     */
    public void setPath(String path) {
        this.path = path;
    }

    /**
     * @return Returns the container.
     */
    public S2Container getContainer() {
        return container;
    }

    /**
     * @param container
     *            The container to set.
     */
    public void setContainer(S2Container container) {
        this.container = container;
    }

}
