/*
 * Copyright 2005-2006 Portal Application Laboratory project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, 
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package jp.sf.pal.tomahawk.filter;

import java.io.BufferedWriter;
import java.io.IOException;
import java.io.Reader;
import java.io.Writer;

import javax.portlet.ActionRequest;
import javax.portlet.ActionResponse;
import javax.portlet.PortletConfig;
import javax.portlet.PortletException;
import javax.portlet.RenderRequest;
import javax.portlet.RenderResponse;

import jp.sf.pal.facesresponse.io.BufferedResponseStream;
import jp.sf.pal.facesresponse.io.BufferedResponseStreamFactory;
import jp.sf.pal.tomahawk.handler.ResourceHandlerImpl;
import jp.sf.pal.tomahawk.multipart.MultipartPortletRequestWrapper;
import jp.sf.pal.tomahawk.wrapper.HttpServletRequestWrapper;
import jp.sf.pal.tomahawk.wrapper.HttpServletResponseWrapper;

import org.apache.commons.fileupload.portlet.PortletFileUpload;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.myfaces.renderkit.html.util.AddResource;
import org.apache.myfaces.renderkit.html.util.AddResourceFactory;
import org.apache.myfaces.webapp.filter.ExtensionsFilter;
import org.apache.myfaces.webapp.filter.ExtensionsResponseWrapper;
import org.apache.portals.bridges.portletfilter.PortletFilter;
import org.apache.portals.bridges.portletfilter.PortletFilterChain;
import org.apache.portals.bridges.portletfilter.PortletFilterConfig;

/**
 * This portlet filter supports Tomahawk's extended components, such as
 * inputHtml and fileUpload.
 * 
 * @author <a href="mailto:shinsuke@yahoo.co.jp">Shinsuke Sugaya</a>
 */
public class ExtensionsPortletFilter implements PortletFilter {
    private static final Log log = LogFactory
            .getLog(ExtensionsPortletFilter.class);

    private static final int BLOCK_SIZE = 4096;

    private static final String UPLOAD_REPOSITORY_PATH = "uploadRepositoryPath";

    private static final String UPLOAD_THRESHOLD_SIZE = "uploadThresholdSize";

    private static final String UPLOAD_MAX_FILE_SIZE = "uploadMaxFileSize";

    private int uploadMaxFileSize = 100 * 1024 * 1024; // 10 MB

    private int uploadThresholdSize = 1 * 1024 * 1024; // 1 MB

    private String uploadRepositoryPath = null; // standard temp directory

    private PortletConfig portletConfig;

    /**
     * Called by init method of MyFacesFilterPortlet to initialize this portlet
     * filter.
     * 
     * @param filterConfig
     * @throws PortletException
     */
    public void init(PortletFilterConfig filterConfig) throws PortletException {
        if (log.isDebugEnabled())
            log.debug("Initializing ExtensionsPortletFilter.");

        setPortletConfig(filterConfig.getPortletConfig());

        // for inputFileUpload
        String param = filterConfig.getInitParameter(UPLOAD_MAX_FILE_SIZE);

        uploadMaxFileSize = resolveSize(param, uploadMaxFileSize);

        param = filterConfig.getInitParameter(UPLOAD_THRESHOLD_SIZE);

        uploadThresholdSize = resolveSize(param, uploadThresholdSize);

        uploadRepositoryPath = filterConfig
                .getInitParameter(UPLOAD_REPOSITORY_PATH);

        if (log.isDebugEnabled()) {
            log.debug("uploadMaxFileSize=" + uploadMaxFileSize);
            log.debug("uploadThresholdSize=" + uploadThresholdSize);
            log.debug("uploadRepositoryPath=" + uploadRepositoryPath);
            // log.debug("mimeType=" + mimeType);
        }
    }

    /**
     * Called by render method of MyFacesFilterPortlet to put tags, such as
     * &lt;style&gt;, into &lt;head&gt;.
     * 
     * @param request
     * @param response
     * @param chain
     *            PortletFilterChain instance
     * @throws PortletException
     */
    public void renderFilter(RenderRequest request, RenderResponse response,
            PortletFilterChain chain) throws PortletException, IOException {
        if (log.isDebugEnabled()) {
            log.debug("called renderFilter.");
            log.debug("RenderRequest=" + request.getClass().getName());
            log.debug("RenderResponse=" + response.getClass().getName());
        }
        if (request.getAttribute(ExtensionsFilter.DOFILTER_CALLED) != null) {
            chain.renderFilter(request, response);
            return;
        }

        request.setAttribute(ExtensionsFilter.DOFILTER_CALLED, "true");

        HttpServletRequestWrapper extendedRequest = new HttpServletRequestWrapper(
                request, getPortletConfig().getPortletContext());

        // Serve resources
        AddResource addResource = null;
        try {
            addResource = AddResourceFactory.getInstance(extendedRequest);
        } catch (Throwable th) {
            log.error("Exception while retrieving addResource", th);
            throw new PortletException(th);
        }

        try {
            addResource.responseStarted();

            if (addResource.requiresBuffer()) {
                // String outputEncoding = response.getCharacterEncoding();
                // ByteArrayOutputStream outputStream = new
                // ByteArrayOutputStream();
                // Writer streamWriter = new BufferedWriter(
                // new OutputStreamWriter(outputStream,
                // java.nio.charset.Charset
                // .forName(outputEncoding)));
                // request.setAttribute(ExtensionsRenderKit.WRITER,
                // streamWriter);

                HttpServletResponseWrapper servletResponse = new HttpServletResponseWrapper(
                        response);
                ExtensionsResponseWrapper extendedResponse = new ExtensionsResponseWrapper(
                        servletResponse);

                // call next rednerFilter
                chain.renderFilter(request, response);

                // flush stream
                // streamWriter.flush();
                BufferedResponseStream bufferedResponseStream = BufferedResponseStreamFactory
                        .getBufferedResponseStream(request);
                if (bufferedResponseStream != null) {
                    synchronized (bufferedResponseStream) {
                        if (log.isDebugEnabled()) {
                            log
                                    .debug("renderFilter(RenderRequest, RenderResponse, PortletFilterChain) - bufferedResponseStream="
                                            + bufferedResponseStream);
                        }

                        // just in case, commit it.
                        bufferedResponseStream.commit();

                        Reader reader = bufferedResponseStream.getReader();
                        BufferedWriter writer = new BufferedWriter(
                                bufferedResponseStream.getWriter());

                        addResource.parseResponse(extendedRequest,
                                ResourceHandlerImpl.HTML_CONTENTS,
                                servletResponse);
                        addResource.writeMyFacesJavascriptBeforeBodyEnd(
                                extendedRequest, servletResponse);
                        // Some headerInfo has to be added
                        addResource.writeWithFullHeader(extendedRequest,
                                extendedResponse);
                        addResource.writeResponse(extendedRequest,
                                extendedResponse);
                        extendedResponse.finishResponse();

                        if (log.isDebugEnabled()) {
                            log
                                    .debug("renderFilter extendedResponse.toString()="
                                            + extendedResponse.toString());
                        }

                        ResourceHandlerImpl resourceHandler = new ResourceHandlerImpl();

                        resourceHandler.parse(extendedResponse.toString());

                        resourceHandler.renderBeforeContents(response, writer);
                        resourceHandler.addResourcesToHead(request, response,
                                writer);
                        resourceHandler.renderResourcesBeforeContents(request,
                                response, writer);

                        // Write Portlet content
                        drain(reader, writer);

                        resourceHandler.renderResourcesAfterContents(request,
                                response, writer);

                        writer.flush();

                        // commit
                        bufferedResponseStream.commit();
                    }
                }
            } else {
                chain.renderFilter(request, response);
            }
        } finally {
            addResource.responseFinished();
        }
    }

    /**
     * Called by render method of MyFacesFilterPortlet to wrap the request when
     * it has a multipart content.
     * 
     * @param request
     * @param response
     * @param chain
     *            PortletFilterChain instance
     * @throws PortletException
     */
    public void processActionFilter(ActionRequest request,
            ActionResponse response, PortletFilterChain chain)
            throws PortletException, IOException {
        if (log.isDebugEnabled())
            log.debug("called processActionFilter.");

        // Check multipart/form-data
        if (PortletFileUpload.isMultipartContent(request)) {
            if (log.isDebugEnabled())
                log.debug("ActionRequest is multipart content.");
            request = new MultipartPortletRequestWrapper(request,
                    uploadMaxFileSize, uploadThresholdSize,
                    uploadRepositoryPath);
        }

        // call next processActionFilter
        chain.processActionFilter(request, response);
    }

    /**
     * Called by destroy method of MyFacesFilterPortlet to destroy this portlet
     * filter.
     */
    public void destroy() {
    }

    private int resolveSize(String param, int defaultValue) {
        int numberParam = defaultValue;

        if (param != null) {
            param = param.toLowerCase();
            int factor = 1;
            String number = param;

            if (param.endsWith("g")) {
                factor = 1024 * 1024 * 1024;
                number = param.substring(0, param.length() - 1);
            } else if (param.endsWith("m")) {
                factor = 1024 * 1024;
                number = param.substring(0, param.length() - 1);
            } else if (param.endsWith("k")) {
                factor = 1024;
                number = param.substring(0, param.length() - 1);
            }

            numberParam = Integer.parseInt(number) * factor;
        }
        return numberParam;
    }

    /**
     * @return Returns the portletConfig.
     */
    public PortletConfig getPortletConfig() {
        return portletConfig;
    }

    /**
     * @param portletConfig
     *            The portletConfig to set.
     */
    public void setPortletConfig(PortletConfig portletConfig) {
        this.portletConfig = portletConfig;
    }

    public boolean isValidContentType(String contentType) {
        return contentType.startsWith("text/html")
                || contentType.startsWith("text/xml")
                || contentType.startsWith("application/xhtml+xml")
                || contentType.startsWith("application/xml");
    }

    private void drain(Reader r, Writer w) throws IOException {
        char[] bytes = new char[BLOCK_SIZE];
        try {
            int length = r.read(bytes);
            while (length != -1) {
                if (length != 0) {
                    w.write(bytes, 0, length);
                }
                length = r.read(bytes);
            }
        } finally {
            bytes = null;
        }

    }

}
