/*
 * Copyright 2005-2006 The Portal Application Laboratory Team.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, 
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package jp.sf.pal.cms.web.editor;

import java.io.IOException;

import javax.faces.context.FacesContext;

import jp.sf.pal.cms.CMSConstants;
import jp.sf.pal.cms.CMSException;
import jp.sf.pal.cms.dao.FileNodeDao;
import jp.sf.pal.cms.dto.FileNodeDto;
import jp.sf.pal.cms.util.CMSUtil;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.myfaces.custom.fileupload.UploadedFile;
import org.seasar.framework.container.S2Container;

import com.marevol.utils.faces.application.FacesMessageUtil;

/**
 * @author shinsuke
 * 
 */
public class UploadFilePage {
    /**
     * Logger for this class
     */
    private static final Log log = LogFactory.getLog(UploadFilePage.class);

    private String path;

    private String name;

    private String title;

    private String description;

    private String mimeType;

    private String encoding;

    private UploadedFile uploadedFile;

    private S2Container container;

    public UploadFilePage(S2Container container) {
        Object obj = FacesContext.getCurrentInstance().getExternalContext()
                .getSessionMap().get(CMSConstants.PATH);
        if (obj != null) {
            setPath((String) obj);
        } else {
            setPath(CMSConstants.ROOT_FOLDER);
        }

        // add new object
        String name = (String) FacesContext.getCurrentInstance()
                .getExternalContext().getSessionMap().get(
                        CMSConstants.NEW_OBJECT_NAME);
        if (log.isDebugEnabled()) {
            log.debug("UploadFilePage() - path=" + path);
            log.debug("UploadFilePage() - name=" + name);
        }
        if (name != null) {
            setName(name);
        } else {
            // update
            name = (String) FacesContext.getCurrentInstance()
                    .getExternalContext().getSessionMap().get(
                            CMSConstants.SELECTED_OBJECT_NAME);
            if (log.isDebugEnabled()) {
                log.debug("UploadFilePage() - name=" + name);
                log
                        .debug("UploadFilePage() - CMSUtil.normalizedPath(getPath() + CMSConstants.PATH_SEPARATOR + name)="
                                + CMSUtil.normalizedPath(getPath()
                                        + CMSConstants.PATH_SEPARATOR) + name);
            }
            // S2Container container =
            // SingletonS2ContainerFactory.getContainer();
            FileNodeDao fileNodeDao = (FileNodeDao) container
                    .getComponent(FileNodeDao.class);

            try {
                FileNodeDto fileNode = fileNodeDao.getFileNode(CMSUtil
                        .getScopeFromRequest(),
                        CMSUtil.normalizedPath(getPath()
                                + CMSConstants.PATH_SEPARATOR)
                                + name);
                if (fileNode != null) {
                    setName(fileNode.getName());
                    setTitle(fileNode.getTitle());
                    setDescription(fileNode.getDescription());
                    setMimeType(fileNode.getMimeType());
                    setEncoding(fileNode.getEncoding());

                } else {
                    // TODO i18n
                    FacesMessageUtil
                            .addErrorMessage("The file object is not specified.");
                }
            } catch (CMSException e) {
                // TODO i18n
                FacesMessageUtil.addErrorMessage("Could not load file: "
                        + getPath() + CMSConstants.PATH_SEPARATOR + name, e);
            }
        }
        // Set MIME type
        if (getMimeType() == null || getMimeType().equals("")) {
            setMimeType(CMSUtil.getMimeType(getName()));
        }
    }

    public String save() {
        String fileName = (String) FacesContext.getCurrentInstance()
                .getExternalContext().getSessionMap().get(
                        CMSConstants.NEW_OBJECT_NAME);
        if (log.isDebugEnabled()) {
            log.debug("save() - name=" + fileName);
        }
        if (fileName != null) {
            // create a new object
            // S2Container container =
            // SingletonS2ContainerFactory.getContainer();
            FileNodeDao fileNodeDao = (FileNodeDao) container
                    .getComponent(FileNodeDao.class);
            FileNodeDto fileNode = (FileNodeDto) container
                    .getComponent(FileNodeDto.class);
            // FileNodeDto fileNode = new FileNodeDto();
            fileNode.setScope(CMSUtil.getScopeFromRequest());
            fileNode.setParentPath(getPath());
            fileNode.setName(fileName);
            fileNode.setTitle(getTitle());
            fileNode.setDescription(getDescription());
            fileNode.setMimeType(getMimeType());
            fileNode.setEncoding(getEncoding());
            if (getUploadedFile() != null) {
                try {
                    fileNode.setInputStream(getUploadedFile().getInputStream());
                } catch (IOException e) {
                    log
                            .warn(
                                    "Could not load an input stream from the uploaded file.",
                                    e);
                }
            } else {
                fileNode.setInputStream(null);
            }
            try {
                fileNodeDao.insert(fileNode);
                // TODO i18n
                FacesMessageUtil.addInfoMessage("Added a new file object.");
            } catch (CMSException e) {
                log.warn("Could not insert a new file object: path="
                        + fileNode.getParentPath() + ", name="
                        + fileNode.getName(), e);
                // TODO i18n
                FacesMessageUtil.addErrorMessage(
                        "Could not insert a new file object.", e);
            }
            // clear session
            CMSUtil.clearSession();
            return CMSConstants.DISPLAY_FOLDER_LIST_PAGE;
        } else {
            // update
            fileName = (String) FacesContext.getCurrentInstance()
                    .getExternalContext().getSessionMap().get(
                            CMSConstants.SELECTED_OBJECT_NAME);
            if (fileName != null) {
                // S2Container container = SingletonS2ContainerFactory
                // .getContainer();
                FileNodeDao fileNodeDao = (FileNodeDao) container
                        .getComponent(FileNodeDao.class);
                try {
                    FileNodeDto fileNode = fileNodeDao.getFileNode(CMSUtil
                            .getScopeFromRequest(), CMSUtil
                            .normalizedPath(getPath()
                                    + CMSConstants.PATH_SEPARATOR)
                            + fileName);
                    if (log.isDebugEnabled()) {
                        log.debug("save() - fileNode=" + fileNode);
                    }
                    fileNode.setTitle(getTitle());
                    fileNode.setDescription(getDescription());
                    fileNode.setMimeType(getMimeType());
                    fileNode.setEncoding(getEncoding());
                    if (getUploadedFile() != null) {
                        try {
                            fileNode.setInputStream(getUploadedFile()
                                    .getInputStream());
                        } catch (IOException e) {
                            log
                                    .warn(
                                            "Could not load an input stream from the uploaded file.",
                                            e);
                        }
                    } else {
                        fileNode.setInputStream(null);
                    }
                    fileNodeDao.updateContent(fileNode);
                    // TODO i18n
                    FacesMessageUtil.addInfoMessage("Updated the file object.");
                } catch (CMSException e) {
                    log.warn("Could not update the file object: path="
                            + getPath() + ", name=" + getName(), e);
                    // TODO i18n
                    FacesMessageUtil.addErrorMessage(
                            "Could not update the file object.", e);
                }
                return null;
            } else {
                // TODO i18n
                FacesMessageUtil
                        .addErrorMessage("The file object is not specified.");
            }
        }
        return null;
    }

    public String backToList() {
        if (log.isDebugEnabled()) {
            log.debug("backToList()");
        }

        CMSUtil.clearSession();
        return CMSConstants.DISPLAY_FOLDER_LIST_PAGE;
    }

    /**
     * @return Returns the path.
     */
    public String getPath() {
        return path;
    }

    /**
     * @param path
     *            The path to set.
     */
    public void setPath(String path) {
        this.path = path;
    }

    /**
     * @return Returns the description.
     */
    public String getDescription() {
        return description;
    }

    /**
     * @param description
     *            The description to set.
     */
    public void setDescription(String description) {
        this.description = description;
    }

    /**
     * @return Returns the encoding.
     */
    public String getEncoding() {
        return encoding;
    }

    /**
     * @param encoding
     *            The encoding to set.
     */
    public void setEncoding(String encoding) {
        this.encoding = encoding;
    }

    /**
     * @return Returns the mimeType.
     */
    public String getMimeType() {
        return mimeType;
    }

    /**
     * @param mimeType
     *            The mimeType to set.
     */
    public void setMimeType(String mimeType) {
        this.mimeType = mimeType;
    }

    /**
     * @return Returns the name.
     */
    public String getName() {
        return name;
    }

    /**
     * @param name
     *            The name to set.
     */
    public void setName(String name) {
        this.name = name;
    }

    /**
     * @return Returns the title.
     */
    public String getTitle() {
        return title;
    }

    /**
     * @param title
     *            The title to set.
     */
    public void setTitle(String title) {
        this.title = title;
    }

    /**
     * @return Returns the uploadedFile.
     */
    public UploadedFile getUploadedFile() {
        return uploadedFile;
    }

    /**
     * @param uploadedFile
     *            The uploadedFile to set.
     */
    public void setUploadedFile(UploadedFile uploadedFile) {
        this.uploadedFile = uploadedFile;
    }

    /**
     * @return Returns the container.
     */
    public S2Container getContainer() {
        return container;
    }

    /**
     * @param container
     *            The container to set.
     */
    public void setContainer(S2Container container) {
        this.container = container;
    }
}
