/*
 * Copyright 2005-2006 The Portal Application Laboratory Team.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package jp.sf.pal.vfs.helper.impl;

import java.text.DateFormat;
import java.util.Date;
import java.util.Locale;

import javax.faces.context.ExternalContext;
import javax.faces.context.FacesContext;
import javax.portlet.PortletRequest;

import jp.sf.pal.vfs.VFSConstants;
import jp.sf.pal.vfs.helper.FileParameterHelper;
import jp.sf.pal.vfs.util.VFSUtil;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.commons.vfs.FileObject;
import org.apache.commons.vfs.FileSystemException;
import org.apache.commons.vfs.FileType;
import org.seasar.framework.container.factory.SingletonS2ContainerFactory;

/**
 * @author <a href="mailto:shinsuke@users.sourceforge.jp">Shinsuke Sugaya</a>
 */
public class FileParameterHelperImpl implements FileParameterHelper {
    /**
     * Logger for this class
     */
    private static final Log log = LogFactory
            .getLog(FileParameterHelperImpl.class);

    private static final long THOUSAND = 1000;

    private FileObject fileObject;

    private long time = 0;

    public FileParameterHelperImpl(FileObject fObject) {
        setFileObject(fObject);
    }

    /*
     * (non-Javadoc)
     * 
     * @see jp.sf.pal.vfs.helper.impl.FileParameterHelper#getFileObject()
     */
    public FileObject getFileObject() {
        return fileObject;
    }

    /*
     * (non-Javadoc)
     * 
     * @see jp.sf.pal.vfs.helper.impl.FileParameterHelper#setFileObject(org.apache.commons.vfs.FileObject)
     */
    public void setFileObject(FileObject object) {
        fileObject = object;
    }

    /*
     * (non-Javadoc)
     * 
     * @see jp.sf.pal.vfs.helper.impl.FileParameterHelper#getBaseName()
     */
    public String getBaseName() {
        return getFileObject().getName().getBaseName();
    }

    /*
     * (non-Javadoc)
     * 
     * @see jp.sf.pal.vfs.helper.impl.FileParameterHelper#getURI()
     */
    public String getURI() {
        return getFileObject().getName().getURI();
    }

    /*
     * (non-Javadoc)
     * 
     * @see jp.sf.pal.vfs.helper.impl.FileParameterHelper#getParentURI()
     */
    public String getParentURI() {
        try {
            return getFileObject().getParent().getName().getURI();
        } catch (FileSystemException e) {
            return "";
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see jp.sf.pal.vfs.helper.impl.FileParameterHelper#getSize()
     */
    public long getSize() {
        try {
            return getFileObject().getContent().getSize();
        } catch (FileSystemException e) {
            return -1;
        } catch (Exception e) {
            return -1;
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see jp.sf.pal.vfs.helper.impl.FileParameterHelper#isFolder()
     */
    public boolean isFolder() {
        try {
            if (fileObject.getType() == FileType.FOLDER) {
                return true;
            }
        } catch (FileSystemException e) {
            // TODO Auto-generated catch block
            log.error("isFolder()", e);
        }
        return false;
    }

    /*
     * (non-Javadoc)
     * 
     * @see jp.sf.pal.vfs.helper.impl.FileParameterHelper#isFile()
     */
    public boolean isFile() {
        try {
            if (fileObject.getType() == FileType.FILE) {
                return true;
            }
        } catch (FileSystemException e) {
            // TODO Auto-generated catch block
            log.error("isFile()", e);
        }
        return false;
    }

    /*
     * (non-Javadoc)
     * 
     * @see jp.sf.pal.vfs.helper.impl.FileParameterHelper#getFormattedSize()
     */
    public String getFormattedSize() {
        long size = getSize();
        if (log.isDebugEnabled()) {
            log.debug("getFormattedSize() - size=" + size);
        }

        if (size == -1) {
            // TODO I18N
            return VFSConstants.NULL_VALUE;
        } else if (size < THOUSAND) {
            // TODO I18N
            return new String(size + "byte");
        } else if (size < THOUSAND * THOUSAND) {
            long value = (long) (size / (THOUSAND));
            // TODO I18N
            return new String(value + "K byte");
        } else if (size < THOUSAND * THOUSAND * THOUSAND) {
            long value = (long) (size / (THOUSAND * THOUSAND));
            // TODO I18N
            return new String(value + "M byte");
        } else if (size < THOUSAND * THOUSAND * THOUSAND * THOUSAND) {
            long value = (long) (size / (THOUSAND * THOUSAND * THOUSAND));
            // TODO I18N
            return new String(value + "G byte");
        }
        // TODO I18N
        return new String(size + "byte");
    }

    /*
     * (non-Javadoc)
     * 
     * @see jp.sf.pal.vfs.helper.impl.FileParameterHelper#getContentType()
     */
    public String getContentType() {
        FacesContext facesContext = FacesContext.getCurrentInstance();
        if (isFolder()) {
            return VFSConstants.NULL_VALUE;
        }
        String mimeType = null;
        if (facesContext != null) {
            mimeType = VFSUtil.getMimeType(facesContext.getExternalContext()
                    .getContext(), getBaseName());
        } else {
            mimeType = VFSUtil.getMimeType(SingletonS2ContainerFactory
                    .getContainer().getExternalContext().getApplication(),
                    getBaseName());
        }
        if (mimeType == null) {
            return VFSConstants.OCTET_STREAM;
        }
        return mimeType;
    }

    /*
     * (non-Javadoc)
     * 
     * @see jp.sf.pal.vfs.helper.impl.FileParameterHelper#getFormattedLastModifiedTime()
     */
    public String getFormattedLastModifiedTime() {
        try {
            if (fileObject.getContent() == null || fileObject.getType() == null) {
                if (log.isDebugEnabled()) {
                    log
                            .debug("getFormattedLastModifiedTime() - fileObject.getContent() = null or fileObject.getType() == null");
                }
                return VFSConstants.NULL_VALUE;
            }
            long time = fileObject.getContent().getLastModifiedTime();

            FacesContext context = FacesContext.getCurrentInstance();
            ExternalContext exContext = context.getExternalContext();
            PortletRequest request = (PortletRequest) exContext.getRequest();
            Locale locale = request.getLocale();
            DateFormat df = DateFormat.getDateTimeInstance(DateFormat.DEFAULT,
                    DateFormat.DEFAULT, locale);
            return df.format(new Date(time));
        } catch (FileSystemException e) {
            // TODO Auto-generated catch block
            log.error("getFormattedLastModifiedTime()", e);
        } catch (Exception e) {
            // TODO Auto-generated catch block
            log.error("getFormattedLastModifiedTime()", e);
        }

        return VFSConstants.NULL_VALUE;
    }

    /*
     * (non-Javadoc)
     * 
     * @see jp.sf.pal.vfs.helper.impl.FileParameterHelper#getTime()
     */
    public long getTime() {
        return time;
    }

    /*
     * (non-Javadoc)
     * 
     * @see jp.sf.pal.vfs.helper.impl.FileParameterHelper#setTime(long)
     */
    public void setTime(long time) {
        this.time = time;
    }
}
