/*
 * Copyright 2005-2006 The Portal Application Laboratory Team.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, 
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package jp.sf.pal.vfs.web.navigator;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Map;

import javax.faces.context.ExternalContext;
import javax.faces.context.FacesContext;
import javax.portlet.PortletSession;

import jp.sf.pal.vfs.VFSConstants;
import jp.sf.pal.vfs.helper.FileParameterHelper;
import jp.sf.pal.vfs.util.FileParameterHelperComparator;
import jp.sf.pal.vfs.util.VFSUtil;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.commons.vfs.FileObject;
import org.apache.commons.vfs.FileSystemException;
import org.apache.commons.vfs.FileSystemManager;
import org.apache.commons.vfs.FileType;
import org.apache.commons.vfs.Selectors;
import org.seasar.framework.container.S2Container;
import org.seasar.portlet.util.PortletRequestUtil;

import com.marevol.utils.faces.application.FacesMessageUtil;

public class DisplayFolderListPage extends DefaultPage {
    /**
     * Logger for this class
     */
    private static final Log log = LogFactory
            .getLog(DisplayFolderListPage.class);

    public DisplayFolderListPage(S2Container container) {
        super(container);
    }

    private FileParameterHelper createFileParameterHelper(FileObject fObject) {
        FileParameterHelper fileParam = (FileParameterHelper) getContainer()
                .getComponent(FileParameterHelper.class);
        fileParam.setFileObject(fObject);
        return fileParam;
    }

    public List getObjects() {
        if (log.isDebugEnabled()) {
            log.debug("getObjects()");
        }

        String uri = getUri();
        // Get File System
        ArrayList objects = new ArrayList();
        if (log.isDebugEnabled()) {
            log.debug("getObjects() - uri=" + uri);
        }
        if (VFSUtil.checkPath(uri)) {
            try {
                FileSystemManager fsManager = VFSUtil.getFileSystemManager();
                FileObject currentDirectory = fsManager.resolveFile(uri);

                FileObject[] children = currentDirectory.getChildren();
                ArrayList folderList = new ArrayList();
                ArrayList fileList = new ArrayList();
                for (int i = 0; i < children.length; i++) {
                    try {
                        if (children[i].getType() == FileType.FOLDER) {
                            folderList
                                    .add(createFileParameterHelper(children[i]));
                        } else {
                            fileList
                                    .add(createFileParameterHelper(children[i]));
                        }
                    } catch (FileSystemException e) {
                        fileList.add(createFileParameterHelper(children[i]));
                    }
                }
                Object[] folders = folderList.toArray();
                // TODO Comparator
                Arrays.sort(folders, new FileParameterHelperComparator());
                for (int i = 0; i < folders.length; i++) {
                    objects.add(folders[i]);
                }
                Object[] files = fileList.toArray();
                // TODO Comparator
                Arrays.sort(files, new FileParameterHelperComparator());
                for (int i = 0; i < files.length; i++) {
                    objects.add(files[i]);
                }
            } catch (FileSystemException e) {
                // TODO i18n
                FacesMessageUtil.addErrorMessage(
                        "The file system error occurs.", e.toString());
                log.error("The file system error occurs.", e);
            }
        } else {
            // TODO i18n
            FacesMessageUtil.addErrorMessage("access denied.");
        }
        return (List) objects;

    }

    public String select() {
        if (log.isDebugEnabled()) {
            log.debug("select()");
        }
        if (!VFSUtil.checkPath(getUri())) {
            // TODO i18n
            FacesMessageUtil.addErrorMessage("access denied.");
            setUri(VFSUtil.getDefaultUri());
            return VFSConstants.DISPLAY_FOLDER_LIST_PAGE;
        }

        FacesContext context = FacesContext.getCurrentInstance();
        ExternalContext exContext = context.getExternalContext();
        Map map = exContext.getRequestMap();
        FileParameterHelper fileParam = (FileParameterHelper) map.get("item");
        if (fileParam != null) {
            if (fileParam.isFolder()) {
                setUri(fileParam.getURI());
            } else if (fileParam.isFile()) {
                PortletRequestUtil.getPortletSession().setAttribute(
                        VFSConstants.DOWNLOAD_OBJECT, fileParam,
                        PortletSession.APPLICATION_SCOPE);
                // TODO I18N
                VFSUtil
                        .publishOutputHtml("<script language=\"javascript\">\n<!--\nwindow.open(\""
                                + exContext.getRequestContextPath()
                                + "/FileServlet\", \"_blank\");\n//--></script>");
                // VFSUtil
                // .publishOutputHtml("<script
                // language=\"javascript\">\n<!--\nlocation.href=\""
                // + exContext.getRequestContextPath()
                // + "/FileServlet\";\n//--></script>");
            } else {
                // TODO I18N
                FacesMessageUtil
                        .addWarnMessage("The selected object is not FOLDER or FILE.");
            }
        } else {
            // TODO I18N
            FacesMessageUtil
                    .addWarnMessage("The selected object is not found.");
        }
        return null;
    }

    public String delete() {
        if (log.isDebugEnabled()) {
            log.debug("delete()");
        }
        if (!VFSUtil.checkPath(getUri())) {
            // TODO i18n
            FacesMessageUtil.addErrorMessage("access denied.");
            setUri(VFSUtil.getDefaultUri());
            return VFSConstants.DISPLAY_FOLDER_LIST_PAGE;
        }

        FacesContext context = FacesContext.getCurrentInstance();
        ExternalContext exContext = context.getExternalContext();
        Map map = exContext.getRequestMap();
        FileParameterHelper fileParam = (FileParameterHelper) map.get("item");
        if (fileParam != null) {
            if (fileParam.isFolder()) {
                try {
                    FileSystemManager fsManager = VFSUtil
                            .getFileSystemManager();
                    FileObject fileObject = fsManager.resolveFile(fileParam
                            .getURI());

                    // TODO select FileSelector
                    int num = fileObject.delete(Selectors.SELECT_ALL);
                    if (num > 0) {
                        // TODO i18n
                        FacesMessageUtil.addInfoMessage("Deleted " + num
                                + " objects.");
                    } else {
                        // TODO i18n
                        FacesMessageUtil
                                .addErrorMessage("Failed delete action.");
                    }
                } catch (FileSystemException e) {
                    // TODO i18n
                    FacesMessageUtil.addErrorMessage("Failed delete action.");
                    log.error("Failed delete action.", e);
                }
            } else if (fileParam.isFile()) {
                try {
                    FileSystemManager fsManager = VFSUtil
                            .getFileSystemManager();
                    FileObject fileObject = fsManager.resolveFile(fileParam
                            .getURI());

                    if (fileObject.delete()) {
                        // TODO i18n
                        FacesMessageUtil.addInfoMessage("Deleted the file: "
                                + fileParam.getURI());
                    } else {
                        // TODO i18n
                        FacesMessageUtil
                                .addWarnMessage("Failed delete action.");
                    }
                } catch (FileSystemException e) {
                    // TODO i18n
                    FacesMessageUtil.addErrorMessage("Failed delete action.", e
                            .toString());
                    log.error("Failed delete action.", e);
                }
            } else {
                // TODO I18N
                FacesMessageUtil
                        .addWarnMessage("The selected object is not FOLDER or FILE.");
            }
        } else {
            // TODO I18N
            FacesMessageUtil
                    .addWarnMessage("The selected object is not found.");
        }
        return goDisplayFolderListPage();
    }

    public String go() {
        if (log.isDebugEnabled()) {
            log.debug("go()");
        }
        if (!VFSUtil.checkPath(getUri())) {
            // TODO i18n
            FacesMessageUtil.addErrorMessage("access denied.");
            setUri(VFSUtil.getDefaultUri());
            return VFSConstants.DISPLAY_FOLDER_LIST_PAGE;
        }

        // update uri
        return VFSConstants.DISPLAY_FOLDER_LIST_PAGE;
    }

    public String openParent() {
        if (log.isDebugEnabled()) {
            log.debug("openParent()");
        }
        if (!VFSUtil.checkPath(getUri())) {
            // TODO i18n
            FacesMessageUtil.addErrorMessage("access denied.");
            setUri(VFSUtil.getDefaultUri());
            return VFSConstants.DISPLAY_FOLDER_LIST_PAGE;
        }

        try {
            FileSystemManager fsManager = VFSUtil.getFileSystemManager();
            FileObject currentDirectory = fsManager.resolveFile(getUri());
            FileObject parent = currentDirectory.getParent();
            if (parent != null) {
                setUri(parent.getName().getURI());
                if (!VFSUtil.checkPath(getUri())) {
                    // TODO i18n
                    FacesMessageUtil.addErrorMessage("access denied.");
                    setUri(VFSUtil.getDefaultUri());
                    return VFSConstants.DISPLAY_FOLDER_LIST_PAGE;
                }
            } else {
                // TODO i18n
                FacesMessageUtil
                        .addWarnMessage("The parent folder does not exist.");
            }
        } catch (FileSystemException e) {
            // TODO i18n
            FacesMessageUtil.addErrorMessage(
                    "Failed to open the parent directory.", e.toString());
            log.error("Failed to open the parent directory.", e);
        }

        return VFSConstants.DISPLAY_FOLDER_LIST_PAGE;
    }

    /**
     * @return the outputHtml
     */
    public String getOutputHtml() {
        return VFSUtil.consumeOutputHtml();
    }
}
