/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package jp.sf.pal.admin.deployer;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.Calendar;

import jp.sf.pal.admin.entity.PortletApplication;
import jp.sf.pal.admin.logic.PortletDeploymentLogic;
import jp.sf.pal.admin.util.PALAdminUtil;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.jetspeed.deployment.DeploymentException;
import org.apache.jetspeed.deployment.DeploymentManager;

public class PortletDeployer extends Thread
{
    public static final String FILEPATH_PROTOCOL_HTTPS = "https:";

    public static final String PATH_PROTOCOL_HTTP = "http:";

    public static final String FILEPATH_PROTOCOL_FILE = "file:";

    public static final String FILEPATH_PROTOCOL_TEMP = "temp:";

    /**
     * Logger for this class
     */
    private static final Log log = LogFactory
            .getLog(PortletDeploymentLogic.class);

    private DeploymentManager deploymentManager;

    private PortletApplication portletApplication;

    private DeploymentStatus deploymentStatus;

    public PortletDeployer(PortletApplication portletApplication)
    {
        this.portletApplication = portletApplication;
        deploymentStatus = new DeploymentStatus(portletApplication);
    }

    /* (non-Javadoc)
     * @see java.lang.Thread#run()
     */
    public void run()
    {
        deploymentStatus.setStatus(DeploymentStatus.DEPLOYING);
        try
        {
            deploymentStatus.setStartTime(Calendar.getInstance()
                    .getTimeInMillis());
            if (getDeploymentManager() != null)
            {
                String binaryUrl = portletApplication.getBinaryUrl();
                if (binaryUrl != null && !binaryUrl.equals(""))
                {
                    File targetFile = null;
                    try
                    {
                        File tempFile = null;
                        if (binaryUrl.startsWith(FILEPATH_PROTOCOL_TEMP))
                        {
                            tempFile=new File(binaryUrl.substring(FILEPATH_PROTOCOL_TEMP.length()));
                        }
                        else
                        {
                            tempFile = File.createTempFile("rpad_", "."
                                    + portletApplication.getPackaging());
                            FileOutputStream out = new FileOutputStream(
                                    tempFile);
                            PALAdminUtil.drain(getInputStream(binaryUrl), out);
                        }
                        try
                        {
                            targetFile = new File(tempFile.getParentFile(),
                                    portletApplication.getArtifactId() + "."
                                            + portletApplication.getPackaging());
                            tempFile.renameTo(targetFile);
                        }
                        catch (Exception e)
                        {
                            targetFile = tempFile;
                        }
                        if (getDeploymentManager().deploy(targetFile)
                                .getStatus() == org.apache.jetspeed.deployment.DeploymentStatus.STATUS_OKAY)
                        {
                            log.info(portletApplication.getName()
                                    + " was deployed.");
                        }
                        else
                        {
                            log.error("Could not deploy "
                                    + portletApplication.getName());
                        }
                    }
                    catch (FileNotFoundException e)
                    {
                        log.error(e);
                    }
                    catch (IOException e)
                    {
                        log.error(e);
                    }
                    catch (DeploymentException e)
                    {
                        log.error(e);
                    }
                    if (targetFile != null && targetFile.exists())
                    {
                        targetFile.delete();
                    }
                }
                else
                {
                    log.error("The target url is invalid. The path is "
                            + binaryUrl);
                }
            }
            else
            {
                log.error("Could not find the deployment manager.");
            }
        }
        catch (Exception e)
        {
            log.error("Unexpected exception.", e);
        }
        finally
        {
            deploymentStatus.setStatus(DeploymentStatus.DEPLOYED);
        }
    }

    protected InputStream getInputStream(String path)
    {
        if (path.startsWith(PATH_PROTOCOL_HTTP)
                || path.startsWith(FILEPATH_PROTOCOL_HTTPS))
        {
            try
            {
                URL url = new URL(path);
                return url.openStream();
            }
            catch (MalformedURLException e)
            {
                log.error("Wrong url: " + path, e);
            }
            catch (IOException e)
            {
                log.error("Could not load " + path, e);
            }
        }
        else if (path.startsWith(FILEPATH_PROTOCOL_FILE))
        {
            try
            {
                return new FileInputStream(new File(path.substring(5)));
            }
            catch (FileNotFoundException e)
            {
                log.error("Could not load " + path, e);
            }
        }
        return null;
    }

    /**
     * @return the portletApplication
     */
    public PortletApplication getPortletApplication()
    {
        return portletApplication;
    }

    /**
     * @param portletApplication the portletApplication to set
     */
    public void setPortletApplication(PortletApplication portletApplication)
    {
        this.portletApplication = portletApplication;
    }

    /**
     * @return the deploymentManager
     */
    public DeploymentManager getDeploymentManager()
    {
        return deploymentManager;
    }

    /**
     * @param deploymentManager the deploymentManager to set
     */
    public void setDeploymentManager(DeploymentManager deploymentManager)
    {
        this.deploymentManager = deploymentManager;
    }

    /**
     * @return the deploymentStatus
     */
    public DeploymentStatus getDeploymentStatus()
    {
        return deploymentStatus;
    }

    /**
     * @param deploymentStatus the deploymentStatus to set
     */
    public void setDeploymentStatus(DeploymentStatus deploymentStatus)
    {
        this.deploymentStatus = deploymentStatus;
    }

}
