/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package jp.sf.pal.admin.logic;

import java.io.Serializable;
import java.security.Principal;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Iterator;
import java.util.List;
import java.util.Set;

import javax.security.auth.Subject;

import jp.sf.pal.admin.PALAdminConstants;
import jp.sf.pal.admin.PALAdminException;
import jp.sf.pal.admin.entity.Group;
import jp.sf.pal.admin.entity.PrincipalRule;
import jp.sf.pal.admin.entity.Role;
import jp.sf.pal.admin.entity.User;
import jp.sf.pal.admin.pager.GroupPager;
import jp.sf.pal.admin.pager.RolePager;
import jp.sf.pal.admin.pager.UserPager;
import jp.sf.pal.admin.util.PortalComponentUtil;
import jp.sf.pal.admin.util.PALAdminUtil;

import org.apache.jetspeed.om.folder.Folder;
import org.apache.jetspeed.page.PageManager;
import org.apache.jetspeed.profiler.Profiler;
import org.apache.jetspeed.security.AuthenticationProviderProxy;
import org.apache.jetspeed.security.GroupManager;
import org.apache.jetspeed.security.PasswordCredential;
import org.apache.jetspeed.security.RoleManager;
import org.apache.jetspeed.security.SecurityException;
import org.apache.jetspeed.security.SecurityProvider;
import org.apache.jetspeed.security.UserManager;
import org.apache.jetspeed.security.UserPrincipal;
import org.apache.jetspeed.security.om.InternalCredential;
import org.apache.jetspeed.security.spi.GroupSecurityHandler;
import org.apache.jetspeed.security.spi.RoleSecurityHandler;
import org.apache.log4j.Logger;

public class SecurityProviderLogic implements Serializable
{
    /**
     * 
     */
    private static final long serialVersionUID = 2380951929327928610L;

    /**
     * Logger for this class
     */
    private static final Logger logger = Logger
            .getLogger(SecurityProviderLogic.class);

    private transient SecurityProvider securityProvider = null;

    private transient AuthenticationProviderProxy authenticationProviderProxy;

    //    private UserSecurityHandler userSecurityHandler = null;

    private transient UserManager userManager = null;

    private transient RoleSecurityHandler roleSecurityHandler = null;

    private transient RoleManager roleManager = null;

    private transient GroupSecurityHandler groupSecurityHandler = null;

    private transient GroupManager groupManager = null;

    private transient Profiler profiler = null;

    private transient PageManager pageManager = null;

    /**
     * @return the authenticationProviderProxy
     */
    public AuthenticationProviderProxy getAuthenticationProviderProxy()
    {
        if (authenticationProviderProxy == null)
        {
            authenticationProviderProxy = getSecurityProvider()
                    .getAuthenticationProviderProxy();
        }
        return authenticationProviderProxy;
    }

    /**
     * @param authenticationProviderProxy the authenticationProviderProxy to set
     */
    public void setAuthenticationProviderProxy(
            AuthenticationProviderProxy authenticationProviderProxy)
    {
        this.authenticationProviderProxy = authenticationProviderProxy;
    }

    /**
     * @return the roleSecurityHandler
     */
    public RoleSecurityHandler getRoleSecurityHandler()
    {
        if (roleSecurityHandler == null)
        {
            roleSecurityHandler = getSecurityProvider()
                    .getRoleSecurityHandler();
        }
        return roleSecurityHandler;
    }

    /**
     * @param roleSecurityHandler the roleSecurityHandler to set
     */
    public void setRoleSecurityHandler(RoleSecurityHandler roleSecurityHandler)
    {
        this.roleSecurityHandler = roleSecurityHandler;
    }

    /**
     * @return the groupSecurityHandler
     */
    public GroupSecurityHandler getGroupSecurityHandler()
    {
        if (groupSecurityHandler == null)
        {
            groupSecurityHandler = getSecurityProvider()
                    .getGroupSecurityHandler();
        }
        return groupSecurityHandler;
    }

    /**
     * @param groupSecurityHandler the groupSecurityHandler to set
     */
    public void setGroupSecurityHandler(
            GroupSecurityHandler groupSecurityHandler)
    {
        this.groupSecurityHandler = groupSecurityHandler;
    }

    /**
     * @return the userManager
     */
    public UserManager getUserManager()
    {
        if (userManager == null)
        {
            userManager = PortalComponentUtil.getUserManager();
        }
        return userManager;
    }

    /**
     * @param userManager the userManager to set
     */
    public void setUserManager(UserManager userManager)
    {
        this.userManager = userManager;
    }

    /**
     * @return the roleManager
     */
    public RoleManager getRoleManager()
    {
        if (roleManager == null)
        {
            roleManager = PortalComponentUtil.getRoleManager();
        }
        return roleManager;
    }

    /**
     * @param roleManager the roleManager to set
     */
    public void setRoleManager(RoleManager roleManager)
    {
        this.roleManager = roleManager;
    }

    /**
     * @return the groupManager
     */
    public GroupManager getGroupManager()
    {
        if (groupManager == null)
        {
            groupManager = PortalComponentUtil.getGroupManager();
        }
        return groupManager;
    }

    /**
     * @param groupManager the groupManager to set
     */
    public void setGroupManager(GroupManager groupManager)
    {
        this.groupManager = groupManager;
    }

    /**
     * @return the profiler
     */
    public Profiler getProfiler()
    {
        if (profiler == null)
        {
            profiler = PortalComponentUtil.getProfiler();
        }
        return profiler;
    }

    /**
     * @param profiler the profiler to set
     */
    public void setProfiler(Profiler profiler)
    {
        this.profiler = profiler;
    }

    /**
     * @return the pageManager
     */
    public PageManager getPageManager()
    {
        if (pageManager == null)
        {
            pageManager = PortalComponentUtil.getPageManager();
        }
        return pageManager;
    }

    /**
     * @param pageManager the pageManager to set
     */
    public void setPageManager(PageManager pageManager)
    {
        this.pageManager = pageManager;
    }

    /**
     * @return the securityProvider
     */
    public SecurityProvider getSecurityProvider()
    {
        if (securityProvider == null)
        {
            securityProvider = PortalComponentUtil.getSecurityProvider();
        }
        return securityProvider;
    }

    /**
     * @param securityProvider the securityProvider to set
     */
    public void setSecurityProvider(SecurityProvider securityProvider)
    {
        this.securityProvider = securityProvider;
    }

    //
    // User
    //

    public List<User> getUsers(UserPager pager) throws PALAdminException
    {
        //Debug
        if (PALAdminConstants.DEBUG)
        {
            List<User> list = new ArrayList<User>();
            User entity = new User();
            entity.setName("USER");
            list.add(entity);
            return list;
        }

        if (pager.getAuthenticationProviderName() == null)
        {
            pager
                    .setAuthenticationProviderName(getAuthenticationProviderProxy()
                            .getDefaultAuthenticationProvider());
        }

        StringBuffer fb = new StringBuffer("AdvanceFilter: ");
        fb.append("username=").append(pager.getFilter());

        List<User> userList = new ArrayList<User>();

        try
        {
            // set count
            pager.setCount(getAuthenticationProviderProxy().getUserCount(
                    fb.toString(), pager.getAuthenticationProviderName()));

            int startAtIndex = pager.getOffset() + 1;
            int endAtIndex = startAtIndex + pager.getLimit();
            fb.append(", startAtIndex=").append(startAtIndex).append(
                    ", endAtIndex=").append(endAtIndex);

            List<Principal> principals = getAuthenticationProviderProxy()
                    .getUserPrincipals(fb.toString(),
                            pager.getAuthenticationProviderName());

            int size = principals.size();
            if (logger.isDebugEnabled())
            {
                logger.debug("getUsers(UserPager) - pager.getFilter()="
                        + pager.getFilter());
                logger.debug("getUsers(UserPager) - size=" + size + ", filter="
                        + fb.toString());
            }

            for (int i = 0; i < size; i++)
            {
                User entity = new User();
                entity.setName(principals.get(i).getName());
                // TODO Others
                userList.add(entity);
            }
        }
        catch (SecurityException e)
        {
            logger.error("Could not get users.", e);
            throw new PALAdminException(e);
        }

        return userList;

    }

    public void insertUser(User user) throws PALAdminException
    {
        //Debug
        if (PALAdminConstants.DEBUG)
        {
            return;
        }

        if (PALAdminUtil.isEmpty(user.getName())
                || PALAdminUtil.isEmpty(user.getPassword()))
        {
            throw new PALAdminException(
                    "Invalid user name or password. username=" + user.getName()
                            + ", password=" + user.getPassword());
        }

        //TODO check duplicate name
        try
        {
            // Add user
            getUserManager().addUser(user.getName(), user.getPassword());

            // Set updateRequired
            PasswordCredential credential = getPasswordCredential(user
                    .getName());
            if (credential != null)
            {
                if (user.isUpdateRequired() != credential.isUpdateRequired())
                {
                    getUserManager().setPasswordUpdateRequired(user.getName(),
                            user.isUpdateRequired());
                }
            }

            // Set role
            for (Iterator<Role> ite = user.getRoles().iterator(); ite.hasNext();)
            {
                Role role = ite.next();
                if (!PALAdminUtil.isEmpty(role.getName()))
                {
                    getRoleManager().addRoleToUser(user.getName(),
                            role.getName());
                }
            }

            // Add principal rule
            Principal principal = getUserPrincipal(user.getName());
            for (Iterator<PrincipalRule> ite = user.getPrincipalRules()
                    .iterator(); ite.hasNext();)

            {
                PrincipalRule principalRule = ite.next();
                if (!PALAdminUtil.isEmpty(principalRule.getLocatorName())
                        && !PALAdminUtil.isEmpty(principalRule
                                .getProfilingRuleName()))
                {
                    getProfiler().setRuleForPrincipal(
                            principal,
                            getProfiler().getRule(
                                    principalRule.getProfilingRuleName()),
                            principalRule.getLocatorName());
                }
            }

            // Create user's psml 
            if (!getPageManager().userFolderExists(user.getName()))
            {
                //TODO where is template folder
                String templateFolder = PALAdminConstants.DEFAULT_TEMPLATE_FOLDER;
                Folder source = getPageManager().getFolder(templateFolder);
                getPageManager().deepCopyFolder(source,
                        Folder.USER_FOLDER + user.getName(), user.getName());
            }
        }
        catch (Exception e)
        {
            logger.error("Could not create user: " + user, e);
            throw new PALAdminException(e);
        }
    }

    private Subject getUserSubject(String username) throws SecurityException
    {
        org.apache.jetspeed.security.User user = getUserManager().getUser(
                username);
        return user.getSubject();
    }

    private PasswordCredential getPasswordCredential(String username)
            throws SecurityException
    {
        org.apache.jetspeed.security.User user = getUserManager().getUser(
                username);

        PasswordCredential credential = null;

        Set<Object> credentials = user.getSubject().getPrivateCredentials();
        Iterator iter = credentials.iterator();
        while (iter.hasNext())
        {
            Object o = iter.next();
            if (o instanceof PasswordCredential)
            {
                credential = (PasswordCredential) o;
                break;
            }
        }
        return credential;
    }

    public void deleteUser(User user) throws PALAdminException
    {
        //Debug
        if (PALAdminConstants.DEBUG)
        {
            return;
        }

        try
        {
            getUserManager().removeUser(user.getName());

            // Remove user's psml folder
            if (getPageManager().folderExists(
                    Folder.USER_FOLDER + user.getName()))
            {
                Folder folder = pageManager.getFolder(Folder.USER_FOLDER
                        + user.getName());
                pageManager.removeFolder(folder);
            }

        }
        catch (Exception e)
        {
            throw new PALAdminException(e);
        }
    }

    public User getUser(String username)
    {
        User user;
        try
        {
            org.apache.jetspeed.security.User j2user = getUserManager()
                    .getUser(username);
            if (j2user == null)
            {
                return null;
            }
            PasswordCredential credential = getPasswordCredential(username);

            user = new User();
            user.setName(username);
            user.setPassword(null);//TODO null?
            user.setUpdateRequired(credential.isUpdateRequired());
            user.setLastLogon(credential.getLastAuthenticationDate());
            user.setUserEnabled(credential.isEnabled());
            user.setExpirationDate(credential.getExpirationDate());
            if (credential.isExpired())
            {
                user.setExpires(PALAdminConstants.USER_CREDENTIAL_EXPIRED);
            }
            else
            {
                user.setExpires(PALAdminConstants.USER_CREDENTIAL_ACTIVE);
            }

            //TODO groups
            //TODO roles
            //TODO principal rules
            return user;
        }
        catch (SecurityException e)
        {
            logger.error("Could not get user: " + username, e);
            return null;
        }

    }

    public void updateUser(User user) throws PALAdminException
    {
        // cannot change username

        try
        {
            // change password
            boolean passwordSet = false;
            if (user.getPassword() != null
                    && user.getPassword().trim().length() > 0)
            {
                getUserManager().setPassword(user.getName(), null,
                        user.getPassword());
                passwordSet = true;
            }

            PasswordCredential credential = getPasswordCredential(user
                    .getName());
            if (credential != null)
            {
                // update update required
                if (user.isUpdateRequired() != credential.isUpdateRequired())
                {
                    getUserManager().setPasswordUpdateRequired(user.getName(),
                            user.isUpdateRequired());
                }

                // update user enabled
                if (user.isUserEnabled() != credential.isEnabled())
                {
                    userManager.setPasswordEnabled(user.getName(), user
                            .isUserEnabled());
                }

                // update expired
                if (!passwordSet
                        && user.getExpires().equals(
                                PALAdminConstants.USER_CREDENTIAL_ACTIVE))
                {
                    //TODO is it correct?
                    userManager.setPasswordExpiration(user.getName(), null);
                }
                else if (!passwordSet
                        && user.getExpires().equals(
                                PALAdminConstants.USER_CREDENTIAL_EXPIRED))
                {
                    java.sql.Date today = new java.sql.Date(Calendar
                            .getInstance().getTimeInMillis());
                    userManager.setPasswordExpiration(user.getName(), today);
                }
                else if (user.getExpires().equals(
                        PALAdminConstants.USER_CREDENTIAL_EXTEND))
                {
                    userManager.setPasswordExpiration(user.getName(), null);
                }
                else if (user.getExpires().equals(
                        PALAdminConstants.USER_CREDENTIAL_EXTEND_UNLIMITED))
                {
                    userManager.setPasswordExpiration(user.getName(),
                            InternalCredential.MAX_DATE);
                }

            }
        }
        catch (SecurityException e)
        {
            logger.error("Could not update user information correctly. user="
                    + user, e);
            throw new PALAdminException(
                    "Could not update user information correctly. user=" + user,
                    e);
        }
    }

    public void addUserToGroup(User user, Group group) throws PALAdminException
    {
        try
        {
            if (getGroupManager().groupExists(group.getName()))
            {
                getGroupManager().addUserToGroup(user.getName(),
                        group.getName());
            }
            else
            {
                throw new PALAdminException("The group does not exist. group="
                        + group);
            }
        }
        catch (SecurityException e)
        {
            logger.error("Could not add the group to the user. user=" + user
                    + ", group=" + group, e);
            throw new PALAdminException(
                    "Could not add the group to the user. user=" + user
                            + ", group=" + group, e);
        }
    }

    public void removeGroupFromUser(User user, Group group)
            throws PALAdminException
    {
        try
        {
            if (getGroupManager().groupExists(group.getName()))
            {
                getGroupManager().removeUserFromGroup(user.getName(),
                        group.getName());
            }
            else
            {
                throw new PALAdminException("The group does not exist. group="
                        + group);
            }
        }
        catch (SecurityException e)
        {
            logger.error("Could not remove the group from the user. user="
                    + user + ", group=" + group, e);
            throw new PALAdminException(
                    "Could not remove the group from the user. user=" + user
                            + ", group=" + group, e);
        }

    }

    public boolean exists(Group group)
    {
        return getGroupManager().groupExists(group.getName());
    }

    public void addUserToRole(User user, Role role) throws PALAdminException
    {
        try
        {
            if (getRoleManager().roleExists(role.getName()))
            {
                getRoleManager().addRoleToUser(user.getName(), role.getName());
            }
            else
            {
                throw new PALAdminException("The role does not exist. role="
                        + role);
            }
        }
        catch (SecurityException e)
        {
            logger.error("Could not add the role to the user. user=" + user
                    + ", role=" + role, e);
            throw new PALAdminException(
                    "Could not add the role to the user. user=" + user
                            + ", role=" + role, e);
        }
    }

    public void removeRoleFromUser(User user, Role role)
            throws PALAdminException
    {
        try
        {
            if (getRoleManager().roleExists(role.getName()))
            {
                getRoleManager().removeRoleFromUser(user.getName(),
                        role.getName());
            }
            else
            {
                throw new PALAdminException("The role does not exist. role="
                        + role);
            }
        }
        catch (SecurityException e)
        {
            logger.error("Could not remove the role from the user. user="
                    + user + ", role=" + role, e);
            throw new PALAdminException(
                    "Could not remove the role from the user. user=" + user
                            + ", role=" + role, e);
        }

    }

    public boolean exists(Role role)
    {
        return getRoleManager().roleExists(role.getName());
    }

    public void removePrincipalRuleFromUser(User user,
            PrincipalRule principalRule) throws PALAdminException
    {
        try
        {
            Iterator<org.apache.jetspeed.profiler.rules.PrincipalRule> it = getProfiler()
                    .getRulesForPrincipal(getUserPrincipal(user.getName()))
                    .iterator();
            while (it.hasNext())
            {
                org.apache.jetspeed.profiler.rules.PrincipalRule rule = it
                        .next();
                if (rule.getLocatorName()
                        .equals(principalRule.getLocatorName()))
                {
                    //TODO does not work??
                    getProfiler().deletePrincipalRule(rule);
                }
            }

        }
        catch (Exception e)
        {
            logger.error(
                    "Could not remove the principalRule from the user. user="
                            + user + ", principalRule=" + principalRule, e);
            throw new PALAdminException(
                    "Could not remove the principalRule from the user. user="
                            + user + ", principalRule=" + principalRule, e);
        }

    }

    //
    // Role
    //

    public List<Role> getRoles(RolePager pager)
    {
        //Debug
        if (PALAdminConstants.DEBUG)
        {
            List<Role> list = new ArrayList<Role>();
            Role r = new Role();
            r.setName("ROLE");
            list.add(r);
            return list;
        }

        List<Role> roleList = new ArrayList<Role>();
        List<Principal> principals = getRoleSecurityHandler()
                .getRolePrincipals(pager.getFilter());

        // set count
        pager.setCount(principals.size());

        // create list
        int offset = pager.getOffset();
        int limit = pager.getLimit();
        int size = principals.size();
        if (logger.isDebugEnabled())
        {
            logger.debug("getRoles(RolePager) - pager.getFilter()="
                    + pager.getFilter());
            logger.debug("getRoles(RolePager) - offset=" + offset + ", limit="
                    + limit + ", size=" + size);
        }
        for (int i = offset; i < offset + limit && i < size; i++)
        {
            Role r = new Role();
            r.setName(principals.get(i).getName());
            roleList.add(r);
        }

        return roleList;

    }

    public void insertRole(Role role) throws PALAdminException
    {
        //Debug
        if (PALAdminConstants.DEBUG)
        {
            return;
        }

        try
        {
            getRoleManager().addRole(role.getName());
        }
        catch (SecurityException e)
        {
            throw new PALAdminException(e);
        }
    }

    public void deleteRole(Role role) throws PALAdminException
    {
        //Debug
        if (PALAdminConstants.DEBUG)
        {
            return;
        }

        try
        {
            getRoleManager().removeRole(role.getName());
        }
        catch (SecurityException e)
        {
            throw new PALAdminException(e);
        }
    }

    public List<String> getRoleNames()
    {
        //Debug
        if (PALAdminConstants.DEBUG)
        {
            List<String> list = new ArrayList<String>();
            list.add("ROLE");
            return list;
        }

        List<String> roleList = new ArrayList<String>();
        List<Principal> principals = getRoleSecurityHandler()
                .getRolePrincipals("");

        for (Iterator<Principal> ite = principals.iterator(); ite.hasNext();)
        {
            roleList.add(ite.next().getName());
        }

        return roleList;

    }

    public void addPrincipalRuleToUser(User user, PrincipalRule principalRule)
            throws PALAdminException
    {
        try
        {
            getProfiler()
                    .setRuleForPrincipal(
                            getUserPrincipal(user.getName()),
                            getProfiler().getRule(
                                    principalRule.getProfilingRuleName()),
                            principalRule.getLocatorName());
        }
        catch (SecurityException e)
        {
            logger.error("Could not add the principal rule to user. user="
                    + user + ", principalRule=" + principalRule, e);
            throw new PALAdminException(
                    "Could not add the principal rule to user. user=" + user
                            + ", principalRule=" + principalRule, e);
        }
    }

    //
    // Group
    //

    public List<Group> getGroups(GroupPager pager)
    {
        //Debug
        if (PALAdminConstants.DEBUG)
        {
            List<Group> list = new ArrayList<Group>();
            Group entity = new Group();
            entity.setName("GROUP");
            list.add(entity);
            return list;
        }

        List<Group> groupList = new ArrayList<Group>();
        List<Principal> principals = getGroupSecurityHandler()
                .getGroupPrincipals(pager.getFilter());

        // set count
        pager.setCount(principals.size());

        // create list
        int offset = pager.getOffset();
        int limit = pager.getLimit();
        int size = principals.size();
        if (logger.isDebugEnabled())
        {
            logger.debug("getGroups(GroupPager) - pager.getFilter()="
                    + pager.getFilter());
            logger.debug("getGroups(GroupPager) - offset=" + offset
                    + ", limit=" + limit + ", size=" + size);
        }
        for (int i = offset; i < offset + limit && i < size; i++)
        {
            Group r = new Group();
            r.setName(principals.get(i).getName());
            groupList.add(r);
        }

        return groupList;

    }

    public void insertGroup(Group group) throws PALAdminException
    {
        //Debug
        if (PALAdminConstants.DEBUG)
        {
            return;
        }

        try
        {
            getGroupManager().addGroup(group.getName());
        }
        catch (SecurityException e)
        {
            throw new PALAdminException(e);
        }
    }

    public void deleteGroup(Group group) throws PALAdminException
    {
        //Debug
        if (PALAdminConstants.DEBUG)
        {
            return;
        }

        try
        {
            getGroupManager().removeGroup(group.getName());
        }
        catch (SecurityException e)
        {
            throw new PALAdminException(e);
        }
    }

    public List<String> getGroupNames()
    {
        //Debug
        if (PALAdminConstants.DEBUG)
        {
            List<String> list = new ArrayList<String>();
            list.add("GROUP");
            return list;
        }

        List<String> groupList = new ArrayList<String>();
        List<Principal> principals = getGroupSecurityHandler()
                .getGroupPrincipals("");

        for (Iterator<Principal> ite = principals.iterator(); ite.hasNext();)
        {
            groupList.add(ite.next().getName());
        }

        return groupList;

    }

    //TODO String to User
    public List<String> getGroupNamesForUsername(String username)
    {
        //Debug
        if (PALAdminConstants.DEBUG)
        {
            List<String> list = new ArrayList<String>();
            list.add("GROUP");
            return list;
        }

        List<String> groupList = new ArrayList<String>();
        Set<Principal> principals = getSecurityProvider()
                .getSecurityMappingHandler().getGroupPrincipals(username);
        for (Iterator<Principal> ite = principals.iterator(); ite.hasNext();)
        {
            groupList.add(ite.next().getName());
        }

        return groupList;

    }

    public List<String> getRoleNamesForUsername(String username)
    {
        //Debug
        if (PALAdminConstants.DEBUG)
        {
            List<String> list = new ArrayList<String>();
            list.add("GROUP");
            return list;
        }

        List<String> roleList = new ArrayList<String>();
        Set<Principal> principals = getSecurityProvider()
                .getSecurityMappingHandler().getRolePrincipals(username);
        for (Iterator<Principal> ite = principals.iterator(); ite.hasNext();)
        {
            roleList.add(ite.next().getName());
        }

        return roleList;

    }

    //
    // ProfilingRule
    //
    public List<String> getProfilingRuleNames()
    {
        ArrayList<String> list = new ArrayList<String>();
        for (Iterator<org.apache.jetspeed.profiler.rules.ProfilingRule> ite = getProfiler()
                .getRules().iterator(); ite.hasNext();)
        {
            org.apache.jetspeed.profiler.rules.ProfilingRule profilingRule = ite
                    .next();
            list.add(profilingRule.getId());
        }
        return list;
    }

    public List<PrincipalRule> getPrincipalRuleForUser(User user)
    {
        ArrayList<PrincipalRule> list = new ArrayList<PrincipalRule>();
        try
        {
            Principal userPrincipal = getUserPrincipal(user.getName());
            for (Iterator<org.apache.jetspeed.profiler.rules.PrincipalRule> ite = getProfiler()
                    .getRulesForPrincipal(userPrincipal).iterator(); ite
                    .hasNext();)
            {
                org.apache.jetspeed.profiler.rules.PrincipalRule rule = ite
                        .next();
                PrincipalRule principalRule = new PrincipalRule();
                principalRule.setLocatorName(rule.getLocatorName());
                principalRule.setProfilingRuleName(rule.getProfilingRule()
                        .getId());
                list.add(principalRule);
            }
        }
        catch (SecurityException e)
        {
            logger.error("Could not get principal rules.", e);
        }
        return list;
    }

    private Principal getUserPrincipal(String username)
            throws SecurityException
    {
        return PALAdminUtil.getPrincipal(getUserSubject(username),
                UserPrincipal.class);
    }

}
