/*
 * Copyright 2005-2006 Portal Application Laboratory project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, 
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package jp.sf.pal.jsf.filter;

import java.io.IOException;

import javax.portlet.ActionRequest;
import javax.portlet.ActionResponse;
import javax.portlet.PortletConfig;
import javax.portlet.PortletException;
import javax.portlet.RenderRequest;
import javax.portlet.RenderResponse;

import jp.sf.pal.jsf.multipart.MultipartPortletRequestWrapper;

import org.apache.commons.fileupload.portlet.PortletFileUpload;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.portals.bridges.portletfilter.PortletFilter;
import org.apache.portals.bridges.portletfilter.PortletFilterChain;
import org.apache.portals.bridges.portletfilter.PortletFilterConfig;

public class FileUploadPortletFilter implements PortletFilter {
    private static final Log log = LogFactory
            .getLog(FileUploadPortletFilter.class);

    private static final String UPLOAD_REPOSITORY_PATH = "uploadRepositoryPath";

    private static final String UPLOAD_THRESHOLD_SIZE = "uploadThresholdSize";

    private static final String UPLOAD_MAX_FILE_SIZE = "uploadMaxFileSize";

    private static final String MULTIPART_ENCODING = "multipartEncoding";

    private int uploadMaxFileSize = 100 * 1024 * 1024; // 10 MB

    private int uploadThresholdSize = 1 * 1024 * 1024; // 1 MB

    private String uploadRepositoryPath = null; // standard temp directory

    private String multipartEncoding = null;

    private PortletConfig portletConfig;

    /**
     * Called by init method of Portlet to initialize this portlet
     * filter.
     * 
     * @param filterConfig
     * @throws PortletException
     */
    public void init(PortletFilterConfig filterConfig) throws PortletException {
        if (log.isDebugEnabled())
            log.debug("Initializing ExtensionsPortletFilter.");

        setPortletConfig(filterConfig.getPortletConfig());

        // for inputFileUpload
        String param = filterConfig.getInitParameter(UPLOAD_MAX_FILE_SIZE);

        uploadMaxFileSize = resolveSize(param, uploadMaxFileSize);

        param = filterConfig.getInitParameter(UPLOAD_THRESHOLD_SIZE);

        uploadThresholdSize = resolveSize(param, uploadThresholdSize);

        uploadRepositoryPath = filterConfig
                .getInitParameter(UPLOAD_REPOSITORY_PATH);

        multipartEncoding = filterConfig.getInitParameter(MULTIPART_ENCODING);

        if (log.isDebugEnabled()) {
            log.debug("uploadMaxFileSize=" + uploadMaxFileSize);
            log.debug("uploadThresholdSize=" + uploadThresholdSize);
            log.debug("uploadRepositoryPath=" + uploadRepositoryPath);
            log.debug("multipartEncoding=" + multipartEncoding);
            // log.debug("mimeType=" + mimeType);
        }
    }

    /**
     * Called by render method of Portlet to put tags, such as
     * &lt;style&gt;, into &lt;head&gt;.
     * 
     * @param request
     * @param response
     * @param chain
     *            PortletFilterChain instance
     * @throws PortletException
     */
    public void renderFilter(RenderRequest request, RenderResponse response,
            PortletFilterChain chain) throws PortletException, IOException {
        if (log.isDebugEnabled()) {
            log.debug("called renderFilter.");
        }

        chain.renderFilter(request, response);
    }

    /**
     * Called by render method of Portlet to wrap the request when
     * it has a multipart content.
     * 
     * @param request
     * @param response
     * @param chain
     *            PortletFilterChain instance
     * @throws PortletException
     */
    public void processActionFilter(ActionRequest request,
            ActionResponse response, PortletFilterChain chain)
            throws PortletException, IOException {
        if (log.isDebugEnabled()) {
            log.debug("called processActionFilter.");
        }
        // Check multipart/form-data
        if (PortletFileUpload.isMultipartContent(request)) {
            if (log.isDebugEnabled())
                log.debug("ActionRequest is multipart content.");
            if (multipartEncoding != null) {
                if (log.isDebugEnabled())
                    log.debug("Mutlipart encoding is " + multipartEncoding);
                request.setCharacterEncoding(multipartEncoding);
            }
            request = new MultipartPortletRequestWrapper(request,
                    uploadMaxFileSize, uploadThresholdSize,
                    uploadRepositoryPath);
        }

        // call next processActionFilter
        chain.processActionFilter(request, response);
    }

    /**
     * Called by destroy method of Portlet to destroy this portlet
     * filter.
     */
    public void destroy() {
        uploadRepositoryPath = null;
        multipartEncoding = null;
        portletConfig = null;
    }

    private int resolveSize(String param, int defaultValue) {
        int numberParam = defaultValue;

        if (param != null) {
            param = param.toLowerCase();
            int factor = 1;
            String number = param;

            if (param.endsWith("g")) {
                factor = 1024 * 1024 * 1024;
                number = param.substring(0, param.length() - 1);
            } else if (param.endsWith("m")) {
                factor = 1024 * 1024;
                number = param.substring(0, param.length() - 1);
            } else if (param.endsWith("k")) {
                factor = 1024;
                number = param.substring(0, param.length() - 1);
            }

            numberParam = Integer.parseInt(number) * factor;
        }
        return numberParam;
    }

    /**
     * @return Returns the portletConfig.
     */
    public PortletConfig getPortletConfig() {
        return portletConfig;
    }

    /**
     * @param portletConfig
     *            The portletConfig to set.
     */
    public void setPortletConfig(PortletConfig portletConfig) {
        this.portletConfig = portletConfig;
    }

    public boolean isValidContentType(String contentType) {
        return contentType.startsWith("text/html")
                || contentType.startsWith("text/xml")
                || contentType.startsWith("application/xhtml+xml")
                || contentType.startsWith("application/xml");
    }

}
