/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package jp.sf.pal.admin.logic;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.Serializable;

import jp.sf.pal.admin.PALAdminConstants;
import jp.sf.pal.admin.PALAdminException;
import jp.sf.pal.admin.deployer.PortletDeployer;
import jp.sf.pal.admin.entity.PortletApplication;
import jp.sf.pal.admin.util.DeploymentUtil;
import jp.sf.pal.admin.util.PALAdminUtil;
import jp.sf.pal.admin.util.PortalComponentUtil;

import org.apache.jetspeed.components.portletregistry.PortletRegistry;
import org.apache.jetspeed.components.portletregistry.RegistryException;
import org.apache.jetspeed.om.common.portlet.MutablePortletApplication;
import org.apache.jetspeed.tools.pamanager.servletcontainer.ApplicationServerManager;
import org.apache.jetspeed.tools.pamanager.servletcontainer.ApplicationServerManagerResult;
import org.seasar.framework.log.Logger;

public class PortletDeploymentLogic implements Serializable
{
    /**
     * 
     */
    private static final long serialVersionUID = -4957269606718394576L;

    /**
     * Logger for this class
     */
    private static final Logger logger = Logger
            .getLogger(PortletManagementLogic.class);

    private transient ApplicationServerManager applicationServerManager = null;

    private transient PortletRegistry portletRegistry = null;

    /**
     * @return the applicationServerManager
     */
    public ApplicationServerManager getApplicationServerManager()
    {
        if (applicationServerManager == null)
        {
            applicationServerManager = PortalComponentUtil
                    .getApplicationServerManager();
        }
        return applicationServerManager;
    }

    /**
     * @param applicationServerManager the applicationServerManager to set
     */
    public void setApplicationServerManager(
            ApplicationServerManager applicationServerManager)
    {
        this.applicationServerManager = applicationServerManager;
    }

    /**
     * @return the portletRegistry
     */
    public PortletRegistry getPortletRegistry()
    {
        if (portletRegistry == null)
        {
            portletRegistry = PortalComponentUtil.getPortletRegistry();
        }
        return portletRegistry;
    }

    /**
     * @param portletRegistry the portletRegistry to set
     */
    public void setPortletRegistry(PortletRegistry portletRegistry)
    {
        this.portletRegistry = portletRegistry;
    }

    synchronized public void deploy(PortletApplication portlet)
            throws PALAdminException
    {

        if (DeploymentUtil.isDeploying())
        {
            //TODO check timeout

            throw new PALAdminException("Other deployment process is running.");
        }

        // create deployer
        PortletDeployer deployer = new PortletDeployer(portlet);
        deployer.setDeploymentManager(PortalComponentUtil
                .getDeploymentManager());

        // set deployment status to application context
        DeploymentUtil.setDeploymentStatus(deployer.getDeploymentStatus());

        try
        {
            // start deployer
            deployer.start();
        }
        catch (Exception e)
        {
            logger.error("Could not start deployment process.", e);
            throw new PALAdminException("Could not start deployment process.",
                    e);
        }
    }

    public File storeLocalFile(InputStream inputStream)
            throws PALAdminException
    {
        File localTempFile = null;
        OutputStream outputStream = null;
        try
        {
            localTempFile = File.createTempFile(
                    PALAdminConstants.LOCAL_TEMP_FILENAME_PREFIX,
                    PALAdminConstants.LOCAL_TEMP_FILENAME_SUFFIX);
            outputStream = new FileOutputStream(localTempFile);
            PALAdminUtil.drain(inputStream, outputStream);
            outputStream.flush();
        }
        catch (FileNotFoundException e)
        {
            logger
                    .error(
                            "Could not store the uploaded file. Could not find a local temp file.",
                            e);
            throw new PALAdminException(
                    "Could not store the uploaded file. Could not find a local temp file.",
                    e);
        }
        catch (IOException e)
        {
            logger.error("Could not store the uploaded file.", e);
            throw new PALAdminException("Could not store the uploaded file.", e);
        }
        finally
        {
            if (inputStream != null)
            {
                try
                {
                    inputStream.close();
                }
                catch (IOException e)
                {
                    logger.error("Could not close the input stream.", e);
                    throw new PALAdminException(
                            "Could not close the input stream.", e);
                }
            }
            if (outputStream != null)
            {
                try
                {
                    outputStream.close();
                }
                catch (IOException e)
                {
                    logger.error("Could not close the output stream.", e);
                    throw new PALAdminException(
                            "Could not close the output stream.", e);

                }
            }
        }

        return localTempFile;
    }

    public void startPortletApplication(String portletName)
            throws PALAdminException
    {
        MutablePortletApplication pa = getPortletRegistry()
                .getPortletApplication(portletName);
        try
        {
            ApplicationServerManagerResult result = getApplicationServerManager()
                    .start(pa.getWebApplicationDefinition().getContextRoot());
            if (!result.isOk())
            {
                throw new PALAdminException(result.getMessage());
            }
        }
        catch (IOException e)
        {
            logger.error("Could not start the portlet: " + portletName, e);
            throw new PALAdminException("Could not start the portlet: "
                    + portletName, e);
        }
    }

    public void stopPortletApplication(String portletName)
            throws PALAdminException
    {
        MutablePortletApplication pa = getPortletRegistry()
                .getPortletApplication(portletName);
        try
        {
            ApplicationServerManagerResult result = getApplicationServerManager()
                    .stop(pa.getWebApplicationDefinition().getContextRoot());
            if (!result.isOk())
            {
                throw new PALAdminException(result.getMessage());
            }
        }
        catch (IOException e)
        {
            logger.error("Could not stop the portlet: " + portletName, e);
            throw new PALAdminException("Could not stop the portlet: "
                    + portletName, e);
        }
    }

    public void undeployPortletApplication(String portletName)
            throws PALAdminException
    {
        MutablePortletApplication pa = getPortletRegistry()
                .getPortletApplication(portletName);
        try
        {
            ApplicationServerManagerResult result = getApplicationServerManager()
                    .undeploy(pa.getWebApplicationDefinition().getContextRoot());
            if (!result.isOk())
            {
                throw new PALAdminException(result.getMessage());
            }
        }
        catch (IOException e)
        {
            logger.error("Could not undeploy the portlet: " + portletName, e);
            throw new PALAdminException("Could not undeploy the portlet: "
                    + portletName, e);
        }
    }

    public void deletePortletApplication(String portletName)
            throws PALAdminException
    {
        MutablePortletApplication pa = getPortletRegistry()
                .getPortletApplication(portletName);
        try
        {
            getPortletRegistry().removeApplication(pa);
        }
        catch (RegistryException e)
        {
            logger.error("Could not delete the portlet: " + portletName, e);
            throw new PALAdminException("Could not delete the portlet: "
                    + portletName, e);
        }
    }

}
