/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package jp.sf.pal.admin.logic;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.List;

import jp.sf.pal.admin.PALAdminConstants;
import jp.sf.pal.admin.entity.PortletApplication;
import jp.sf.pal.admin.pager.PortletApplicationPager;
import jp.sf.pal.admin.pager.RemotePortletApplicationPager;
import jp.sf.pal.admin.rpad.repository.RepositoryManager;
import jp.sf.pal.admin.util.PortalComponentUtil;

import org.apache.jetspeed.components.portletregistry.PortletRegistry;
import org.apache.jetspeed.factory.PortletFactory;
import org.apache.jetspeed.om.common.portlet.MutablePortletApplication;
import org.apache.jetspeed.tools.pamanager.servletcontainer.ApplicationServerManager;
import org.seasar.framework.log.Logger;

public class PortletManagementLogic implements Serializable
{
    /**
     * 
     */
    private static final long serialVersionUID = 4388302393628406530L;

    /**
     * Logger for this class
     */
    private static final Logger logger = Logger
            .getLogger(PortletManagementLogic.class);

    private transient ApplicationServerManager applicationServerManager = null;

    private transient PortletRegistry portletRegistry = null;

    private transient PortletFactory portletFactory = null;

    /**
     * @return the applicationServerManager
     */
    public ApplicationServerManager getApplicationServerManager()
    {
        if (applicationServerManager == null)
        {
            applicationServerManager = PortalComponentUtil
                    .getApplicationServerManager();
        }
        return applicationServerManager;
    }

    /**
     * @param applicationServerManager the applicationServerManager to set
     */
    public void setApplicationServerManager(
            ApplicationServerManager applicationServerManager)
    {
        this.applicationServerManager = applicationServerManager;
    }

    /**
     * @return the portletFactory
     */
    public PortletFactory getPortletFactory()
    {
        if (portletFactory == null)
        {
            portletFactory = PortalComponentUtil.getPortletFactory();
        }
        return portletFactory;
    }

    /**
     * @param portletFactory the portletFactory to set
     */
    public void setPortletFactory(PortletFactory portletFactory)
    {
        this.portletFactory = portletFactory;
    }

    /**
     * @return the portletRegistry
     */
    public PortletRegistry getPortletRegistry()
    {
        if (portletRegistry == null)
        {
            portletRegistry = PortalComponentUtil.getPortletRegistry();
        }
        return portletRegistry;
    }

    /**
     * @param portletRegistry the portletRegistry to set
     */
    public void setPortletRegistry(PortletRegistry portletRegistry)
    {
        this.portletRegistry = portletRegistry;
    }

    public List<PortletApplication> getPortletApplications(
            PortletApplicationPager pager)
    {
        List<PortletApplication> portletApplicationList = new ArrayList<PortletApplication>();
        List<MutablePortletApplication> portlets = new ArrayList<MutablePortletApplication>(
                getPortletRegistry().getPortletApplications());

        // set count
        pager.setCount(portlets.size());

        // create list
        int offset = pager.getOffset();
        int limit = pager.getLimit();
        int size = portlets.size();
        if (logger.isDebugEnabled())
        {
            logger
                    .debug("getPortletApplications(PortletApplicationPager) - offset="
                            + offset + ", limit=" + limit + ", size=" + size);
        }
        for (int i = offset; i < offset + limit && i < size; i++)
        {
            MutablePortletApplication pa = portlets.get(i);

            PortletApplication portlet = new PortletApplication();
            portlet.setName(pa.getName());
            portlet.setVersion(pa.getVersion());
            if (pa.getApplicationType() == MutablePortletApplication.LOCAL)
            {
                portlet.setPath("<local>");
                portlet
                        .setApplicationType(PALAdminConstants.PORTLET_APPLICATION_TYPE_LOCAL);
            }
            else
            {
                portlet.setPath(pa.getWebApplicationDefinition()
                        .getContextRoot());
                portlet
                        .setApplicationType(PALAdminConstants.PORTLET_APPLICATION_TYPE_WEBAPP);
            }
            if (getPortletFactory().isPortletApplicationRegistered(pa))
            {
                portlet
                        .setStatus(PALAdminConstants.PORTLET_APPLICATION_STATUS_RUNNING);
            }
            else
            {
                portlet
                        .setStatus(PALAdminConstants.PORTLET_APPLICATION_STATUS_STOPPED);
            }

            portletApplicationList.add(portlet);
        }

        return portletApplicationList;
    }

    public boolean isServerManagerAvailable()
    {
        return getApplicationServerManager() != null
                && getApplicationServerManager().isConnected();
    }

    public List<PortletApplication> getPortletApplications(
            RemotePortletApplicationPager pager)
    {
        RepositoryManager repositoryManager = RepositoryManager.getInstance();

        List<PortletApplication> portletApplicationList = new ArrayList<PortletApplication>();
        List<PortletApplication> portlets = null;
        if (pager.getRepositoryName() == null)
        {
            portlets = repositoryManager.getPortletApplications();
        }
        else
        {
            portlets = repositoryManager.getPortletApplications(pager
                    .getRepositoryName());
        }

        // set count
        pager.setCount(portlets.size());

        // create list
        int offset = pager.getOffset();
        int limit = pager.getLimit();
        int size = portlets.size();
        if (logger.isDebugEnabled())
        {
            logger
                    .debug("getPortletApplications(PortletApplicationPager) - offset="
                            + offset + ", limit=" + limit + ", size=" + size);
        }
        for (int i = offset; i < offset + limit && i < size; i++)
        {
            portletApplicationList.add(portlets.get(i));
        }

        return portletApplicationList;
    }

    public PortletApplication getPortletApplication(String groupId,
            String artifactId, String version, String packaging)
    {
        RepositoryManager repositoryManager = RepositoryManager.getInstance();
        return repositoryManager.getPortletApplication(groupId, artifactId,
                version, packaging);
    }
}
