/*
 * Copyright 2005-2007 Portal Application Laboratory and the Others.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package jp.sf.pal.admin.logic;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import javax.faces.context.FacesContext;

import jp.sf.pal.admin.PALAdminConstants;
import jp.sf.pal.admin.entity.NodeInfo;
import jp.sf.pal.admin.util.PALAdminUtil;
import jp.sf.pal.admin.util.PortalComponentUtil;
import jp.sf.pal.common.CommonException;

import org.apache.jetspeed.JetspeedActions;
import org.apache.jetspeed.PortalReservedParameters;
import org.apache.jetspeed.components.portletentity.PortletEntityAccessComponent;
import org.apache.jetspeed.components.portletentity.PortletEntityNotStoredException;
import org.apache.jetspeed.components.portletregistry.PortletRegistry;
import org.apache.jetspeed.container.window.FailedToRetrievePortletWindow;
import org.apache.jetspeed.container.window.PortletWindowAccessor;
import org.apache.jetspeed.decoration.DecorationFactory;
import org.apache.jetspeed.decoration.LayoutInfo;
import org.apache.jetspeed.om.common.SecurityConstraint;
import org.apache.jetspeed.om.common.SecurityConstraints;
import org.apache.jetspeed.om.common.portlet.MutablePortletApplication;
import org.apache.jetspeed.om.common.portlet.PortletDefinitionComposite;
import org.apache.jetspeed.om.folder.Folder;
import org.apache.jetspeed.om.folder.FolderNotFoundException;
import org.apache.jetspeed.om.folder.InvalidFolderException;
import org.apache.jetspeed.om.page.ContentFragment;
import org.apache.jetspeed.om.page.ContentPage;
import org.apache.jetspeed.om.page.Fragment;
import org.apache.jetspeed.om.page.Page;
import org.apache.jetspeed.om.page.SecurityConstraintsDef;
import org.apache.jetspeed.page.FolderNotUpdatedException;
import org.apache.jetspeed.page.PageManager;
import org.apache.jetspeed.page.PageNotFoundException;
import org.apache.jetspeed.page.PageNotUpdatedException;
import org.apache.jetspeed.page.document.DocumentException;
import org.apache.jetspeed.page.document.DocumentNotFoundException;
import org.apache.jetspeed.page.document.Node;
import org.apache.jetspeed.page.document.NodeException;
import org.apache.jetspeed.page.document.UnsupportedDocumentTypeException;
import org.apache.jetspeed.profiler.Profiler;
import org.apache.jetspeed.request.RequestContext;
import org.apache.jetspeed.security.PermissionManager;
import org.apache.jetspeed.security.SecurityAccessController;
import org.apache.pluto.om.window.PortletWindow;
import org.seasar.framework.log.Logger;

public class SiteEditorLogic implements Serializable {
    /**
     * 
     */
    private static final long serialVersionUID = 3488261005315934120L;

    /**
     * Logger for this class
     */
    private static final Logger logger = Logger
            .getLogger(SiteEditorLogic.class);

    private transient PageManager pageManager = null;

    private transient Profiler profiler = null;

    private transient PortletRegistry portletRegistry = null;

    private transient DecorationFactory decorationFactory = null;

    private transient PermissionManager permissionManager = null;

    private transient PortletWindowAccessor portletWindowAccessor = null;

    private transient PortletEntityAccessComponent portletEntityAccessComponent = null;

    private transient SecurityAccessController securityAccessController = null;

    private transient RequestContext requestContext = null;

    /**
     * @return the pageManager
     */
    public PageManager getPageManager() {
        if (pageManager == null) {
            pageManager = PortalComponentUtil.getPageManager();
        }
        return pageManager;
    }

    /**
     * @param pageManager the pageManager to set
     */
    public void setPageManager(PageManager pageManager) {
        this.pageManager = pageManager;
    }

    /**
     * @return the profiler
     */
    public PortletRegistry getPortletRegistry() {
        if (portletRegistry == null) {
            portletRegistry = PortalComponentUtil.getPortletRegistry();
        }
        return portletRegistry;
    }

    /**
     * @param profiler the profiler to set
     */
    public void setPortletRegistry(PortletRegistry portletRegistry) {
        this.portletRegistry = portletRegistry;
    }

    /**
     * @return the profiler
     */
    public Profiler getProfiler() {
        if (profiler == null) {
            profiler = PortalComponentUtil.getProfiler();
        }
        return profiler;
    }

    /**
     * @param profiler the profiler to set
     */
    public void setProfiler(Profiler profiler) {
        this.profiler = profiler;
    }

    /**
     * @return the decorationFactory
     */
    public DecorationFactory getDecorationFactory() {
        if (decorationFactory == null) {
            decorationFactory = PortalComponentUtil.getDecorationFactory();
        }
        return decorationFactory;
    }

    /**
     * @param decorationFactory the decorationFactory to set
     */
    public void setDecorationFactory(DecorationFactory decorationFactory) {
        this.decorationFactory = decorationFactory;
    }

    /**
     * @return the permissionManager
     */
    public PermissionManager getPermissionManager() {
        if (permissionManager == null) {
            permissionManager = PortalComponentUtil.getPermissionManager();
        }
        return permissionManager;
    }

    /**
     * @param permissionManager the permissionManager to set
     */
    public void setPermissionManager(PermissionManager permissionManager) {
        this.permissionManager = permissionManager;
    }

    /**
     * @return the permissionManager
     */
    public PortletWindowAccessor getPortletWindowAccessor() {
        if (portletWindowAccessor == null) {
            portletWindowAccessor = PortalComponentUtil
                    .getPortletWindowAccessor();
        }
        return portletWindowAccessor;
    }

    /**
     * @param portletWindowAccessor the portletWindowAccessor to set
     */
    public void setPortletWindowAccessor(
            PortletWindowAccessor portletWindowAccessor) {
        this.portletWindowAccessor = portletWindowAccessor;
    }

    /**
     * @return the portletEntityAccessComponent
     */
    public PortletEntityAccessComponent getPortletEntityAccessComponent() {
        if (portletEntityAccessComponent == null) {
            portletEntityAccessComponent = PortalComponentUtil
                    .getPortletEntityAccessComponent();
        }
        return portletEntityAccessComponent;
    }

    /**
     * @param portletEntityAccessComponent the portletEntityAccessComponent to set
     */
    public void setPortletEntityAccessComponent(
            PortletEntityAccessComponent portletEntityAccessComponent) {
        this.portletEntityAccessComponent = portletEntityAccessComponent;
    }

    /**
     * @return the SecurityAccessController
     */
    public SecurityAccessController getSecurityAccessController() {
        if (securityAccessController == null) {
            securityAccessController = PortalComponentUtil
                    .getSecurityAccessController();
        }
        return securityAccessController;
    }

    /**
     * @param SecurityAccessController the SecurityAccessController to set
     */
    public void setSecurityAccessController(
            SecurityAccessController securityAccessController) {
        this.securityAccessController = securityAccessController;
    }

    /**
     * @return the profiler
     */
    public RequestContext getRequestContext() {
        if (requestContext == null) {
            requestContext = (RequestContext) FacesContext.getCurrentInstance()
                    .getExternalContext().getRequestMap().get(
                            PortalReservedParameters.REQUEST_CONTEXT_ATTRIBUTE);
        }
        return requestContext;
    }

    /**
     * @param profiler the profiler to set
     */
    public void setRequestContext(RequestContext requestContext) {
        this.requestContext = requestContext;
    }

    public List<NodeInfo> getTreeNode(String path, String returnPath)
            throws CommonException {
        List<NodeInfo> treeList = new ArrayList<NodeInfo>();

        try {
            Folder rootFolder = getPageManager().getFolder("/");
            if (PALAdminUtil.checkAccess(rootFolder, JetspeedActions.VIEW)) {
                treeList
                        .add(new NodeInfo(NodeInfo.FOLDER, "/", "/",
                                returnPath, 0, true, true/*"/".equals(getFolderOrPagePath(path))*/));

                List<NodeInfo> childTreeList = constructTreeNode(rootFolder,
                        path, returnPath, 1);
                if (childTreeList != null) {
                    treeList.addAll(childTreeList);
                }
            }
            return treeList;
        } catch (FolderNotFoundException e) {
            logger.error("Could not access a folder: " + path, e);
            throw new CommonException("could.not.access.folder",
                    "Could not access a folder: " + path, e);
        } catch (InvalidFolderException e) {
            logger.error("Could not access a folder: " + path, e);
            throw new CommonException("could.not.access.folder",
                    "Could not access a folder: " + path, e);
        } catch (NodeException e) {
            logger.error("Could not access a folder: " + path, e);
            throw new CommonException("could.not.access.folder",
                    "Could not access a folder: " + path, e);
        }

    }

    protected boolean checkFolderName(String nodeName) {
        // TODO remove this folder?
        //        if (PALAdminConstants.DEFAULT_USER_TEMPLATE_FOLDER_NAME
        //                .equals(nodeName)) {
        //            return false;
        //        }
        return true;
    }

    protected List<NodeInfo> constructTreeNode(Folder parent, String path,
            String returnPath, int depth) throws CommonException {
        String targetPathName = "";
        String folderOrPagePath = getFolderOrPagePath(path);
        int parentPathLength = parent.getPath().length();
        if (parentPathLength < folderOrPagePath.length()) {
            targetPathName = folderOrPagePath.substring(parentPathLength);
            int startIndex = targetPathName.indexOf("/");
            if (startIndex == 0 && targetPathName.length() != 1) {
                targetPathName = targetPathName.substring(1);
            }
            int endIndex = targetPathName.indexOf("/");
            if (endIndex > 0) {
                targetPathName = targetPathName.substring(0, endIndex);
            }
        }

        if (targetPathName.endsWith(".psml")) {
            // page
            List<NodeInfo> treeList = new ArrayList<NodeInfo>();
            try {
                // child folders
                for (Iterator<Folder> ite = parent.getFolders().iterator(); ite
                        .hasNext();) {
                    Folder child = ite.next();
                    if (checkFolderName(child.getName())
                            && PALAdminUtil.checkAccess(child,
                                    JetspeedActions.VIEW)) {
                        treeList.add(new NodeInfo(NodeInfo.FOLDER, child
                                .getName(), child.getPath(), returnPath, depth,
                                false, false));
                        if (targetPathName.equals(child.getName())) {
                            List<NodeInfo> childTreeList = constructTreeNode(
                                    child, path, returnPath, depth + 1);
                            if (childTreeList != null) {
                                treeList.addAll(childTreeList);
                            }
                        }
                    }
                }
            } catch (DocumentException e) {
                logger.warn("Could not get folders: " + path, e);
            }

            try {
                // child pages
                for (Iterator<Page> ite = parent.getPages().iterator(); ite
                        .hasNext();) {
                    Page child = ite.next();
                    if (PALAdminUtil.checkAccess(child, JetspeedActions.VIEW)) {
                        if (targetPathName.equals(child.getName())) {
                            treeList.add(new NodeInfo(NodeInfo.PAGE, child
                                    .getName(), child.getPath(), returnPath,
                                    depth, true, true));
                            // layout and portlet
                            Fragment rootFragment = child.getRootFragment();
                            if (rootFragment != null) {
                                String fragmentId = getFragmentId(path);
                                boolean selected = true;
                                if (fragmentId == null || fragmentId.equals("")) {
                                    selected = false;
                                }
                                treeList.add(new NodeInfo(NodeInfo.LAYOUT,
                                        getLayoutOrPortletName(rootFragment
                                                .getName()), createPath(child
                                                .getPath(), rootFragment
                                                .getId()), returnPath,
                                        depth + 1, true, selected));
                                treeList.addAll(constructTreeNode(rootFragment,
                                        path, returnPath, depth + 2));
                            }
                        } else {
                            treeList.add(new NodeInfo(NodeInfo.PAGE, child
                                    .getName(), child.getPath(), returnPath,
                                    depth, false, false));
                        }
                    }
                }
            } catch (NodeException e) {
                logger.warn("Could not get pages: " + path, e);
            }

            return treeList;
        } else {
            // folder
            List<NodeInfo> treeList = new ArrayList<NodeInfo>();
            try {
                // child folders
                for (Iterator<Folder> ite = parent.getFolders().iterator(); ite
                        .hasNext();) {
                    Folder child = ite.next();
                    if (checkFolderName(child.getName())
                            && PALAdminUtil.checkAccess(child,
                                    JetspeedActions.VIEW)) {
                        if (targetPathName.equals(child.getName())) {
                            List<NodeInfo> childTreeList = constructTreeNode(
                                    child, path, returnPath, depth + 1);
                            if (childTreeList != null) {
                                treeList.add(new NodeInfo(NodeInfo.FOLDER,
                                        child.getName(), child.getPath(),
                                        returnPath, depth, true, true));
                                treeList.addAll(childTreeList);
                            } else {
                                treeList.add(new NodeInfo(NodeInfo.FOLDER,
                                        child.getName(), child.getPath(),
                                        returnPath, depth, true, true));
                            }
                        } else {
                            treeList.add(new NodeInfo(NodeInfo.FOLDER, child
                                    .getName(), child.getPath(), returnPath,
                                    depth, false, false));
                        }
                    }
                }
            } catch (DocumentException e) {
                logger.warn("Could not get folders: " + path, e);
            }

            try {
                // child pages
                for (Iterator<Page> ite = parent.getPages().iterator(); ite
                        .hasNext();) {
                    Page child = ite.next();
                    if (PALAdminUtil.checkAccess(child, JetspeedActions.VIEW)) {
                        treeList.add(new NodeInfo(NodeInfo.PAGE, child
                                .getName(), child.getPath(), returnPath, depth,
                                false, false));
                    }
                }
            } catch (NodeException e) {
                logger.warn("Could not get pages: " + path, e);
            }
            return treeList;
        }
    }

    protected List<NodeInfo> constructTreeNode(Fragment parent, String path,
            String returnPath, int depth) throws CommonException {
        List<NodeInfo> treeList = new ArrayList<NodeInfo>();
        String folderOrPagePath = getFolderOrPagePath(path);
        String fragmentId = getFragmentId(path);
        for (Iterator<Fragment> itr = parent.getFragments().iterator(); itr
                .hasNext();) {
            Fragment child = itr.next();
            String id = child.getId();
            if ("layout".equals(child.getType())) {
                // layout
                if (id.equals(fragmentId)) {
                    treeList.add(new NodeInfo(NodeInfo.LAYOUT,
                            getLayoutOrPortletName(child.getName()),
                            createPath(folderOrPagePath, id), returnPath,
                            depth, false, true));
                } else {
                    treeList.add(new NodeInfo(NodeInfo.LAYOUT,
                            getLayoutOrPortletName(child.getName()),
                            createPath(folderOrPagePath, id), returnPath,
                            depth, false, false));
                }
                treeList.addAll(constructTreeNode(child, path, returnPath,
                        depth + 1));
            } else {
                // portlet
                if (id.equals(fragmentId)) {
                    treeList.add(new NodeInfo(NodeInfo.PORTLET,
                            getLayoutOrPortletName(child.getName()),
                            createPath(folderOrPagePath, id), returnPath,
                            depth, false, true));
                } else {
                    treeList.add(new NodeInfo(NodeInfo.PORTLET,
                            getLayoutOrPortletName(child.getName()),
                            createPath(folderOrPagePath, id), returnPath,
                            depth, false, false));
                }
            }
        }
        return treeList;
    }

    protected String createPath(String folderOrPagePath, String fragmentId) {
        if (fragmentId != null) {
            return folderOrPagePath + PALAdminConstants.SEPARATOR + fragmentId;
        }
        return folderOrPagePath;
    }

    protected String getFolderOrPagePath(String path) {
        if (path == null) {
            return "/";
        }
        int index = path.indexOf(PALAdminConstants.SEPARATOR);
        if (index < 0) {
            return path;
        }
        return path.substring(0, index);
    }

    protected String getFragmentId(String path) {
        if (path == null) {
            return null;
        }
        int index = path.indexOf(PALAdminConstants.SEPARATOR);
        if (index < 0) {
            return null;
        }
        return path.substring(index + 2);
    }

    protected String getLayoutOrPortletName(String name) {
        if (name == null) {
            return null;
        }
        int index = name.indexOf(PALAdminConstants.SEPARATOR);
        if (index < 0) {
            return name;
        }
        return name.substring(index + 2);
    }

    public List<Map<String, String>> createLayoutNamesList() {
        List<Map<String, String>> list = new ArrayList<Map<String, String>>();
        for (Iterator<LayoutInfo> itr = getDecorationFactory().getLayouts(
                getRequestContext()).iterator(); itr.hasNext();) {
            Map<String, String> map = new HashMap<String, String>();
            LayoutInfo value = itr.next();
            map.put("label", value.getDisplayName());
            map.put("value", value.getName());
            list.add(map);
        }
        return list;
    }

    public List<Map<String, String>> createPortletNamesList() {
        List<Map<String, String>> list = new ArrayList<Map<String, String>>();
        Iterator<PortletDefinitionComposite> portlets = getPortletRegistry()
                .getAllPortletDefinitions().iterator();
        while (portlets.hasNext()) {
            PortletDefinitionComposite portlet = (PortletDefinitionComposite) portlets
                    .next();
            if (getSecurityAccessController().checkPortletAccess(portlet,
                    JetspeedActions.MASK_VIEW)) { // check acl
                MutablePortletApplication muta = (MutablePortletApplication) portlet
                        .getPortletApplicationDefinition();
                String appName = muta.getName();
                if (appName == null)
                    continue;
                if (appName.equals("jetspeed-layouts"))
                    continue;

                String uniqueName = appName + "::" + portlet.getName();
                Map<String, String> map = new HashMap<String, String>();
                map.put("label", portlet.getDisplayNameText(getRequestContext()
                        .getLocale()));
                map.put("value", uniqueName);
                list.add(map);
            }
        }
        return list;
    }

    public List<Map<String, String>> createPageDecorationsList() {
        List<Map<String, String>> list = new ArrayList<Map<String, String>>();
        for (Iterator<String> itr = getDecorationFactory().getPageDecorations(
                getRequestContext()).iterator(); itr.hasNext();) {
            Map<String, String> map = new HashMap<String, String>();
            String value = itr.next();
            map.put("label", value);
            map.put("value", value);
            list.add(map);
        }
        return list;
    }

    public List<Map<String, String>> createPortletDecorationsList() {
        List<Map<String, String>> list = new ArrayList<Map<String, String>>();
        for (Iterator<String> itr = getDecorationFactory()
                .getPortletDecorations(getRequestContext()).iterator(); itr
                .hasNext();) {
            Map<String, String> map = new HashMap<String, String>();
            String value = itr.next();
            map.put("label", value);
            map.put("value", value);
            list.add(map);
        }
        return list;
    }

    public List<Map<String, String>> createDesktopPageDecorationsList() {
        List<Map<String, String>> list = new ArrayList<Map<String, String>>();
        for (Iterator<String> itr = getDecorationFactory()
                .getDesktopPageDecorations(getRequestContext()).iterator(); itr
                .hasNext();) {
            Map<String, String> map = new HashMap<String, String>();
            String value = itr.next();
            map.put("label", value);
            map.put("value", value);
            list.add(map);
        }
        return list;
    }

    public List<Map<String, String>> createSecurityConstraintsRefsList()
            throws CommonException {
        List<Map<String, String>> list = new ArrayList<Map<String, String>>();
        try {
            for (Iterator<SecurityConstraintsDef> itr = getPageManager()
                    .getPageSecurity().getSecurityConstraintsDefs().iterator(); itr
                    .hasNext();) {
                Map<String, String> map = new HashMap<String, String>();
                SecurityConstraintsDef value = itr.next();
                map.put("label", value.getName());
                map.put("value", value.getName());
                list.add(map);
            }
        } catch (UnsupportedDocumentTypeException e) {
            logger.error("Could not access a page security. ", e);
            throw new CommonException("could.not.access.page.security",
                    "Could not access a page security. ", e);
        } catch (DocumentNotFoundException e) {
            logger.error("Could not access a page security. ", e);
            throw new CommonException("could.not.access.page.security",
                    "Could not access a page security. ", e);
        } catch (NodeException e) {
            logger.error("Could not access a page security. ", e);
            throw new CommonException("could.not.access.page.security",
                    "Could not access a page security. ", e);
        }
        return list;
    }

    public List<Map<String, String>> createPermissionsList()
            throws CommonException {
        List<Map<String, String>> list = new ArrayList<Map<String, String>>();
        //TODO
        Map<String, String> map = null;

        map = new HashMap<String, String>();
        map.put("label", "View");
        map.put("value", JetspeedActions.VIEW);
        list.add(map);
        map = new HashMap<String, String>();
        map.put("label", "Edit");
        map.put("value", JetspeedActions.EDIT);
        list.add(map);
        map = new HashMap<String, String>();
        map.put("label", "Help");
        map.put("value", JetspeedActions.HELP);
        list.add(map);
        //        map = new HashMap<String, String>();
        //        map.put("label", "Print");
        //        map.put("value", JetspeedActions.PRINT);
        //        list.add(map);

        return list;
    }

    public Map<String, Object> getPageInfo(String path) throws CommonException {
        Map<String, Object> map = new HashMap<String, Object>();

        Page page = getPage(getFolderOrPagePath(path));
        if (page != null) {
            map.put("pageTitle", page.getTitle());
            map.put("pageShortTitle", page.getShortTitle());
            map.put("pageDecorator", page.getDefaultDecorator(Fragment.LAYOUT));
            map.put("portletDecorator", page
                    .getDefaultDecorator(Fragment.PORTLET));
            map.put("desktopTheme", page.getSkin());
            map.put("pageHidden", page.isHidden());
        }
        return map;
    }

    protected Page getPage(String path) throws CommonException {
        if (path == null) {
            // TODO exception?
            return null;
        }

        try {
            Folder folder = getPageManager().getFolder("/");
            String[] names = path.split("/");
            for (int i = 0; i < names.length - 1; i++) {
                if (!"".equals(names[i])) {
                    folder = folder.getFolder(names[i]);
                }
            }
            if (names[names.length - 1].endsWith(".psml")) {
                // page
                return folder.getPage(names[names.length - 1]);
            } else {
                // folder
                return null;
            }
        } catch (PageNotFoundException e) {
            logger.error("Could not access a page: " + path, e);
            throw new CommonException("could.not.access.page",
                    "Could not access a page: " + path, e);
        } catch (FolderNotFoundException e) {
            logger.error("Could not access a page: " + path, e);
            throw new CommonException("could.not.access.page",
                    "Could not access a page: " + path, e);
        } catch (InvalidFolderException e) {
            logger.error("Could not access a page: " + path, e);
            throw new CommonException("could.not.access.page",
                    "Could not access a page: " + path, e);
        } catch (DocumentException e) {
            logger.error("Could not access a page: " + path, e);
            throw new CommonException("could.not.access.page",
                    "Could not access a page: " + path, e);
        } catch (NodeException e) {
            logger.error("Could not access a page: " + path, e);
            throw new CommonException("could.not.access.page",
                    "Could not access a page: " + path, e);
        }
    }

    public void updatePage(String path, String pageTitle,
            String pageShortTitle, String pageDecorator,
            String portletDecorator, String desktopTheme, boolean pageHidden)
            throws CommonException {
        Page page = getPage(getFolderOrPagePath(path));
        if (page == null) {
            throw new CommonException("could.not.find.page",
                    "Could not find a page: " + path);
        }
        page.setTitle(pageTitle);
        page.setShortTitle(pageShortTitle);
        page.setDefaultDecorator(pageDecorator, Fragment.LAYOUT);
        page.setDefaultDecorator(portletDecorator, Fragment.PORTLET);
        page.setSkin(desktopTheme);
        page.setHidden(pageHidden);
        try {
            getPageManager().updatePage(page);
            getPageManager().reset();
        } catch (PageNotUpdatedException e) {
            throw new CommonException("could.not.update.page",
                    "Could not update a page: " + path, e);
        } catch (NodeException e) {
            throw new CommonException("could.not.update.page",
                    "Could not update a page: " + path, e);
        }
    }

    public String deletePage(String path) throws CommonException {
        Page page = getPage(getFolderOrPagePath(path));
        if (page == null) {
            throw new CommonException("could.not.find.page",
                    "Could not find a page: " + path);
        }
        Node parent = page.getParent();
        String parentPath = "/";
        if (parent != null) {
            parentPath = parent.getPath();
        }

        try {
            getPageManager().removePage(page);
            getPageManager().reset();
        } catch (PageNotUpdatedException e) {
            throw new CommonException("could.not.delete.page",
                    "Could not delete a page: " + path, e);
        } catch (NodeException e) {
            throw new CommonException("could.not.delete.page",
                    "Could not delete a page: " + path, e);
        }
        return parentPath;
    }

    public void addPage(String path, String pageName, String pageTitle,
            String pageShortTitle, String layoutName, String pageDecorator,
            String portletDecorator, String desktopTheme, boolean pageHidden)
            throws CommonException {
        Folder folder = getFolder(getFolderOrPagePath(path));
        if (folder == null) {
            throw new CommonException("could.not.find.folder",
                    "Could not find a folder: " + path);
        }

        // add page
        String pagePath = folder.getPath();
        if (pagePath.endsWith(Folder.PATH_SEPARATOR)) {
            pagePath = pagePath + pageName;
        } else {
            pagePath = pagePath + Folder.PATH_SEPARATOR + pageName;
        }
        Page childPage = getPageManager().newPage(pagePath);
        childPage.getRootFragment().setName(layoutName);
        childPage.setDefaultDecorator(pageDecorator, Fragment.LAYOUT);
        childPage.setDefaultDecorator(portletDecorator, Fragment.PORTLET);
        childPage.setTitle(pageTitle);
        childPage.setShortTitle(pageShortTitle);
        childPage.setSkin(desktopTheme);
        childPage.setHidden(pageHidden);
        try {
            getPageManager().updatePage(childPage);
            getPageManager().reset();
        } catch (PageNotUpdatedException e) {
            throw new CommonException("could.not.add.page",
                    "Could not add a page: " + path, e);
        } catch (NodeException e) {
            throw new CommonException("could.not.add.page",
                    "Could not add a page: " + path, e);
        }

        List<String> orderList = folder.getDocumentOrder();
        if (orderList != null) {
            String name = childPage.getName();
            if (orderList.indexOf(name) < 0) {
                orderList.add(name);
                folder.setDocumentOrder(orderList);
                try {
                    getPageManager().updateFolder(folder);
                    getPageManager().reset();
                } catch (FolderNotUpdatedException e) {
                    throw new CommonException("could.not.change.page.order",
                            "Could not change a page order: " + path, e);
                } catch (NodeException e) {
                    throw new CommonException("could.not.change.page.order",
                            "Could not change a page order: " + path, e);
                }
            }
        }

    }

    public Map<String, Object> getFolderInfo(String path)
            throws CommonException {
        Map<String, Object> map = new HashMap<String, Object>();

        Folder folder = getFolder(getFolderOrPagePath(path));
        if (folder != null) {
            map.put("folderTitle", folder.getTitle());
            map.put("folderShortTitle", folder.getShortTitle());
            map.put("pageDecorator", folder
                    .getDefaultDecorator(Fragment.LAYOUT));
            map.put("portletDecorator", folder
                    .getDefaultDecorator(Fragment.PORTLET));
            map.put("desktopTheme", folder.getSkin());
            map.put("folderHidden", folder.isHidden());
        }
        return map;
    }

    protected Folder getFolder(String path) throws CommonException {
        if (path == null) {
            // TODO exception?
            return null;
        }

        try {
            Folder folder = getPageManager().getFolder("/");
            String[] names = path.split("/");
            if (names.length == 0) {
                return folder;
            }
            for (int i = 0; i < names.length - 1; i++) {
                if (!"".equals(names[i])) {
                    folder = folder.getFolder(names[i]);
                }
            }
            if (names[names.length - 1].endsWith(".psml")) {
                // page
                return null;
            } else {
                // folder
                return folder.getFolder(names[names.length - 1]);
            }
        } catch (FolderNotFoundException e) {
            logger.error("Could not access a folder: " + path, e);
            throw new CommonException("could.not.access.folder",
                    "Could not access a folder: " + path, e);
        } catch (InvalidFolderException e) {
            logger.error("Could not access a folder: " + path, e);
            throw new CommonException("could.not.access.folder",
                    "Could not access a folder: " + path, e);
        } catch (DocumentException e) {
            logger.error("Could not access a folder: " + path, e);
            throw new CommonException("could.not.access.folder",
                    "Could not access a folder: " + path, e);
        } catch (NodeException e) {
            logger.error("Could not access a folder: " + path, e);
            throw new CommonException("could.not.access.folder",
                    "Could not access a folder: " + path, e);
        }
    }

    public void updateFolder(String path, String pageTitle,
            String pageShortTitle, String pageDecorator,
            String portletDecorator, String desktopTheme, boolean pageHidden)
            throws CommonException {
        Folder folder = getFolder(getFolderOrPagePath(path));
        if (folder == null) {
            throw new CommonException("could.not.find.folder",
                    "Could not find a folder: " + path);
        }
        folder.setTitle(pageTitle);
        folder.setShortTitle(pageShortTitle);
        folder.setDefaultDecorator(pageDecorator, Fragment.LAYOUT);
        folder.setDefaultDecorator(portletDecorator, Fragment.PORTLET);
        folder.setSkin(desktopTheme);
        folder.setHidden(pageHidden);
        try {
            getPageManager().updateFolder(folder);
            getPageManager().reset();
        } catch (PageNotUpdatedException e) {
            throw new CommonException("could.not.update.folder",
                    "Could not update a folder: " + path, e);
        } catch (NodeException e) {
            throw new CommonException("could.not.update.folder",
                    "Could not update a folder: " + path, e);
        }
    }

    public String deleteFolder(String path) throws CommonException {
        if ("/".equals(path)) {
            throw new CommonException("could.not.delete.root.folder",
                    "Could not find a folder: " + path);
        }
        Folder folder = getFolder(getFolderOrPagePath(path));
        if (folder == null) {
            throw new CommonException("could.not.find.folder",
                    "Could not find a folder: " + path);
        }
        Node parent = folder.getParent();
        String parentPath = "/";
        if (parent != null) {
            parentPath = parent.getPath();
        }

        try {
            getPageManager().removeFolder(folder);
            getPageManager().reset();
        } catch (PageNotUpdatedException e) {
            throw new CommonException("could.not.delete.folder",
                    "Could not update a folder: " + path, e);
        } catch (NodeException e) {
            throw new CommonException("could.not.delete.folder",
                    "Could not update a folder: " + path, e);
        }

        return parentPath;
    }

    public void addFolder(String path, String folderName, String folderTitle,
            String folderShortTitle, String layoutName, String pageDecorator,
            String portletDecorator, String desktopTheme, boolean folderHidden)
            throws CommonException {
        Folder folder = getFolder(getFolderOrPagePath(path));
        if (folder == null) {
            throw new CommonException("could.not.find.folder",
                    "Could not find a folder: " + path);
        }

        String folderPath = folder.getPath();
        if (folderPath.endsWith(Folder.PATH_SEPARATOR)) {
            folderPath = folderPath + folderName;
        } else {
            folderPath = folderPath + Folder.PATH_SEPARATOR + folderName;
        }
        Folder child = getPageManager().newFolder(folderPath);
        child.setTitle(folderTitle);
        child.setShortTitle(folderShortTitle);
        child.setDefaultDecorator(pageDecorator, Fragment.LAYOUT);
        child.setDefaultDecorator(portletDecorator, Fragment.PORTLET);
        child.setSkin(desktopTheme);
        child.setHidden(folderHidden);
        try {
            getPageManager().updateFolder(child);
            getPageManager().reset();
        } catch (PageNotUpdatedException e) {
            throw new CommonException("could.not.add.folder",
                    "Could not add a folder: " + path, e);
        } catch (NodeException e) {
            throw new CommonException("could.not.add.folder",
                    "Could not add a folder: " + path, e);
        }

        // change a folder order
        List<String> orderList = folder.getDocumentOrder();
        if (orderList != null) {
            String name = child.getName();
            if (orderList.indexOf(name) < 0) {
                orderList.add(name);
                folder.setDocumentOrder(orderList);
                try {
                    getPageManager().updateFolder(folder);
                    getPageManager().reset();
                } catch (FolderNotUpdatedException e) {
                    throw new CommonException("could.not.change.folder.order",
                            "Could not change a folder order: " + path, e);
                } catch (NodeException e) {
                    throw new CommonException("could.not.change.folder.order",
                            "Could not change a folder order: " + path, e);
                }
            }
        }

        // add default childPage
        String pagePath = child.getPath();
        if (pagePath.endsWith(Folder.PATH_SEPARATOR)) {
            pagePath = pagePath + "default-page";
        } else {
            pagePath = pagePath + Folder.PATH_SEPARATOR + "default-page";
        }
        Page childPage = getPageManager().newPage(pagePath);
        childPage.getRootFragment().setName(layoutName);
        childPage.setDefaultDecorator(pageDecorator, Fragment.LAYOUT);
        childPage.setDefaultDecorator(portletDecorator, Fragment.PORTLET);
        childPage.setTitle(folderTitle);
        childPage.setShortTitle(folderShortTitle);
        childPage.setSkin(desktopTheme);
        childPage.setHidden(folderHidden);
        try {
            getPageManager().updatePage(childPage);
            getPageManager().reset();
        } catch (PageNotUpdatedException e) {
            throw new CommonException("could.not.add.default.page",
                    "Could not add a default page: " + path, e);
        } catch (NodeException e) {
            throw new CommonException("could.not.add.default.page",
                    "Could not add a default page: " + path, e);
        }

        orderList = child.getDocumentOrder();
        if (orderList != null) {
            String name = childPage.getName();
            if (orderList.indexOf(name) < 0) {
                orderList.add(name);
                child.setDocumentOrder(orderList);
                try {
                    getPageManager().updateFolder(child);
                    getPageManager().reset();
                } catch (FolderNotUpdatedException e) {
                    throw new CommonException(
                            "could.not.change.default.page.order",
                            "Could not change a default page order: " + path, e);
                } catch (NodeException e) {
                    throw new CommonException(
                            "could.not.change.default.page.order",
                            "Could not change a default page order: " + path, e);
                }
            }
        }

    }

    public Map<String, Object> getLayoutInfo(String path)
            throws CommonException {
        Map<String, Object> map = new HashMap<String, Object>();

        Page page = getPage(getFolderOrPagePath(path));
        if (page != null) {
            String fid = getFragmentId(path);
            Fragment fragment = page.getFragmentById(fid);
            if (fragment != null) {
                map.put("fragmentId", fragment.getId());
                map.put("layoutName", fragment.getName());
                map.put("decorator", fragment.getDecorator());
                map.put("desktopTheme", fragment.getSkin());
            } else {
                //TODO throw exception
            }
        }
        return map;
    }

    public void updateLayout(String path, String layoutName, String decorator,
            String desktopTheme) throws CommonException {
        Page page = getPage(getFolderOrPagePath(path));
        if (page == null) {
            throw new CommonException("could.not.find.layout",
                    "Could not find a page: " + path);
        }

        String fid = getFragmentId(path);
        if (fid == null || fid.equals("")) {
            throw new CommonException("could.not.find.layout",
                    "Could not find a page: " + path);
        }

        Fragment fragment = page.getFragmentById(fid);
        fragment.setName(layoutName);
        fragment.setDecorator(decorator);
        fragment.setSkin(desktopTheme);
        try {
            getPageManager().updatePage(page);
            getPageManager().reset();
        } catch (PageNotUpdatedException e) {
            throw new CommonException("could.not.update.layout",
                    "Could not update a layout: " + path, e);
        } catch (NodeException e) {
            throw new CommonException("could.not.update.layout",
                    "Could not update a layout: " + path, e);
        }

        try {
            ContentPage cpage = getPageManager().getContentPage(page.getPath());
            ContentFragment cfragment = cpage.getContentFragmentById(fid);
            PortletWindow portletWindow = getPortletWindowAccessor()
                    .getPortletWindow(cfragment);
            getPortletEntityAccessComponent().updatePortletEntity(
                    portletWindow.getPortletEntity(), cfragment);
            getPortletEntityAccessComponent().storePortletEntity(
                    portletWindow.getPortletEntity());

            getPortletWindowAccessor().createPortletWindow(
                    portletWindow.getPortletEntity(), cfragment.getId());
        } catch (PageNotFoundException e) {
            throw new CommonException(
                    "updated.layout.but.could.not.update.cache",
                    "Update a layout configuration, but could not update a cache.: "
                            + path, e);
        } catch (NodeException e) {
            throw new CommonException(
                    "updated.layout.but.could.not.update.cache",
                    "Update a layout configuration, but could not update a cache.: "
                            + path, e);
        } catch (FailedToRetrievePortletWindow e) {
            throw new CommonException(
                    "updated.layout.but.could.not.update.cache",
                    "Update a layout configuration, but could not update a cache.: "
                            + path, e);
        } catch (PortletEntityNotStoredException e) {
            throw new CommonException(
                    "updated.layout.but.could.not.update.cache",
                    "Update a layout configuration, but could not update a cache.: "
                            + path, e);
        }

    }

    public String deleteLayout(String path) throws CommonException {
        String p = getFolderOrPagePath(path);
        Page page = getPage(p);
        if (page == null) {
            throw new CommonException("could.not.find.layout",
                    "Could not find a page: " + path);
        }

        String fid = getFragmentId(path);
        if (fid == null || fid.equals("")) {
            throw new CommonException("could.not.find.layout",
                    "Could not find a page: " + path);
        }

        Fragment rootFragment = page.getRootFragment();
        if (rootFragment != null && fid.equals(rootFragment.getId())) {
            throw new CommonException("could.not.delete.root.layout",
                    "Could not delete a root layout: " + path);
        }

        page.removeFragmentById(fid);

        try {
            getPageManager().updatePage(page);
            getPageManager().reset();
        } catch (PageNotUpdatedException e) {
            throw new CommonException("could.not.delete.layout",
                    "Could not delete a layout: " + path, e);
        } catch (NodeException e) {
            throw new CommonException("could.not.delete.layout",
                    "Could not delete a layout: " + path, e);
        }
        return p;
    }

    public Map<String, Object> getPortletInfo(String path)
            throws CommonException {
        Map<String, Object> map = new HashMap<String, Object>();

        Page page = getPage(getFolderOrPagePath(path));
        if (page != null) {
            String fid = getFragmentId(path);
            Fragment fragment = page.getFragmentById(fid);
            if (fragment != null) {
                map.put("fragmentId", fragment.getId());
                map.put("portletName", fragment.getName());
                map.put("decorator", fragment.getDecorator());
                map.put("desktopTheme", fragment.getSkin());
            } else {
                //TODO throw exception
            }
        }
        return map;
    }

    public void updatePortlet(String path, String portletName,
            String decorator, String desktopTheme) throws CommonException {
        Page page = getPage(getFolderOrPagePath(path));
        if (page == null) {
            throw new CommonException("could.not.find.portlet",
                    "Could not find a page: " + path);
        }
        String fid = getFragmentId(path);
        Fragment fragment = page.getFragmentById(fid);
        fragment.setName(portletName);
        fragment.setDecorator(decorator);
        fragment.setSkin(desktopTheme);
        try {
            getPageManager().updatePage(page);
            getPageManager().reset();
        } catch (PageNotUpdatedException e) {
            throw new CommonException("could.not.update.portlet",
                    "Could not update a page: " + path, e);
        } catch (NodeException e) {
            throw new CommonException("could.not.update.portlet",
                    "Could not update a page: " + path, e);
        }

        try {
            ContentPage cpage = getPageManager().getContentPage(page.getPath());
            ContentFragment cfragment = cpage.getContentFragmentById(fid);
            PortletWindow portletWindow = getPortletWindowAccessor()
                    .getPortletWindow(cfragment);
            getPortletEntityAccessComponent().updatePortletEntity(
                    portletWindow.getPortletEntity(), cfragment);
            getPortletEntityAccessComponent().storePortletEntity(
                    portletWindow.getPortletEntity());

            getPortletWindowAccessor().createPortletWindow(
                    portletWindow.getPortletEntity(), cfragment.getId());
        } catch (PageNotFoundException e) {
            throw new CommonException(
                    "updated.portlet.but.could.not.update.cache",
                    "Update a portlet configuration, but could not update a cache.: "
                            + path, e);
        } catch (NodeException e) {
            throw new CommonException(
                    "updated.portlet.but.could.not.update.cache",
                    "Update a portlet configuration, but could not update a cache.: "
                            + path, e);
        } catch (FailedToRetrievePortletWindow e) {
            throw new CommonException(
                    "updated.portlet.but.could.not.update.cache",
                    "Update a portlet configuration, but could not update a cache.: "
                            + path, e);
        } catch (PortletEntityNotStoredException e) {
            throw new CommonException(
                    "updated.portlet.but.could.not.update.cache",
                    "Update a portlet configuration, but could not update a cache.: "
                            + path, e);
        }

    }

    public String deletePortlet(String path) throws CommonException {
        String p = getFolderOrPagePath(path);
        Page page = getPage(p);
        if (page == null) {
            throw new CommonException("could.not.find.portlet",
                    "Could not find a page: " + path);
        }

        String fid = getFragmentId(path);
        if (fid == null || fid.equals("")) {
            throw new CommonException("could.not.find.layout",
                    "Could not find a page: " + path);
        }

        page.removeFragmentById(fid);

        try {
            getPageManager().updatePage(page);
            getPageManager().reset();
        } catch (PageNotUpdatedException e) {
            throw new CommonException("could.not.delete.portlet",
                    "Could not delete a portlet: " + path, e);
        } catch (NodeException e) {
            throw new CommonException("could.not.delete.portlet",
                    "Could not delete a portlet: " + path, e);
        }
        return p;
    }

    public void addLayout(String path, String layoutName)
            throws CommonException {
        Page page = getPage(getFolderOrPagePath(path));
        if (page == null) {
            throw new CommonException("could.not.find.layout",
                    "Could not find a page: " + path);
        }
        String fid = getFragmentId(path);
        Fragment fragment = page.getFragmentById(fid);

        Fragment child = getPageManager().newFragment();
        child.setName(layoutName);
        child.setDecorator(fragment.getDecorator());
        child.setSkin(fragment.getSkin());
        child.setType("layout");

        fragment.getFragments().add(child);

        try {
            getPageManager().updatePage(page);
            getPageManager().reset();
        } catch (PageNotUpdatedException e) {
            throw new CommonException("could.not.add.layout",
                    "Could not update a page: " + path, e);
        } catch (NodeException e) {
            throw new CommonException("could.not.add.layout",
                    "Could not update a page: " + path, e);
        }
    }

    public void addPortlet(String path, String portletName)
            throws CommonException {
        Page page = getPage(getFolderOrPagePath(path));
        if (page == null) {
            throw new CommonException("could.not.find.page",
                    "Could not find a page: " + path);
        }
        String fid = getFragmentId(path);
        Fragment fragment = page.getFragmentById(fid);

        Fragment child = getPageManager().newFragment();
        child.setName(portletName);
        child.setDecorator(fragment.getDecorator());
        child.setSkin(fragment.getSkin());
        child.setType("portlet");

        fragment.getFragments().add(child);

        try {
            getPageManager().updatePage(page);
            getPageManager().reset();
        } catch (PageNotUpdatedException e) {
            throw new CommonException("could.not.add.portlet",
                    "Could not update a page: " + path, e);
        } catch (NodeException e) {
            throw new CommonException("could.not.add.portlet",
                    "Could not update a page: " + path, e);
        }
    }

    public List<Map<String, Object>> getSecurityConstraints(String path)
            throws CommonException {
        String folderOrPage = getFolderOrPagePath(path);

        List<Map<String, Object>> list = new ArrayList<Map<String, Object>>();
        if (folderOrPage.endsWith(".psml")) {
            // page
            Page page = getPage(folderOrPage);
            SecurityConstraints constraints = page.getSecurityConstraints();
            if (constraints != null) {
                List<SecurityConstraint> scs = constraints
                        .getSecurityConstraints();
                for (int i = 0; i < scs.size(); i++) {
                    SecurityConstraint constraint = scs.get(i);
                    Map<String, Object> map = new HashMap<String, Object>();

                    List<String> users = constraint.getUsers();
                    List<String> groups = constraint.getGroups();
                    List<String> roles = constraint.getRoles();

                    StringBuffer cname = new StringBuffer();
                    if (users != null && !users.isEmpty()) {
                        cname.append("users(").append(
                                PALAdminUtil.formatCSVList(users)).append(") ");
                    }
                    if (groups != null && !groups.isEmpty()) {
                        cname.append("groups(").append(
                                PALAdminUtil.formatCSVList(groups))
                                .append(") ");
                    }
                    if (roles != null && !roles.isEmpty()) {
                        cname.append("roles(").append(
                                PALAdminUtil.formatCSVList(roles)).append(") ");
                    }
                    map.put(PALAdminConstants.SECURITY_CONSTRAINT_NAME, cname
                            .toString());

                    List<String> permissions = constraint.getPermissions();
                    map.put(PALAdminConstants.SECURITY_CONSTRAINT_PERMISSION,
                            PALAdminUtil.formatCSVList(permissions));

                    map.put(PALAdminConstants.SECURITY_CONSTRAINT_TYPE,
                            PALAdminConstants.SECURITY_CONSTRAINT);
                    map.put(PALAdminConstants.SECURITY_CONSTRAINT_ORDER, i);
                    list.add(map);
                }

                List<String> scrs = constraints.getSecurityConstraintsRefs();
                for (int i = 0; i < scrs.size(); i++) {
                    Map<String, Object> map = new HashMap<String, Object>();
                    map.put(PALAdminConstants.SECURITY_CONSTRAINT_NAME, scrs
                            .get(i));
                    map.put(PALAdminConstants.SECURITY_CONSTRAINT_TYPE,
                            PALAdminConstants.SECURITY_CONSTRAINTS_REF);
                    map.put(PALAdminConstants.SECURITY_CONSTRAINT_ORDER, i);
                    list.add(map);
                }
            }
        } else {
            // folder
            Folder folder = getFolder(folderOrPage);
            SecurityConstraints constraints = folder.getSecurityConstraints();
            if (constraints != null) {
                List<SecurityConstraint> scs = constraints
                        .getSecurityConstraints();
                for (int i = 0; i < scs.size(); i++) {
                    SecurityConstraint constraint = scs.get(i);
                    Map<String, Object> map = new HashMap<String, Object>();

                    List<String> users = constraint.getUsers();
                    List<String> groups = constraint.getGroups();
                    List<String> roles = constraint.getRoles();

                    StringBuffer cname = new StringBuffer();
                    if (users != null && !users.isEmpty()) {
                        cname.append("users(").append(
                                PALAdminUtil.formatCSVList(users)).append(") ");
                    }
                    if (groups != null && !groups.isEmpty()) {
                        cname.append("groups(").append(
                                PALAdminUtil.formatCSVList(groups))
                                .append(") ");
                    }
                    if (roles != null && !roles.isEmpty()) {
                        cname.append("roles(").append(
                                PALAdminUtil.formatCSVList(roles)).append(") ");
                    }
                    map.put(PALAdminConstants.SECURITY_CONSTRAINT_NAME, cname
                            .toString());

                    List<String> permissions = constraint.getPermissions();
                    map.put(PALAdminConstants.SECURITY_CONSTRAINT_PERMISSION,
                            PALAdminUtil.formatCSVList(permissions));

                    map.put(PALAdminConstants.SECURITY_CONSTRAINT_TYPE,
                            PALAdminConstants.SECURITY_CONSTRAINT);
                    map.put(PALAdminConstants.SECURITY_CONSTRAINT_ORDER, i);
                    list.add(map);
                }

                List<String> scrs = constraints.getSecurityConstraintsRefs();
                for (int i = 0; i < scrs.size(); i++) {
                    Map<String, Object> map = new HashMap<String, Object>();
                    map.put(PALAdminConstants.SECURITY_CONSTRAINT_NAME, scrs
                            .get(i));
                    map.put(PALAdminConstants.SECURITY_CONSTRAINT_TYPE,
                            PALAdminConstants.SECURITY_CONSTRAINTS_REF);
                    map.put(PALAdminConstants.SECURITY_CONSTRAINT_ORDER, i);
                    list.add(map);
                }
            }

        }
        return list;
    }

    public void addConstraint(String path, List<String> users,
            List<String> roles, List<String> groups, List<String> permissions)
            throws CommonException {
        String folderOrPage = getFolderOrPagePath(path);
        if (folderOrPage == null) {
            throw new CommonException("invalid.path", "Invalid path: " + path);
        }
        SecurityConstraint sc = getPageManager().newPageSecurityConstraint();
        sc.setUsers(users);
        sc.setRoles(roles);
        sc.setGroups(groups);
        sc.setPermissions(permissions);
        if (folderOrPage.endsWith(".psml")) {
            // page
            Page page = getPage(folderOrPage);
            if (page == null) {
                throw new CommonException("could.not.find.page",
                        "Could not find a page: " + path);
            }
            addConstraint(page, sc);
        } else {
            // folder
            Folder folder = getFolder(folderOrPage);
            if (folder == null) {
                throw new CommonException("could.not.find.folder",
                        "Could not find a folder: " + path);
            }
            addConstraint(folder, sc);
        }
    }

    public void addConstraint(Page page, SecurityConstraint sc)
            throws CommonException {
        if (page.getSecurityConstraints() == null) {
            page.setSecurityConstraints(getPageManager()
                    .newSecurityConstraints());
        }
        page.getSecurityConstraints().getSecurityConstraints().add(sc);

        try {
            getPageManager().updatePage(page);
            getPageManager().reset();
        } catch (PageNotUpdatedException e) {
            throw new CommonException("could.not.update.page",
                    "Could not update a page: " + page.getPath(), e);
        } catch (NodeException e) {
            throw new CommonException("could.not.update.page",
                    "Could not update a page: " + page.getPath(), e);
        }
    }

    public void addConstraint(Folder folder, SecurityConstraint sc)
            throws CommonException {
        if (folder.getSecurityConstraints() == null) {
            folder.setSecurityConstraints(getPageManager()
                    .newSecurityConstraints());
        }
        folder.getSecurityConstraints().getSecurityConstraints().add(sc);

        try {
            getPageManager().updateFolder(folder);
            getPageManager().reset();
        } catch (FolderNotUpdatedException e) {
            throw new CommonException("could.not.update.folder",
                    "Could not update a folder: " + folder.getPath(), e);
        } catch (NodeException e) {
            throw new CommonException("could.not.update.folder",
                    "Could not update a folder: " + folder.getPath(), e);
        }
    }

    public void addConstraintRef(String path, String constraintRef)
            throws CommonException {
        String folderOrPage = getFolderOrPagePath(path);
        if (folderOrPage == null) {
            throw new CommonException("invalid.path", "Invalid path: " + path);
        }

        if (folderOrPage.endsWith(".psml")) {
            // page
            Page page = getPage(folderOrPage);
            if (page == null) {
                throw new CommonException("could.not.find.page",
                        "Could not find a page: " + path);
            }
            addConstraintRef(page, constraintRef);
        } else {
            // folder
            Folder folder = getFolder(folderOrPage);
            if (folder == null) {
                throw new CommonException("could.not.find.folder",
                        "Could not find a folder: " + path);
            }
            addConstraintRef(folder, constraintRef);
        }
    }

    public void addConstraintRef(Page page, String constraintRef)
            throws CommonException {

        if (page.getSecurityConstraints() == null) {
            page.setSecurityConstraints(getPageManager()
                    .newSecurityConstraints());
        }
        page.getSecurityConstraints().getSecurityConstraintsRefs().add(
                constraintRef);

        try {
            getPageManager().updatePage(page);
            getPageManager().reset();
        } catch (PageNotUpdatedException e) {
            throw new CommonException("could.not.update.page",
                    "Could not update a page: " + page.getPath(), e);
        } catch (NodeException e) {
            throw new CommonException("could.not.update.page",
                    "Could not update a page: " + page.getPath(), e);
        }
    }

    public void addConstraintRef(Folder folder, String constraintRef)
            throws CommonException {

        if (folder.getSecurityConstraints() == null) {
            folder.setSecurityConstraints(getPageManager()
                    .newSecurityConstraints());
        }
        folder.getSecurityConstraints().getSecurityConstraintsRefs().add(
                constraintRef);

        try {
            getPageManager().updateFolder(folder);
            getPageManager().reset();
        } catch (FolderNotUpdatedException e) {
            throw new CommonException("could.not.update.folder",
                    "Could not update a folder: " + folder.getPath(), e);
        } catch (NodeException e) {
            throw new CommonException("could.not.update.folder",
                    "Could not update a folder: " + folder.getPath(), e);
        }
    }

    public void removeConstraint(String path, int applyOrder)
            throws CommonException {
        String folderOrPage = getFolderOrPagePath(path);
        if (folderOrPage == null) {
            throw new CommonException("invalid.path", "Invalid path: " + path);
        }

        if (folderOrPage.endsWith(".psml")) {
            // page
            Page page = getPage(folderOrPage);
            if (page == null) {
                throw new CommonException("could.not.find.page",
                        "Could not find a page: " + path);
            }
            removeConstraint(page, applyOrder);
        } else {
            // folder
            Folder folder = getFolder(folderOrPage);
            if (folder == null) {
                throw new CommonException("could.not.find.folder",
                        "Could not find a folder: " + path);
            }
            removeConstraint(folder, applyOrder);
        }
    }

    public void removeConstraint(Page page, int applyOrder)
            throws CommonException {

        if (page.getSecurityConstraints() == null) {
            throw new CommonException("could.not.find.page.security",
                    "Could not find a page security: " + page.getPath());
        }
        page.getSecurityConstraints().getSecurityConstraints().remove(
                applyOrder);

        try {
            getPageManager().updatePage(page);
            getPageManager().reset();
        } catch (PageNotUpdatedException e) {
            throw new CommonException("could.not.update.page",
                    "Could not update a page: " + page.getPath(), e);
        } catch (NodeException e) {
            throw new CommonException("could.not.update.page",
                    "Could not update a page: " + page.getPath(), e);
        }
    }

    public void removeConstraint(Folder folder, int applyOrder)
            throws CommonException {

        if (folder.getSecurityConstraints() == null) {
            throw new CommonException("could.not.find.folder.security",
                    "Could not find a folder security: " + folder.getPath());
        }
        folder.getSecurityConstraints().getSecurityConstraints().remove(
                applyOrder);

        try {
            getPageManager().updateFolder(folder);
            getPageManager().reset();
        } catch (FolderNotUpdatedException e) {
            throw new CommonException("could.not.update.folder",
                    "Could not update a folder: " + folder.getPath(), e);
        } catch (NodeException e) {
            throw new CommonException("could.not.update.folder",
                    "Could not update a folder: " + folder.getPath(), e);
        }
    }

    public void removeConstraintRef(String path, int applyOrder)
            throws CommonException {
        String folderOrPage = getFolderOrPagePath(path);
        if (folderOrPage == null) {
            throw new CommonException("invalid.path", "Invalid path: " + path);
        }

        if (folderOrPage.endsWith(".psml")) {
            // page
            Page page = getPage(folderOrPage);
            if (page == null) {
                throw new CommonException("could.not.find.page",
                        "Could not find a page: " + path);
            }
            removeConstraintRef(page, applyOrder);
        } else {
            // folder
            Folder folder = getFolder(folderOrPage);
            if (folder == null) {
                throw new CommonException("could.not.find.folder",
                        "Could not find a folder: " + path);
            }
            removeConstraintRef(folder, applyOrder);
        }
    }

    public void removeConstraintRef(Page page, int applyOrder)
            throws CommonException {
        if (page.getSecurityConstraints() == null) {
            throw new CommonException("could.not.find.page.security",
                    "Could not find a page security: " + page.getPath());
        }
        page.getSecurityConstraints().getSecurityConstraintsRefs().remove(
                applyOrder);

        try {
            getPageManager().updatePage(page);
            getPageManager().reset();
        } catch (PageNotUpdatedException e) {
            throw new CommonException("could.not.update.page",
                    "Could not update a page: " + page.getPath(), e);
        } catch (NodeException e) {
            throw new CommonException("could.not.update.page",
                    "Could not update a page: " + page.getPath(), e);
        }
    }

    public void removeConstraintRef(Folder folder, int applyOrder)
            throws CommonException {
        if (folder.getSecurityConstraints() == null) {
            throw new CommonException("could.not.find.folder.security",
                    "Could not find a folder security: " + folder.getPath());
        }
        folder.getSecurityConstraints().getSecurityConstraintsRefs().remove(
                applyOrder);

        try {
            getPageManager().updateFolder(folder);
            getPageManager().reset();
        } catch (FolderNotUpdatedException e) {
            throw new CommonException("could.not.update.folder",
                    "Could not update a folder: " + folder.getPath(), e);
        } catch (NodeException e) {
            throw new CommonException("could.not.update.folder",
                    "Could not update a folder: " + folder.getPath(), e);
        }
    }

    public boolean checkFolderOrPageViewAccess(String path)
            throws CommonException {
        return checkFolderOrPageAccess(path, JetspeedActions.VIEW);
    }

    public boolean checkFolderOrPageEditAccess(String path)
            throws CommonException {
        return checkFolderOrPageAccess(path, JetspeedActions.EDIT);
    }

    public boolean checkFolderOrPageAccess(String path, String actions)
            throws CommonException {
        String folderOrPagePath = getFolderOrPagePath(path);
        if (folderOrPagePath.endsWith(".psml")) {
            // page
            Page page = getPage(folderOrPagePath);
            return PALAdminUtil.checkAccess(page, actions);
        } else {
            //folder
            Folder folder = getFolder(folderOrPagePath);
            return PALAdminUtil.checkAccess(folder, actions);
        }
    }

    public int getNodeInfoType(String path) {
        String fragmentId = getFragmentId(path);
        if (fragmentId == null || fragmentId.equals("")) {
            String folderOrPage = getFolderOrPagePath(path);
            if (folderOrPage.endsWith(".psml")) {
                return NodeInfo.PAGE;
            } else {
                return NodeInfo.FOLDER;
            }
        }
        // TODO layout or portlet
        return NodeInfo.LAYOUT;
    }
}
