/***********************************************************************
 * Copyright(C) 2006 Valtech Co.,Ltd.
 * All Rights Reserved. This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.opensource.org/licenses/cpl.php
 ***********************************************************************/
package jp.valtech.bts.data;

import java.util.ArrayList;
import java.util.Iterator;

import jp.valtech.bts.ui.BtsPlugin;
import jp.valtech.bts.ui.navigator.IProjectDialogSettings;
import jp.valtech.bts.ui.navigator.ProjectConfig;
import jp.valtech.bts.ui.navigator.ProjectConfigManager;
import jp.valtech.bts.util.Logging;

import org.eclipse.jface.action.Action;

/**
 * カレントプロジェクトの情報を管理するクラスです。
 * 「カレントプロジェクト」とは現在オープン中のプロジェクトのことです。<br>
 * オープンできるプロジェクト数は「１」が上限です。<br>
 * 
 * @author		<A href="mailto:m_sugitou@valtech.jp">M.Sugito</A>
 * @version	Ver.0.8
 */
public class CurrentProject implements Logging{

	/** ファイルシステムのファイル区切り文字列 */
	private static final String SEPARATOR	= System.getProperty( "file.separator" );

	/** 当クラスインスタンス。の唯一のインスタンスです。 */
	private static CurrentProject instance;

	/** カレントプロジェクトのコンフィグ情報です。 */
	private ProjectConfig currentConfig = new ProjectConfig();

	/** カレントプロジェクトがあるかどうかを示すフラグです。 */
	private boolean open = false;

	/** アクションクラス：カレントプロジェクトの有無で活性／非活性を変えるアクション*/
	private ArrayList actions = new ArrayList();
	
	/**
	 * 外部からは呼ばれません。
	 */
	private CurrentProject() {
	}
	
	
	/**
	 * 唯一のインスタンスを返します。
	 * 
	 * @return		唯一のインスタンス。
	 */
	public static synchronized CurrentProject getInsance() {
		if(instance ==null) {
			instance = new CurrentProject();
		}
		return instance;
	}
	

	/**
	 * カレントプロジェクトを設定します。
	 * 指定のSSIDを持つプロジェクトをカレントプロジェクトに設定します。
	 * 
	 * @param		ssid			SSID
	 */
	public void setCurrentProject(String ssid) {

		IProjectDialogSettings setting = 
			BtsPlugin.getInstance().getProjectDialogSettings().getSection(ssid);
			
		// プロジェクト名
		currentConfig.setProjectName		( setting.get(ProjectConfigManager.KEY_PROJECT) );
		// ユーザ名
		currentConfig.setUserName			( setting.get(ProjectConfigManager.KEY_USER) );
		// SSID
		currentConfig.setSsid				( setting.get(ProjectConfigManager.KEY_SSID));
		// メール通知（チェックボックス）
		currentConfig.setMailCheck			( setting.getBoolean(ProjectConfigManager.KEY_MAILCHECK) );
		// SMTPサーバ名
		currentConfig.setSmtp				( setting.get(ProjectConfigManager.KEY_SMTP) );
		// 差出人メールアドレス
		currentConfig.setSender				( setting.get(ProjectConfigManager.KEY_SENDER) );
		// 送信先メールアドレス
		currentConfig.setRecipient			( setting.get(ProjectConfigManager.KEY_RECIPIENT) );
		// タイトルプレフィックス
		currentConfig.setPrefix				( setting.get(ProjectConfigManager.KEY_PREFIX) );	
		// カテゴリ
		currentConfig.setCategory			( setting.get(ProjectConfigManager.KEY_CATEGORY));
		// 優先度
		currentConfig.setPriority			( setting.get(ProjectConfigManager.KEY_PRIORITY));
		// 担当者
		currentConfig.setAssigned			( setting.get(ProjectConfigManager.KEY_ASSIGNED));
		// 投票（チェックボックス）
		currentConfig.setVoteCheck			( setting.getBoolean(ProjectConfigManager.KEY_VOTECHECK));
		// 投票
		currentConfig.setVote				( setting.get(ProjectConfigManager.KEY_VOTE));
		// 期限（チェックボックス）
		currentConfig.setUntilDeadLineCheck	( setting.getBoolean(ProjectConfigManager.KEY_UNTILDEADLINECHECK));
		// 期限
		currentConfig.setUntilDeadLine		( setting.get(ProjectConfigManager.KEY_UNTILDEADLINE));	
		// 固定同期処理（チェックボックス）
		currentConfig.setFixedSyncCheck		( setting.getBoolean(ProjectConfigManager.KEY_FIXEDSYNCCHECK) );
		// アドレス1番目
		currentConfig.setAddress1			( setting.get(ProjectConfigManager.KEY_ADDRESS1) );
		// アドレス2番目
		currentConfig.setAddress2			( setting.get(ProjectConfigManager.KEY_ADDRESS2) );
		// アドレス3番目
		currentConfig.setAddress3			( setting.get(ProjectConfigManager.KEY_ADDRESS3) );
		// アドレス4番目
		currentConfig.setAddress4			( setting.get(ProjectConfigManager.KEY_ADDRESS4) );
		// ポート番号
		currentConfig.setPort				( setting.get(ProjectConfigManager.KEY_PORT) );
		// 自動同期処理（チェックボックス）
		currentConfig.setAutoSyncCheck		( setting.getBoolean(ProjectConfigManager.KEY_AUTOSYNCCHECK) );
		// 通常の同期（ラジオボックス）
		currentConfig.setNormalSyncRadio	( setting.getBoolean(ProjectConfigManager.KEY_NORMALSYNCRADIO) );
		// 端末指定同期（ラジオボックス）
		currentConfig.setAppointSyncRadio	( setting.getBoolean(ProjectConfigManager.KEY_APPOINTSYNCRADIO) );

		// フラグを「カレントオープン」に更新
		this.open=true;
		
		// 登録されたアクションを活性化します。
		setActionsEnabled(true);
	}
	
	
	/**
	 * {@link ProjectConfig プロジェクトコンフィグ情報}を設定します。
	 * 指定のプロジェクトコンフィグがカレントプロジェクトで無い場合は何も処理をせずに無視されます。
	 * 
	 * @param			projectconfig			プロジェクトコンフィグ情報
	 */
	public void setProjectConfig(ProjectConfig projectconfig) {
		// SSIDを取得
		String thisSSID = currentConfig.getSsid();
		String thatSSID = projectconfig.getSsid();

		// SSIDを比較して、カレントプロジェクトで無い場合は何も処理しない
		if(thatSSID.equals(thisSSID)) {
			currentConfig = projectconfig;
		}
	}

	
	/**
	 * 現在カレントに設定されているプロジェクトのコンフィグ情報を取得します。
	 * 
	 * @return			カレントプロジェクトのコンフィグ情報
	 */
	public ProjectConfig getProjectConfig() {
		if(!isOpen()) {
			return null;
		}
		return currentConfig;
	}
	
	
	/**
	 * カレントプロジェクトのSSIDを取得します。
	 * 
	 * @return		カレントプロジェクトのSSID
	 */
	public String getSSID() {
		if(!isOpen()) {
			return null;
		}
		return currentConfig.getSsid();
	}
	
	
	/**
	 * 指定のSSIDがカレントプロジェクトかどうかを判断します。
	 * 
	 * @param		ssid			指定のSSID
	 * @return		true --指定のSSIDがカレントプロジェクト
	 * 				false--指定のSSIDはカレントプロジェクトでない
	 */
	public boolean isCurrentProject(String ssid) {
		if(!isOpen()) {
			return false;
		}
		return ssid.equals(currentConfig.getSsid());
	}

	
	/**
	 * 指定のプロジェクトがカレントプロジェクトかどうかを判断します。
	 * 
	 * @param		projectconfig	指定のプロジェクト情報
	 * @return		true --指定のSSIDがカレントプロジェクト
	 * 				false--指定のSSIDはカレントプロジェクトでない
	 */
	public boolean isCurrentProject(ProjectConfig projectconfig) {
		if(!isOpen()) {
			return false;
		}
		return projectconfig.getSsid().equals(currentConfig.getSsid());
	}

	
	/**
	 * カレントプロジェクトをクローズします。
	 *
	 */
	public void close() {
		this.open=false;
		// 登録されたアクションを非活性化します。
		setActionsEnabled(false);
	}
	
	
	/**
	 * カレントプロジェクトがあるかどうかを返します。
	 * 
	 * @return		true --カレントプロジェクトが設定されている
	 * 				false--カレントプロジェクトが設定されていない
	 */
	public boolean isOpen() {
		return this.open;
	}


	/**
	 * アクションを加えます。
	 * 
	 * @param		action		アクション
	 */
	public void addAction(Action action) {
		actions.add(action);
	}
	
	/**
	 * 登録されたアクションの活性／非活性を切り替える。
	 * 
	 * @param		enabled		true--アクションを活性にする。false--アクションを非活性にする。
	 */
	public void setActionsEnabled(boolean enabled){
		Iterator it = actions.iterator();
		while(it.hasNext()) {
			Action action = (Action)it.next();
			action.setEnabled(enabled);
		}
	}
	

	/**
	 * 指定のアクションを削除します。
	 * 
	 * @param	action		アクション
	 */
	public void removeAction(Action action) {
		actions.remove(action);
	}
	
	/**
	 * カレントプロジェクトのホームディレクトリを取得します。
	 * 
	 * @return		カレントプロジェクトのホームディレクトリのPATH文字列
	 */
	public String getBaseDir() {
		// ワークスペースのディレクトリ取得
		String baseDir = BtsPlugin.getInstance().getStateLocation().toString();

		// SSID取得
		String ssid = currentConfig.getSsid();
		
		// 当該プロジェクトのファイルを保管するディレクトリのPATH文字列生成
		String ssidDir = baseDir + SEPARATOR + ssid + SEPARATOR;

		return ssidDir;
	}
}
