/***********************************************************************
 * Copyright(C) 2006 Valtech Co.,Ltd.
 * All Rights Reserved. This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.opensource.org/licenses/cpl.php
 ***********************************************************************/
package jp.valtech.bts.facade;

import jp.valtech.bts.connection.IssueDBConnection;
import jp.valtech.bts.dao.AttachmentDAO;
import jp.valtech.bts.dao.BtsDBException;
import jp.valtech.bts.dao.CommentHistoryDAO;
import jp.valtech.bts.dao.IssueDAO;
import jp.valtech.bts.dao.IssueHistoryDAO;
import jp.valtech.bts.data.Attachment;
import jp.valtech.bts.data.CommentHistory;
import jp.valtech.bts.data.Issue;
import jp.valtech.bts.data.IssueHistory;
import jp.valtech.bts.data.IssueType;
import jp.valtech.bts.ui.BtsException;
import jp.valtech.bts.ui.BtsPlugin;
import jp.valtech.bts.util.Logging;

/**
 * 課題票送信時に使う機能を集めたFacadeクラスです。
 * <ul>
 *   <li>1.送信する課題票をリリースバージョンとして保存します。
 *   <li>2.送信する課題票の更新履歴を保存します。
 *   <li>3.(1.の)保存アクションを含めた最新の更新履歴を取得します。
 * </ul>
 * 
 * 
 * @author		<A href="mailto:m_sugitou@valtech.jp">M.Sugito</A>
 * @version	Ver.0.8
 */
public class SendIssueFacade implements Logging {

	/**
	 * 何もしない。
	 */
	public SendIssueFacade() {
		;
	}

	/** 更新履歴 */
	private IssueHistory[] allHistory;
	
	/** コメント履歴 */
	private CommentHistory[] allComment;

	/** 
	 * 新規追加された課題票をローカルDBに登録します。
	 * 
	 * @param		issue		新規追加された課題票
	 */
	public void addIssue(Issue issue, IssueHistory[] histories, CommentHistory newComment)  throws BtsException {

		//	DBコネクション取得
		IssueDBConnection dbcon = new IssueDBConnection();
		try {

			// 課題票IDを取得
			setIssueID(issue, dbcon);
			
			// 課題票を登録
			IssueDAO dao = new IssueDAO( dbcon );
			dao.addIssue( issue );
			
			// 添付ファイルを登録
			AttachmentFacade.add(issue, dbcon);
			
			// コメントを登録
			registComment(issue.getFingerPrint(), newComment, dbcon);

			// 履歴を登録
			registHistory(histories, dbcon);
			
			dbcon.commit();

		} catch (Exception e) {
			dbcon.rollback();

			// エラー情報の出力
			String msg = Messages.getString("SendIssueFacade.0"); //$NON-NLS-1$
			logger.fatal(msg, e);
			BtsPlugin.getInstance().error(msg, e);
			throw new BtsException(e);
			
		} finally {
			dbcon.close();
			dbcon = null;
		}
	}

	
	
	/** 
	 * 指定の{@link Issue 課題票情報}を{@link IssueType#RELEASE リリースバージョン}にして更新します。<br>
	 * 元の{@link IssueType 課題票種別}が{@link IssueType#DRAFT ドラフトバージョン}であったときに、
	 * {@link IssueType 課題票種別}を{@link IssueType#RELEASE リリースバージョン}にして更新します。
 	 * <ul>
	 *   <li>既存の{@link IssueType#RELEASE リリースバージョン}の課題票を削除。
	 *   <li>{@link IssueType#DRAFT ドラフトバージョン}となっている元の課題票を{@link IssueType#RELEASE リリースバージョン}として更新する。
	 *   <li>{@link IssueType 課題票種別}以外の情報も最新の情報で更新。
	 * </ul>
	 * 
	 * @param		issue		課題票
	 */
	public Issue updateRelease
		(Issue newIssue, IssueHistory[] histories, Attachment[] deleteList, CommentHistory newComment) throws BtsException {
		
		Issue deleteBeforeIssue = null;
		
		//	DBコネクション取得
		IssueDBConnection dbcon = new IssueDBConnection();
		try {
			// DAO生成 
			IssueDAO dao = new IssueDAO( dbcon );
			
			// 課題票IDを取得
			setIssueID(newIssue, dbcon);

			
			// 既存のリリースバージョンの情報を削除される前に保持。戻り値として返す為。
			// (既存のリリースバージョンが無いケースも有り。『新規作成→DRAFT登録→配信』の時)
			deleteBeforeIssue = dao.getByFingerPrint(newIssue.getFingerPrint(), IssueType.RELEASE_VALUE);
			if(deleteBeforeIssue != null) {
				// 既存のリリースバージョンを削除する。
				dao.deleteByFingerPrint( newIssue.getFingerPrint() , IssueType.RELEASE_VALUE );
			}

			// ドラフトバージョンをリリースバージョンに更新する。
			dao.modifyIssue(newIssue, IssueType.DRAFT_VALUE);
			
			// 既存のDRAFT課題票が持つ添付情報をRELEASE版に置き換える
			AttachmentFacade.replaceReleaseVersion(newIssue, dbcon);

			// 削除された添付ファイル情報を削除
			AttachmentFacade.delete(deleteList, dbcon);

			// 新規追加添付ファイル
			AttachmentFacade.add(newIssue, dbcon);

			// コメントを登録
			registComment(newIssue.getFingerPrint(), newComment, dbcon);

			// 履歴を保存
			registHistory(histories, dbcon);

			dbcon.commit();

			return deleteBeforeIssue;
		} catch (Exception e) {
			dbcon.rollback();

			// エラー情報の出力
			String msg = Messages.getString("SendIssueFacade.1"); //$NON-NLS-1$
			logger.fatal(msg, e);
			BtsPlugin.getInstance().error(msg, e);
			throw new BtsException(e);
			
		} finally {
			dbcon.close();
			dbcon = null;
		}
	}
	

	/** 
	 * 指定の{@link Issue 課題票情報}を{@link IssueType#RELEASE リリースバージョン}にして更新します。<br>
	 * 元の{@link IssueType 課題票種別}が{@link IssueType#RELEASE リリースバージョン}であったときに、
	 * {@link IssueType 課題票種別}はそのままに他の属性を更新したものを再度{@link IssueType#RELEASE リリースバージョン}
	 * として更新します。
	 * 
	 * @param		issue		課題票
	 */
	public void updateVersion
		(Issue newIssue, IssueHistory[] histories, Attachment[] deleteList, CommentHistory newComment) throws BtsException {

		//	DBコネクション取得
		IssueDBConnection dbcon = new IssueDBConnection();
		try {
			// DAO生成 
			IssueDAO dao = new IssueDAO( dbcon );
			
			// 課題票情報をそのまま更新する。
			dao.modifyIssue(newIssue, newIssue.getType());
			
			// 削除された添付ファイル情報を削除
			AttachmentFacade.delete(deleteList, dbcon);

			// 新規追加添付ファイル
			AttachmentFacade.add(newIssue, dbcon);

			// コメントを登録
			registComment(newIssue.getFingerPrint(), newComment, dbcon);

			// 履歴を保存
			registHistory(histories, dbcon);

			dbcon.commit();

		} catch (Exception e) {
			dbcon.rollback();

			// エラー情報の出力
			String msg = Messages.getString("SendIssueFacade.2"); //$NON-NLS-1$
			logger.fatal(msg, e);
			BtsPlugin.getInstance().error(msg, e);
			throw new BtsException(e);
			
		} finally {
			dbcon.close();
			dbcon = null;
		}
	}


	/** 
	 * 課題票の履歴をローカルDBに登録します。
	 * 登録に成功すると、登録した履歴の元となる課題票が持つ全ての履歴を取得して返します。
	 * 
	 * @param		histories		課題票の履歴
	 */
	private void registHistory(IssueHistory[] histories, IssueDBConnection dbcon)  throws BtsDBException {
		
		IssueHistoryDAO dao = new IssueHistoryDAO( dbcon );
		
		// 履歴を登録
		if(histories !=null && histories.length != 0) {
			for (int i = 0; i < histories.length; i++) {
				dao.addHistory( histories[i] );
			}
		}
		
		// 登録した履歴を含む全ての履歴を取得します。
		this.allHistory = dao.getByFingerPrint( histories[0].getFingerPrint(), histories[0].getType() ); 
	}

	/** 
	 * コメントをローカルDBに登録します。
	 * 登録に成功すると、その課題票が持つ全てのコメントを取得して返します。
	 * 
	 * @param		comment		コメント(1件)
	 */
	private void registComment(String fingerPrint, CommentHistory comment, IssueDBConnection dbcon)  throws BtsDBException {
		
		CommentHistoryDAO dao = new CommentHistoryDAO(dbcon);
		
		// コメントを登録
		if(comment !=null) {
			dao.addComment(comment);
		}
		
		// 登録したコメントを含む全てのコメントを取得します。
		this.allComment = dao.getByFingerPrint( fingerPrint, IssueType.RELEASE_VALUE ); 
	}

	/**
	 * 更新履歴を返します。
	 * 
	 * @return		更新履歴
	 */
	public IssueHistory[] getAllHistory() {
		return this.allHistory;
	}
	
	public CommentHistory[] getAllComment() {
		return this.allComment;
	}

	
	/**
	 * 課題票IDを生成して課題票に設定します。
	 */
	private synchronized static void setIssueID(Issue newIssue, IssueDBConnection dbcon) {
		
		// 課題票IDを割り振る
		if(newIssue.getIssueID()==0) {
			// 課題票IDを取得
			try {
				// DBを検索して現在の有効な課題票の数を取得
				IssueDAO dao = new IssueDAO( dbcon );
				int cnt = dao.getUsefulIssueCnt();
				
				newIssue.setIssueID(cnt+1);
				
			} catch (Exception e) {
				String msg = Messages.getString("SendIssueFacade.3"); //$NON-NLS-1$
				logger.fatal(msg, e);
				BtsPlugin.getInstance().error(msg, e);
				newIssue.setIssueID(0);
			}
		}
	}

	/**
	 * 指定のFingerPrintを持つ課題票を削除します。
	 * 
	 * @param		fingerPrint		課題票のFingerPrint
	 */
	public void garbageByFingerPrint(String fingerPrint, IssueHistory history) {
		
		IssueDBConnection dbcon = new IssueDBConnection();
		try {
			
			IssueDAO issueDAO = new IssueDAO( dbcon );
			IssueHistoryDAO historyDAO = new IssueHistoryDAO(dbcon);
			CommentHistoryDAO commentHistoryDAO = new CommentHistoryDAO(dbcon);

			
			// 既にGARBAGEバージョンの課題票がある場合は削除する 
			Issue garbageIssue = issueDAO.getByFingerPrint(fingerPrint, IssueType.GARBAGE_VALUE);
			if(garbageIssue != null) {
				issueDAO.deleteByFingerPrint(fingerPrint, IssueType.GARBAGE_VALUE);
				
				// 添付を削除
				AttachmentFacade.deleteAll(fingerPrint, IssueType.GARBAGE_VALUE, dbcon);

				// 課題票履歴を削除
				historyDAO.deleteByFingerPrint(fingerPrint, IssueType.GARBAGE_VALUE);
				
				// 課題票コメント履歴を削除
				commentHistoryDAO.deleteByFingerPrint(fingerPrint, IssueType.GARBAGE_VALUE);
			}


			//### 削除後にRELEASEバージョンをGARBAGEバージョンに変える ###
			
			// 課題票の種別をごみ箱に変更
			issueDAO.changeGarbageVersion(fingerPrint);
			
			// 添付の種別をごみ箱に変更
			AttachmentDAO attachmentDAO = new AttachmentDAO( dbcon );
			attachmentDAO.changeGarbageVersion(fingerPrint);

			// 課題票履歴の種別をごみ箱に変更
			historyDAO.changeGarbageVersion(fingerPrint);

			// 課題票コメント履歴の種別をごみ箱に変更
			commentHistoryDAO.changeGarbageVersion(fingerPrint);
			
			// 履歴を更新
			registHistory(new IssueHistory[]{history}, dbcon);
			
			dbcon.commit();
		
		} catch (Exception e) {
			dbcon.rollback();

			// エラー情報の出力
			String msg = Messages.getString("SendIssueFacade.4") + fingerPrint; //$NON-NLS-1$
			logger.fatal(msg, e);
			BtsPlugin.getInstance().error(msg, e);
		} finally {
			dbcon.close();
			dbcon = null;
		}
	}


}
