/***********************************************************************
 * Copyright(C) 2006 Valtech Co.,Ltd.
 * All Rights Reserved. This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.opensource.org/licenses/cpl.php
 ***********************************************************************/
package jp.valtech.bts.network.command;

import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.net.Socket;

import jp.valtech.bts.data.NetworkConfig;
import jp.valtech.bts.network.Request;
import jp.valtech.bts.network.Response;

/**
 * TCPデータ通信を行うコマンド機能のための基底クラス。
 * 指定されたホストに対してデータを送る処理は{@link TCPClientCommand#send(Request)}を使用します。
 * 
 * @author		<A href="mailto:m_sugitou@valtech.jp">M.Sugito</A>
 * @version	Ver.0.8
 */
public abstract class TCPClientCommand {

	/** レスポンスが返ってくるまでのタイムアウト */
	private static final int DEFAULT_RESPONSE_TIME_OUT = 1000 * 10;
	
	/** 接続タイムアウト設定（ミリ秒）*/
	private int timeout = DEFAULT_RESPONSE_TIME_OUT;

	/** ローカルマシンのネットワーク設定 */
	protected NetworkConfig myconfig;

	/** リクエスト送信先ホスト */
	protected String host		= null;
	
	/** リクエスト送信先ポート番号 */
	protected int	port		= 0;

	/** 送信結果 */
	private boolean successful = false;
	

	/**
	 * インスタンスにネットワーク情報を設定します。
	 * 
	 * @param		myconfig 	ローカルマシンのネットワーク情報
	 * @param		host		送信先ホスト名
	 * @param		port 		送信先ポート番号
	 */
	public TCPClientCommand( NetworkConfig myconfig, String host, int port ) {
		this.myconfig = myconfig;
		this.host		= host; 
		this.port		= port;
	}
	

	/**
	 * タイムアウト設定
	 * 
	 * @param		timeout		タイムアウト（ミリ秒）
	 */
	public void setTimeout(int timeout) {
		this.timeout = timeout;
	}
	
	
	/**
	 * TCP接続を確立し、<code>request</code>オブジェクトを送信します。
	 * 
	 * @param		request		送信情報を格納したオブジェクト
	 * @return		送信先からの返信情報を格納したオブジェクト
	 * @throws		IOException
	 * @throws		ClassNotFoundException
	 */
	public Response send( Request request ) throws IOException, ClassNotFoundException {

		// ソケット生成
		Socket socket = new Socket( this.host , this.port );
		socket.setSoTimeout(timeout);
		
		// オブジェクト送受信用のストリーム
		ObjectOutputStream	out	= null;
		ObjectInputStream	ins = null;
		
		try{
			// 送信情報が格納されたオブジェクトを送信
			out = new ObjectOutputStream( socket.getOutputStream() );
			out.writeObject(request);
			out.flush();
			socket.shutdownOutput();

			// 相手側からの返信を受信。受信したオブジェクトをストリームから取り出す
			ins = new ObjectInputStream( socket.getInputStream() );
			Response response = (Response)ins.readObject();
			
			// 受信結果がResponse.OKであれば「送信結果」にtrueを設定します。
			if(Response.OK.equals(response.getCode())) {
				this.successful = true;
			}

			// 受信オブジェクトを返す
			return response;

		}finally{
			if( out != null ){
				out.close();
			}
			if( ins != null ){
				ins.close();
			}
			if( socket != null ){
				socket.close();
			}
		}
	}

	/**
	 * 実行結果を取得する。
	 * 
	 * @return			true --コマンド実行成功。
	 * 					false--コマンド実行失敗。
	 */
	public boolean isSuccessful() {
		return this.successful;
	}

}
