/***********************************************************************
 * Copyright(C) 2006 Valtech Co.,Ltd.
 * All Rights Reserved. This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.opensource.org/licenses/cpl.php
 ***********************************************************************/
package jp.valtech.bts.network.server;

import java.io.IOException;
import java.net.DatagramPacket;
import java.net.InetAddress;
import java.net.MulticastSocket;
import java.net.SocketException;

import jp.valtech.bts.data.CurrentProject;
import jp.valtech.bts.data.NetworkConfig;
import jp.valtech.bts.network.Request;
import jp.valtech.bts.network.command.MulticastServerCommand;
import jp.valtech.bts.util.BTSUtility;
import jp.valtech.bts.util.IOUtility;
import jp.valtech.bts.util.Logging;

/**
 * マルチキャスト通信サーバプロセス。
 * マルチキャストで通知されるコマンドの受信処理を行います。
 * 
 * @author		<A href="mailto:m_sugitou@valtech.jp">M.Sugito</A>
 * @version	Ver.0.8
 */
class MulticastServer implements Runnable, Logging {

	/** ローカルマシンのネットワーク設定 */
	private NetworkConfig myconfig	= null;

	/** マルチキャスト接続 */
	private MulticastSocket	multicastSocket	= null;

	/** 接続したマルチキャストアドレス情報 */
	private InetAddress	multicastAddrss	= null;
	
	/**
	 * マルチキャスト通信サーバインスタンスを生成します。
	 * 
	 * @param	config		設定情報
	 */
	public MulticastServer( NetworkConfig myconfig ) throws IOException {
		super();
		this.myconfig		= myconfig;

		// マルチキャストアドレスと使用するポートを取得
		String address = myconfig.getMulticastAddress();
		int port = myconfig.getMulticastPort();

		// マルチキャストグループに参加。失敗したら例外を投げます。
		this.multicastSocket = new MulticastSocket(port );
		this.multicastAddrss = InetAddress.getByName( address );
		this.multicastSocket.joinGroup( this.multicastAddrss );
		logger.info( Messages.getString("MulticastServer.0") + port + ",MulticastAddress:" + address+ "}"); //$NON-NLS-1$
	}

	/**
	 * マルチキャスト受信処理スレッド。
	 * 
	 * このメソッドは、送信元と受信先のクライアントが同一の場合(ローカルホストからの通信)処理は実行されません。
	 * 他のクライアントからUDP/マルチキャストパケットを受け付けると、受信データから実行されるコマンドクラスを
	 * 判定して{@link ServerCommand#getCommand(String)}よりコマンドクラスを取得します。
	 * 
	 * @see java.lang.Runnable#run()
	 */
	public void run() {
		
		// 通信を待ち受けます。受信後、次の待ち受けに入るようにループします。
		boolean active = true;
		while( active ){
			try {
				// 受信用データグラムパケット生成
				byte[] buf				= new byte[ 1024 ];
				DatagramPacket packet	= new DatagramPacket( buf, buf.length );

				// 接続が確立されるまで、ここでブロックします。
				this.multicastSocket.receive( packet );

				// 受信したデータをリクエストオブジェクトに変換します。
				byte[] receiveData = packet.getData(); 
				Request request = (Request)IOUtility.toObject(receiveData);

				// 送信元情報により実行するかどうかを判断する。
				if( isLocalRequest(packet) ){
					
					//  同一ホストからのリクエストは実行しない
//					logger.info("ローカルホストからのリクエストは実行しません。");
	
				} else if(!isMyProject(request)) {
					
					//  異なるプロジェクトからのリクエストは実行しない
					logger.info(Messages.getString("MulticastServer.1") + request.getSSID()); //$NON-NLS-1$
				
				} else {

					// 実行するコマンドのインスタンスを生成します。
					Object command = BTSUtility.getCommand( request.getCommand());

					//	コマンドがMulticastServerCommandのインスタンスでれば実行します。
					if( command != null && command instanceof MulticastServerCommand){

						// 受信情報を格納
						MulticastServerCommand udpCommand = (MulticastServerCommand)command;
						udpCommand.setRequest( request);
						udpCommand.setPacket(packet);
						udpCommand.setConfig(myconfig);

						// 受信コマンド実行
						Thread t = new Thread(udpCommand, "MulticastServerCommand");
						t.start();

					}else{
						throw new ClassNotFoundException(Messages.getString("MulticastServer.2")); //$NON-NLS-1$
					}
				}
			} catch ( SocketException sx) {
				// この例外を受け取るとループを抜けます。
				logger.info(Messages.getString("MulticastServer.3")); //$NON-NLS-1$
				active = false;
			} catch ( Exception e ) {
				logger.error(Messages.getString("MulticastServer.4"), e ); //$NON-NLS-1$
			}
		}
		logger.info(Messages.getString("MulticastServer.5")); //$NON-NLS-1$
	}
	
	
	/**
	 * マルチキャスト通信サーバプロセスを停止します。 
	 * このメソッドを実行することで{@link #run()}のループ処理内で使っている{@link MulticastSocket}
	 * が{@link SocketException}をスローします。
	 * それにより同ループ処理を抜け出してスレッドが終了します。
	 */
	public void stop() throws IOException {
		if( this.multicastSocket != null ){
			this.multicastSocket.leaveGroup( this.multicastAddrss );
			this.multicastSocket.close();
			this.multicastSocket = null;
		}
	}
	
	
	/**
	 * リクエスト元が自分と同一プロジェクトかどうかを判別します。
	 * 
	 * @param		request			送信元からの送信オブジェクト
	 * @return		true--リクエスト元がローカル。false--リクエスト元がリモート
	 */
	private boolean isMyProject(Request request) {
		String requestSSID = request.getSSID();
		if(requestSSID == null || requestSSID.equals("")) {
			throw new IllegalArgumentException(Messages.getString("MulticastServer.6")); //$NON-NLS-1$
		}
		String mySSID = CurrentProject.getInsance().getSSID();
		return mySSID.equals(requestSSID);
	}

	
	/**
	 * リクエスト元がローカルかどうかを判別します。
	 * 
	 * @param		packet			受信したデータグラムパケット
	 * @return		true--リクエスト元がローカル。false--リクエスト元がリモート
	 */
	private boolean isLocalRequest(DatagramPacket packet) {
		try {
			String localAddress	= InetAddress.getLocalHost().getHostAddress();
			String remoteAddress = packet.getAddress().getHostAddress();
			
			return localAddress.equals( remoteAddress );
		} catch (Exception e) {
			return false;
		}
		
	}
}
