/***********************************************************************
 * Copyright(C) 2006 Valtech Co.,Ltd.
 * All Rights Reserved. This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.opensource.org/licenses/cpl.php
 ***********************************************************************/
package jp.valtech.bts.ui.action;

import java.util.ArrayList;
import java.util.List;

import jp.valtech.bts.data.CurrentProject;
import jp.valtech.bts.data.Issue;
import jp.valtech.bts.data.IssueHistory;
import jp.valtech.bts.data.IssueType;
import jp.valtech.bts.facade.SendIssueFacade;
import jp.valtech.bts.network.ClientInfo;
import jp.valtech.bts.ui.BtsPlugin;
import jp.valtech.bts.ui.IBtsListener;
import jp.valtech.bts.ui.editor.IssueMultiPageEditor;
import jp.valtech.bts.util.BTSUtility;
import jp.valtech.bts.util.IssueUtility;
import jp.valtech.bts.util.Logging;

import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.swt.SWT;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;

/**
 * 指定の課題票をゴミ箱へ移動するアクションです。
 * 
 * @author		<A href="mailto:m_sugitou@valtech.jp">M.Sugito</A>
 * @version	Ver.0.8
 */
public class SendGarbageIssueAction extends SendIssueAction implements Logging{

	/** 課題票 */
	private Issue garbageIssue;


	/**
	 * 呼び出し元のビューインスタンスを設定します。
	 * 
	 */
	public SendGarbageIssueAction() {
		super();
		setText(Messages.getString("SendGarbageIssueAction.0")); //$NON-NLS-1$
		setToolTipText(Messages.getString("SendGarbageIssueAction.1"));  //$NON-NLS-1$
		setImageDescriptor(BtsPlugin.getInstance().getImageDescriptor("garbage.gif"));
	}

	
	/**
	 * 指定の課題票をゴミ箱へ移動します。
	 */
	public void run() {

		super.run();
		
        // 確認ダイアログ表示
        MessageDialog dialog = new MessageDialog
        	( BtsPlugin.getInstance().getShell()
        	, Messages.getString("SendGarbageIssueAction.2") //$NON-NLS-1$
        	, null
        	, Messages.getString("SendGarbageIssueAction.3") + garbageIssue.getTitle() + Messages.getString("SendGarbageIssueAction.4") //$NON-NLS-1$ //$NON-NLS-2$
        	, MessageDialog.QUESTION
        	, new String[]{Messages.getString("SendGarbageIssueAction.5"), Messages.getString("SendGarbageIssueAction.6")} //$NON-NLS-1$ //$NON-NLS-2$
        	, 1
        	);

        // [配信する]以外を押下した場合は何もしない。
        // ボタン押下の戻り値：[×ボタン]:-1、[配信する]:0、[キャンセル]:1
        int res = dialog.open();
        if(res != 0) {
        	return;
        }
        
		// RELEASE版課題票をビュー表示から消す
		List listeners = BtsPlugin.getInstance().getListeners();
		for (int idx = 0; idx < listeners.size(); idx++) {
			IBtsListener listnener = (IBtsListener)listeners.get(idx);
			listnener.deleteIssue(garbageIssue);
		}

		//******************  課題票情報をごみ箱版情報に更新  ********************
        garbageIssue.setType(IssueType.GARBAGE_VALUE);
        garbageIssue.setUpdateDate(System.currentTimeMillis());
		//***************************************************************************
        
        // ごみばこ更新の履歴を取得
        String name = CurrentProject.getInsance().getProjectConfig().getUserName();
        IssueHistory history = IssueUtility.createGarbageHistory(garbageIssue, name);
        
        // DBのデータをGARBAGE版に更新 ＆ ごみ箱移動した履歴を追加
        SendIssueFacade facade = new SendIssueFacade();
        facade.garbageByFingerPrint(garbageIssue.getFingerPrint(), history);
        
		// GARBAGE版に更新したので課題票をビュー表示に追加
		for (int idx = 0; idx < listeners.size(); idx++) {
			IBtsListener listnener = (IBtsListener)listeners.get(idx);
			listnener.addIssue(garbageIssue);
		}
		
        // ごみばこ更新を含めた過去の更新履歴を取得
        IssueHistory[] sendHistory = facade.getAllHistory();

        // 対象の課題票が開いている場合は表示を更新する。
        IssueMultiPageEditor editorPart =	
        	BtsPlugin.getInstance().getIssueEditor(garbageIssue.getFingerPrint(), IssueType.GARBAGE_VALUE);
        
        // !!!! 対象となる課題票のIssueインスタンスがGARBAGE/RELEASE両方のケースがあり得る
        if(editorPart == null) {
        	editorPart = BtsPlugin.getInstance().getIssueEditor(garbageIssue.getFingerPrint(), IssueType.RELEASE_VALUE); 
        }

        if(editorPart != null) {
			editorPart.postIssueSave(garbageIssue);			//「概要タブ」を再表示(タブのアイコン表示を変えるため)
			editorPart.postIssueHistorySave(sendHistory);	//「履歴タブ」を再表示
        }
        
        // ごみ箱移動した時の履歴だけを格納する
        garbageIssue.setIssueHistories(new IssueHistory[]{history});

        // 送信
        sendIssue(garbageIssue);
        
        // 送信メッセージをローカルに保存します。
        saveMessage(garbageIssue, IssueType.GARBAGE_VALUE);
	}
	
	
	/*
	 *  (非 Javadoc)
	 * @see jp.valtech.bts.ui.action.SendAction#sendIssue(jp.valtech.bts.data.Issue)
	 */
	protected void sendIssue(final Issue issue) {

		// 課題票をメール送信
        sendMail(issue);

        // 送信は親クラスを使う
		super.sendIssue(issue);

		// 配信結果通知ダイアログを表示
    	ReceiveUserDialog userDialog = new ReceiveUserDialog(issue.getTitle());
		userDialog.setClientInfo(sendUserList);
        userDialog.open();
	}

	
	/**
	 * 課題票を格納します。
	 * 
	 * @param		issue		課題票
	 */
	public void setIssue(Issue issue) {
		this.garbageIssue = issue;
	}


//	=============================================================================================
//	================================   ここからインナークラス   =================================
//	=============================================================================================
	
	/**
	 * 配信結果通知ダイアログ。
	 * クライアント情報を表示するダイアログです。
	 * 
	 * @author		<A href="mailto:m_sugitou@valtech.jp">$Author$</A>
	 * @version		$Rev$
	 */
	protected class ReceiveUserDialog extends MessageDialog {
	
		/** ユーザ情報 */
		private ArrayList clientList;
		
		/**
		 * ダイアログを生成します。
		 * 
		 * @param		issueTitle		ダイアログに表示する課題票タイトル
		 */
		public ReceiveUserDialog(String issueTitle) {
        	super( BtsPlugin.getInstance().getShell()
    	        	, Messages.getString("SendGarbageIssueAction.7") //$NON-NLS-1$
    	        	, null
    	        	, Messages.getString("SendGarbageIssueAction.8") + issueTitle + Messages.getString("SendGarbageIssueAction.9") //$NON-NLS-1$ //$NON-NLS-2$
    	        	, MessageDialog.INFORMATION
    	        	, new String[]{"OK"}
    	        	, 0
    	        	);
		}

		/**
		 * クライアント情報を設定します。
		 * 
		 * @param		clientInfo		クライアント情報
		 */
		public void setClientInfo(ArrayList clientList) {
			this.clientList = clientList;
		}
		
		
		/**
		 * 配信したユーザ情報を表示します。
		 * 
		 * @param		parent				親コンポジット
		 * @see 		org.eclipse.jface.dialogs.MessageDialog#createCustomArea(org.eclipse.swt.widgets.Composite)
		 */
	    protected Control createCustomArea(Composite parent) {

	    	// このエリアのコンポジット生成。
	        Composite thisArea = new Composite(parent, SWT.NONE);
	        GridLayout gridLayout = new GridLayout();
	        gridLayout.marginLeft = 40;
	        thisArea.setLayout(gridLayout);

	        // リストコントロールにクライアント情報を表示する
	        org.eclipse.swt.widgets.List memberList = new org.eclipse.swt.widgets.List (thisArea, SWT.BORDER | SWT.MULTI | SWT.V_SCROLL);

	        if(clientList ==null || clientList.size()==0 ) {
				memberList.add(Messages.getString("SendGarbageIssueAction.10")); //$NON-NLS-1$
	        } else {
		        for (int i = 0; i < clientList.size(); i++) {
		        	
		        	ClientInfo clientInfo = (ClientInfo)clientList.get(i);
		        	// ユーザ名取得
					String userName = clientInfo.getClientName();
					// アドレス取得
					String address = BTSUtility.rightPad("(" + clientInfo.getClientAddress() + ")", 19);
					
					memberList.add(address + userName );
				}
	        }

	        // リストコントロールのレイアウト設定
	        GridData gridData = new GridData();
	        gridData.heightHint = 60;
	        gridData.widthHint = 280;
	        memberList.setLayoutData(gridData);

	        return thisArea;
	    }
	}
}
