/***********************************************************************
 * Copyright(C) 2006 Valtech Co.,Ltd.
 * All Rights Reserved. This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.opensource.org/licenses/cpl.php
 ***********************************************************************/
package jp.valtech.bts.ui.action;


import java.lang.reflect.InvocationTargetException;
import java.util.ArrayList;
import java.util.Date;

import jp.valtech.bts.command.client.Join;
import jp.valtech.bts.command.client.SendIssue;
import jp.valtech.bts.data.CurrentProject;
import jp.valtech.bts.data.Issue;
import jp.valtech.bts.data.IssueType;
import jp.valtech.bts.data.MessagePacket;
import jp.valtech.bts.data.MessageType;
import jp.valtech.bts.data.NetworkConfig;
import jp.valtech.bts.facade.MessagePacketFacade;
import jp.valtech.bts.network.ClientInfo;
import jp.valtech.bts.network.ClientList;
import jp.valtech.bts.ui.BtsPlugin;
import jp.valtech.bts.ui.navigator.ProjectConfig;
import jp.valtech.bts.util.BTSUtility;
import jp.valtech.bts.util.BtsMailSender;
import jp.valtech.bts.util.Logging;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.jface.action.Action;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.dialogs.ProgressMonitorDialog;
import org.eclipse.jface.operation.IRunnableWithProgress;


/**
 * {@link UpdateReport クライアント用の課題票更新コマンド}を実行するアクションです。
 * 
 * @author		<A href="mailto:m_sugitou@valtech.jp">M.Sugito</A>
 * @version	Ver.0.8
 */
public class SendIssueAction extends Action implements Logging {

	/** ダイアログ表示時間が、この秒数をオーバーしてたら、配信前にもう一度Joinを行う。  */
	private static final int OK_PRESS_INTERVAL = 1000 * 30; 
	
    /** 配信前にJoinコマンドを出した時間 */
	private long time1 = System.currentTimeMillis();

    /** 配信に成功したユーザを確認するリスト */
	protected ArrayList sendUserList = new ArrayList();
	
	/**
	 * 課題票を設定します。
	 */
	public SendIssueAction() {
		super();
	}

	
	/**
	 * 課題票を配信します。
	 */
	public void run() {
		
		// 正確な配信先を取得するため、一旦クライアントリストをクリアする。
		ClientList.clear();
		ClientList.addMyInfo();
		
		// Joinコマンドを送る
    	Join join = new Join( new NetworkConfig());
		join.execute();

		// Joinした時刻をとっておく（ダイアログで「配信する」を選択するまでの時間が長かったら再度Joinするため）
		time1 = System.currentTimeMillis();
	}
	
	
	/**
	 * 指定の課題票を送信します。
	 * 
	 * @param		newIssue			対象となる課題票
	 */
	protected void sendIssue(final Issue issue) {
		try {

			// プログレスダイアログを生成
			IRunnableWithProgress progress = new IRunnableWithProgress() {

				/*
				 *  (非 Javadoc)
				 * @see org.eclipse.jface.operation.IRunnableWithProgress#run(org.eclipse.core.runtime.IProgressMonitor)
				 */
				public void run(IProgressMonitor monitor) throws InvocationTargetException, InterruptedException {
					
					// 最初にクリアしておく
					sendUserList.clear();
					
					monitor.subTask(Messages.getString("SendIssueAction.0")); //$NON-NLS-1$
					
					// time1～time2の間の時間が長かった場合には、再度JOINします
			        long time2 = System.currentTimeMillis();
			        if(time2 - time1 > OK_PRESS_INTERVAL) {
			        	Join join = new Join( new NetworkConfig());
						join.execute();

						// JOINの結果が確実に返るまで時間を稼ぐ
						Thread.sleep(1000 * 5);
					} else {
						// JOINの結果が確実に返るまで時間を稼ぐ
						Thread.sleep(1000 * 2);
					}

					// 配信先ユーザリスト取得
					// sleep後に配信先ユーザリストを生成する。これ以降にJoinをコールバックしてきたユーザは無視。
					// ダイアログを出す前に、既に一回Joinしているのでさほど影響はないと思われる。
					ClientInfo[] clientInfo = ClientList.getClientInfoList();

					// プログレスバーを設定
					monitor.beginTask(Messages.getString("SendIssueAction.1"), clientInfo.length); //$NON-NLS-1$

					// 取得したユーザ分処理する（自分は除く）
					for (int i = 0; i < clientInfo.length; i++) {
						ClientInfo info = clientInfo[i];

						// キャンセルボタンを押した場合
						if (monitor.isCanceled()) {
							throw new InterruptedException();
						}

						// ユーザ情報が自分自身の場合は配信しない
						if(!BTSUtility.isLocalHost(info)) {

							// 課題票送信コマンド生成
							SendIssue send= new SendIssue( new NetworkConfig()
														, info.getClientAddress()
														, info.getPort().intValue());
							send.setIssue(issue);
							
							// ダイアログにメッセージ表示
							monitor.subTask(Messages.getString("SendIssueAction.2") + info.getClientName() + Messages.getString("SendIssueAction.3")); //$NON-NLS-1$ //$NON-NLS-2$

							// 課題票送信
							try {
								send.execute();
							} catch (Exception e) {
								logger.fatal(Messages.getString("SendIssueAction.4") + info, e); //$NON-NLS-1$
							}
								
							// 配信成功したユーザ情報を集める
							if(send.isSuccessful()){
								sendUserList.add(info);
							}
						}
						
						// プログレスバーのメモリを上げる
						monitor.worked(1);
					}
					monitor.done();
				}
			};

			// プログレスダイアログの表示
			ProgressMonitorDialog dialog = new ProgressMonitorDialog(BtsPlugin.getInstance().getShell());
			dialog.run(true, true, progress);


		} catch (InvocationTargetException e) {
			BtsPlugin.getInstance().errorlog(e);
		} catch (InterruptedException e) {
			// キャンセル後の処理
			MessageDialog.openInformation(BtsPlugin.getInstance().getShell(), Messages.getString("SendIssueAction.5"), Messages.getString("SendIssueAction.6")); //$NON-NLS-1$ //$NON-NLS-2$
		}
	}

	
	/**
	 * 送信メッセージを保存します。
	 * 課題票情報と送信情報を元にメッセージを生成してローカルDBに保存します。
	 * 
	 * @param			newIssue			送信した課題票情報
	 */
	protected void saveMessage(Issue newIssue, String type) {
		MessagePacket message = new MessagePacket();

		// 配信ユーザ名設定（自分）
		ProjectConfig conf = CurrentProject.getInsance().getProjectConfig();
		message.setFromUser(conf.getUserName());
		
		// メッセージ種別設定
		message.setMessageType(MessageType.ISSUE_VALUE);
		
		// 送信日時設定
		message.setSent(new Date());
		
		// 課題票のフィンガープリント設定
		message.setFingerPrint(newIssue.getFingerPrint());
		
		// 課題票の種別設定
		message.setIssueType(type);
		
		// メッセージ本文設定
		StringBuffer msg = new StringBuffer();

		// メッセージ本文：１行目の課題票情報
		msg.append(Messages.getString("SendIssueAction.7")).append(newIssue.getDisplayIssueID()).append("]"); //$NON-NLS-1$
		msg.append(newIssue.getTitle());

		// メッセージ本文：配信先情報
		msg.append(Messages.getString("SendIssueAction.8")); //$NON-NLS-1$
        if(sendUserList ==null || sendUserList.size()==0 ) {
        	msg.append(Messages.getString("SendIssueAction.9")); //$NON-NLS-1$
        
        } else {
	        for (int i = 0; i < sendUserList.size(); i++) {
	        	
	        	// ユーザ名取得
	        	ClientInfo clientInfo = (ClientInfo)sendUserList.get(i);
				String userName = clientInfo.getClientName();
				// アドレス取得
				String address = BTSUtility.rightPad("(" + clientInfo.getClientAddress() + ")", 19);
				
				// 配信先情報を１行で表示
				msg.append( "\n  " ).append( address ).append( userName );
			}
        }
        
        // メッセージ本文：配信日
		msg.append(Messages.getString("SendIssueAction.10")); //$NON-NLS-1$
		msg.append( BTSUtility.formatDate( message.getSent()) );
		
        // メッセージ本文：バージョン情報
		msg.append(Messages.getString("SendIssueAction.11")); //$NON-NLS-1$
		if(IssueType.RELEASE_VALUE.equals(type)) {
			msg.append("\n  Ver.").append( newIssue.getVersion());
		} else if(IssueType.GARBAGE_VALUE.equals(type)) {
			msg.append(Messages.getString("SendIssueAction.12")); //$NON-NLS-1$
		}
		
		message.setMessage(msg.toString());
        
        // 保存 ＆ ビュー表示更新
        MessagePacketFacade facade = new MessagePacketFacade();
        facade.addMessage(message);
	}
	

	/**
	 * 更新通知メール送信
	 * 
	 * @param		newIssue		課題票
	 */
	protected void sendMail(Issue newIssue) {

		ProjectConfig projectConfig = CurrentProject.getInsance().getProjectConfig();

		if( projectConfig.isMailCheck() ){
			BtsMailSender mailSender = new BtsMailSender();
			mailSender.setIssue(newIssue);
			
			Thread mailThread = new Thread(mailSender, "MAIL");
			mailThread.start();
		}
	}
}
