/***********************************************************************
 * Copyright(C) 2006 Valtech Co.,Ltd.
 * All Rights Reserved. This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.opensource.org/licenses/cpl.php
 ***********************************************************************/
package jp.valtech.bts.ui.action;


import java.util.ArrayList;

import jp.valtech.bts.data.Attachment;
import jp.valtech.bts.data.CommentHistory;
import jp.valtech.bts.data.CurrentProject;
import jp.valtech.bts.data.Issue;
import jp.valtech.bts.data.IssueHistory;
import jp.valtech.bts.data.IssueType;
import jp.valtech.bts.data.Vote;
import jp.valtech.bts.facade.ModifyIssueFacade;
import jp.valtech.bts.facade.SendIssueFacade;
import jp.valtech.bts.facade.VoteDAOFacade;
import jp.valtech.bts.network.ClientInfo;
import jp.valtech.bts.ui.BtsException;
import jp.valtech.bts.ui.BtsPlugin;
import jp.valtech.bts.ui.IBtsListener;
import jp.valtech.bts.ui.editor.IssueMultiPageEditor;
import jp.valtech.bts.ui.editor.IssueMultiPageEditorInput;
import jp.valtech.bts.ui.navigator.ProjectConfig;
import jp.valtech.bts.util.AttachmentUtility;
import jp.valtech.bts.util.BTSUtility;
import jp.valtech.bts.util.IssueUtility;

import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.swt.SWT;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.List;


/**
 * {@link UpdateReport クライアント用の課題票更新コマンド}を実行するアクションです。
 * 
 * @author		<A href="mailto:m_sugitou@valtech.jp">M.Sugito</A>
 * @version	Ver.0.8
 */
public class SendReleaseIssueAction extends SendIssueAction {

	
	private Issue latestIssue;
	
	private IssueHistory[] sendHistory;
	
	private CommentHistory[] sendComment;
	
	private IssueMultiPageEditor editorPart = null;

	/**
	 * 課題票を設定します。
	 */
	public SendReleaseIssueAction() {
		super();
		setImageDescriptor(BtsPlugin.getInstance().getImageDescriptor("send.gif"));
		setText(Messages.getString("SendReleaseIssueAction.0")); //$NON-NLS-1$
        setToolTipText(Messages.getString("SendReleaseIssueAction.1")); //$NON-NLS-1$
	}

	
	/**
	 * 課題票を配信します。
	 */
	public void run() {
		
        try {
    		super.run();

			// アクティブな課題票エディタを取得する
        	editorPart = BtsPlugin.getInstance().getActiveEditor();
    		if(editorPart == null) {
    			logger.debug(Messages.getString("SendReleaseIssueAction.2")); //$NON-NLS-1$
    			return;
    		}

	        // 保存・送信する課題票情報を取得
	        Issue newIssue = editorPart.getModifiedIssue();
			if(newIssue==null) {
				return;
			}
			
			IssueMultiPageEditorInput issueEditorInput 
				= (IssueMultiPageEditorInput)editorPart.getEditorInput();

			
	        // 確認ダイアログ表示
	        MessageDialog dialog = new MessageDialog
	        	( BtsPlugin.getInstance().getShell()
	        	, Messages.getString("SendReleaseIssueAction.3") //$NON-NLS-1$
	        	, null
	        	, Messages.getString("SendReleaseIssueAction.4") + newIssue.getTitle() + Messages.getString("SendReleaseIssueAction.5") //$NON-NLS-1$ //$NON-NLS-2$
	        	, MessageDialog.QUESTION
	        	, new String[]{Messages.getString("SendReleaseIssueAction.6"), Messages.getString("SendReleaseIssueAction.7")} //$NON-NLS-1$ //$NON-NLS-2$
	        	, 1
	        	);

	        // [配信する]以外を押下した場合は何もしない。
	        // ボタン押下の戻り値：[×ボタン]:-1、[配信する]:0、[キャンセル]:1
	        int res = dialog.open();
	        if(res != 0) {
	        	return;
	        }

	        // バージョンチェック
	        boolean verchk = checkVersion(newIssue);
	        if(!verchk) {
	        	return;
	        }
	        
	        // 削除予定の課題票を開いているエディタを閉じます。
	        closeIssueEditor(newIssue);
	        
	        // 課題票情報を更新・保存します。
	        saveIssue(newIssue, issueEditorInput);

			// 課題票エディタの表示を最新にします。
			editorPart.postIssueSave(newIssue);				//「概要タブ」を再表示
			editorPart.postCommentHistorySave(sendComment); //「コメントタブ」を再表示
			editorPart.postIssueHistorySave(sendHistory);	//「履歴タブ」を再表示

			// 送信前に添付ファイルの実体をファイルシステムから取得しインスタンスに設定します。
			AttachmentUtility utility = new AttachmentUtility();
			utility.read(newIssue.getAttachments());
			
			// 送信前に投票情報を取得します。
			VoteDAOFacade facade = new VoteDAOFacade();
			Vote[] votes = facade.get(newIssue.getFingerPrint());
			
	        // 課題票に色々な情報を積み込みます
			newIssue.setIssueHistories(sendHistory);
			newIssue.setCommentHistories(sendComment);
			newIssue.setVotes(votes);

	        // 課題票を送信します。
			sendIssue(newIssue);
	        
	        // 送信メッセージをローカルに保存します。
	        saveMessage(newIssue, IssueType.RELEASE_VALUE);

		} catch (Exception e) {
			String msg = Messages.getString("SendReleaseIssueAction.8"); //$NON-NLS-1$
			logger.fatal(msg, e);
			BtsPlugin.getInstance().error(msg, e);
		}
	}
	
	
	private CommentHistory getNewComment(Issue newIssue) {
		
		// 入力されたコメントを取得。
        String newComment = editorPart.getNewComment();
		if(newComment == null || newComment.trim().equals("")) {
	        // 入力が無い場合はnullを返す。
			return null;
		}
		
		CommentHistory commentHistory = new CommentHistory();
		// 入力したコメント
		commentHistory.setDescription(newComment);
		
		// FingerPrint
		commentHistory.setFingerPrint(newIssue.getFingerPrint());

		// 課題票種別
		commentHistory.setType(IssueType.RELEASE_VALUE);
	
		// 書き込み日時
		commentHistory.setUpdateDate(System.currentTimeMillis());
		
		// 更新者
		ProjectConfig config = CurrentProject.getInsance().getProjectConfig();
		commentHistory.setUpdateUser(config.getUserName());
		
		return commentHistory;
	}


	/**
	 * エディタを閉じます。<br>
	 * DRAFT→RELEASEとするとき、RELEASE課題票を削除し、DRAFT課題票をRELEASE版にアップデートします。
	 * 削除する予定のRelease課題票がエディタで開かれているときには、そのエディタを自動的に閉じるます。
	 * 
	 * @param			issue				送信する課題票
	 */
	private void closeIssueEditor(Issue issue) {
	
		// 送信する課題票がDRAFTの場合のみ
		if( IssueType.DRAFT_VALUE.equals(issue.getType()) ) {
			// 同じFingerPrintを持つ課題票で、RELEASE版のものが開いていたら閉じます
			BtsPlugin.getInstance().closeIssueEditor(issue.getFingerPrint(), IssueType.RELEASE_VALUE);
		}
	}
	
	
	/**
	 * 課題票のバージョンチェックを行います。
	 * 送信しようとしている課題票のバージョンと、DB上の課題票のバージョンが違う場合は送信を行えません。
	 * 
	 * @param			oldIssue			送信しようとしている課題票
	 * @return			true--バージョンチェックOK。false--バージョンチェックNG
	 */
	private boolean checkVersion(Issue oldIssue) {
		int myVer = oldIssue.getVersion().intValue();
		
		// 課題票のバージョンが０の時は、チェックする必要が無い。
		if(myVer==0) {
			return true;
		}

		// DBから当該課題票の最新バージョンを取得します。
		ModifyIssueFacade facade = new ModifyIssueFacade();
		latestIssue = facade.getByFingerPrint(oldIssue.getFingerPrint(), IssueType.RELEASE_VALUE);
		if(latestIssue==null) {
			return true;
		}
		int latestVer = latestIssue.getVersion().intValue();

		// 送信しようとしている課題票のバージョンと、DB上の課題票のバージョンが違う場合。
		// 警告ダイアログを表示してfalseを返す。
		if(latestVer != myVer) {
			MessageDialog.openWarning
				( BtsPlugin.getInstance().getShell()
				, Messages.getString("SendReleaseIssueAction.9") //$NON-NLS-1$
				, Messages.getString("SendReleaseIssueAction.10") //$NON-NLS-1$
				  + Messages.getString("SendReleaseIssueAction.11") //$NON-NLS-1$
				  + Messages.getString("SendReleaseIssueAction.12")); //$NON-NLS-1$

			return false;
		}
		return true;
	}
	
	
	/**
	 * 課題票を保存します。
	 * 
	 * @param		newIssue			保存する課題票
	 * @param		issueEditorInput	開いている課題票エディタが持つEditorInputインスタンス
	 * @throws		BtsException	
	 */
	private void saveIssue(Issue newIssue, IssueMultiPageEditorInput issueEditorInput) throws BtsException {
 
		//******************  課題票情報をリリース版情報に更新  ********************
    	// 送信する課題票は種別を「正規版」に設定する
    	String typeUpdateBefore = newIssue.getType();
		newIssue.setType(IssueType.RELEASE_VALUE);
    	
		// 更新情報を更新
    	if(newIssue.getVersion().intValue() ==0) {
        	newIssue.setCreateUser(CurrentProject.getInsance().getProjectConfig().getUserName());
        	newIssue.setCreateDate(System.currentTimeMillis());
    	}
    	newIssue.setUpdateDate(System.currentTimeMillis());
		
		// バージョン番号をインクリメントする
		newIssue.setVersion(newIssue.getVersion().intValue() +1);

		// 課題票の添付ファイル情報を更新する
		Attachment[] attachments = newIssue.getAttachments();
		if(attachments != null) {
	       	for (int i = 0; i < attachments.length; i++) {
	       		// 最初の起票時はFingerPrintが設定されてないので、ここで設定する
				attachments[i].setFingerPrint(newIssue.getFingerPrint());
				// 課題票種別を「RELEASE」に設定する
				attachments[i].setType(IssueType.RELEASE_VALUE);
			}
		}
		//***************************************************************************
		
		// 更新前課題票を取得する。
		Issue oldIssue = issueEditorInput.getIssue();

		// 新規コメントを取得する
		CommentHistory newComment = getNewComment(newIssue);
		
		// 更新履歴を取得する。
		IssueHistory[] histories = getIssueHistory(newIssue, newComment, issueEditorInput);
		
		// 「新規登録」or「更新」のフラグを取得する
		int mode = issueEditorInput.getOpenMode();

		// 課題票更新時に削除された課題票情報のリストを取得
		ArrayList oldAttachList = issueEditorInput.getSavedAttachment();
		ArrayList newAttachList = newIssue.getAttachmentList();
		Attachment[] deleteList = IssueUtility.getAttachmentDeleteList(oldAttachList, newAttachList);

		// 課題票とその添付と履歴を保存、戻り値に保存したときのアクションを含めた履歴を取得する。
		SendIssueFacade facade = new SendIssueFacade();
		
		// 「新規登録」or「更新」で処理を分ける
		if(mode == IssueMultiPageEditorInput.OPEN_NEW_ISSUE) {

			// 課題票を新規登録
			facade.addIssue(newIssue, histories, newComment);

			// 画面表示を更新
			java.util.List listeners = BtsPlugin.getInstance().getListeners();
			for (int idx = 0; idx < listeners.size(); idx++) {
				IBtsListener listnener = (IBtsListener)listeners.get(idx);
				listnener.addIssue(newIssue);
			}


		} else if(mode == IssueMultiPageEditorInput.OPEN_EXIST_ISSUE) {
	    	
			// 更新元の課題票種別により処理を分ける
			if(IssueType.RELEASE_VALUE.equals(typeUpdateBefore)) {
				//### CASE1:RELEASE → RELEASE ###
				
				// 更新元の課題票がRELEASEの場合。更新した情報を保存する。
				facade.updateVersion(newIssue, histories, deleteList, newComment);
				
				// 画面表示を更新
				java.util.List listeners = BtsPlugin.getInstance().getListeners();
				for (int idx = 0; idx < listeners.size(); idx++) {
					// 新規課題票情報を更新する。
					IBtsListener listnener = (IBtsListener)listeners.get(idx);
					listnener.updateIssue(oldIssue, newIssue);
				}

			
			} else if (IssueType.DRAFT_VALUE.equals(typeUpdateBefore)) {
				//### CASE2:DRAFT → RELEASE ###
				
				// 更新元の課題票がDRAFT。更新した情報を保存 ＆ これをRELEASE版にする
				// 戻り値は、更新前のRelease版課題票
				Issue beforeUpdateReleaseIssue
					= facade.updateRelease(newIssue, histories, deleteList, newComment);	
				
				// ビュー表示更新
				java.util.List listeners = BtsPlugin.getInstance().getListeners();
				for (int idx = 0; idx < listeners.size(); idx++) {
					IBtsListener listnener = (IBtsListener)listeners.get(idx);

					// ※ビュー表示更新の処理はDBへのデータ操作とは処理が違うので注意。
					//  (1).DRAFT版の表示を削除
					//  (2).既存のRELEASE版の情報を更新情報で置き換える。
					listnener.deleteIssue(oldIssue);
					listnener.updateIssue(beforeUpdateReleaseIssue, newIssue);
				}
			}
		}
		
		// 更新履歴を取得
		this.sendHistory = facade.getAllHistory();
		// コメント履歴を取得
		this.sendComment = facade.getAllComment();
	}
	
	
	/**
	 * 課題票の更新履歴を取得します。
	 * 
	 * @param		newIssue			対象となる課題票
	 * @param		issueEditorInput	開いている課題票エディタが持つEditorInputインスタンス
	 * @return		更新履歴の配列
	 * @throws		BtsException	
	 */
	private IssueHistory[] getIssueHistory(Issue newIssue, CommentHistory newComment, IssueMultiPageEditorInput issueEditorInput) throws BtsException {
		Issue oldIssue = issueEditorInput.getIssue();
		
		IssueHistory[] histories = null;
		
		if(oldIssue.getVersion().intValue() == 0) {
			
			// 新規作成の履歴生成
			histories = IssueUtility.createInitialHistory(newIssue);
		
		} else if( oldIssue.getVersion().intValue() != 0 && IssueType.RELEASE_VALUE.equals(oldIssue.getType()) ) {
			//### RELEASE版 →RELEASE版 ###
			
			// 念のため更新前課題票には保管しておいた課題票情報を設定しておく
			ArrayList savedAttachment = issueEditorInput.getSavedAttachment();
			oldIssue.setAttachmentList(savedAttachment);
				
			// 入力前と入力後を比較して履歴生成
			histories = IssueUtility.createIssueHistory(oldIssue, newIssue, newComment);
		
		} else if( oldIssue.getVersion().intValue() != 0 && IssueType.DRAFT_VALUE.equals(oldIssue.getType()) ) {
			//### DRAFT版 →RELEASE版 ###

			// 入力後とDB上のRELESE版の最新課題票を比較して履歴生成
			histories = IssueUtility.createIssueHistory(latestIssue, newIssue, newComment);
		}
		return histories;
	}
	
	
	/**
	 * 指定の課題票を送信します。
	 * 
	 * @param		issue			対象となる課題票
	 */
	protected void sendIssue(Issue issue) {

		// 課題票をメール送信
        sendMail(issue);

		// 送信は親クラスを使う
		super.sendIssue(issue);
		
		// 配信結果通知ダイアログを表示
    	ReceiveUserDialog userDialog = new ReceiveUserDialog(issue.getTitle());
		userDialog.setClientInfo(sendUserList);
        int res = userDialog.open();

        // 「課題票画面を閉じる」を選択した場合、配信した課題票ビューを閉じる
        if(res == 0) {
        	BtsPlugin.getInstance().closeEditor(editorPart, false);
        }
	}
	
	
//	=============================================================================================
//	================================   ここからインナークラス   =================================
//	=============================================================================================
	
	/**
	 * 配信結果通知ダイアログ。
	 * クライアント情報を表示するダイアログです。
	 * 
	 * @author		<A href="mailto:m_sugitou@valtech.jp">$Author$</A>
	 * @version		$Rev$
	 */
	private class ReceiveUserDialog extends MessageDialog {
	
		/** ユーザ情報 */
		private ArrayList clientList;
		
		/**
		 * ダイアログを生成します。
		 * 
		 * @param		issueTitle		ダイアログに表示する課題票タイトル
		 */
		public ReceiveUserDialog(String issueTitle) {
        	super( BtsPlugin.getInstance().getShell()
    	        	, Messages.getString("SendReleaseIssueAction.13") //$NON-NLS-1$
    	        	, null
    	        	, Messages.getString("SendReleaseIssueAction.14") + issueTitle + Messages.getString("SendReleaseIssueAction.15") //$NON-NLS-1$ //$NON-NLS-2$
    	        	, MessageDialog.INFORMATION
    	        	, new String[]{Messages.getString("SendReleaseIssueAction.16"), Messages.getString("SendReleaseIssueAction.17")} //$NON-NLS-1$ //$NON-NLS-2$
    	        	, 0
    	        	);
		}

		/**
		 * クライアント情報を設定します。
		 * 
		 * @param		clientInfo		クライアント情報
		 */
		public void setClientInfo(ArrayList clientList) {
			this.clientList = clientList;
		}
		
		
		/**
		 * 配信したユーザ情報を表示します。
		 * 
		 * @param		parent				親コンポジット
		 * @see 		org.eclipse.jface.dialogs.MessageDialog#createCustomArea(org.eclipse.swt.widgets.Composite)
		 */
	    protected Control createCustomArea(Composite parent) {

	    	// このエリアのコンポジット生成。
	        Composite thisArea = new Composite(parent, SWT.NONE);
	        GridLayout gridLayout = new GridLayout();
	        gridLayout.marginLeft = 40;
	        thisArea.setLayout(gridLayout);

	        // リストコントロールにクライアント情報を表示する
	        List memberList = new List (thisArea, SWT.BORDER | SWT.MULTI | SWT.V_SCROLL);

	        if(clientList ==null || clientList.size()==0 ) {
				memberList.add(Messages.getString("SendReleaseIssueAction.18")); //$NON-NLS-1$
	        } else {
		        for (int i = 0; i < clientList.size(); i++) {
		        	
		        	ClientInfo clientInfo = (ClientInfo)clientList.get(i);
		        	// ユーザ名取得
					String userName = clientInfo.getClientName();
					// アドレス取得
					String address = BTSUtility.rightPad("(" + clientInfo.getClientAddress() + ")", 19);
					
					memberList.add(address + userName );
				}
	        }

	        // リストコントロールのレイアウト設定
	        GridData gridData = new GridData();
	        gridData.heightHint = 60;
	        gridData.widthHint = 280;
	        memberList.setLayoutData(gridData);

	        
	        Label label = new Label(thisArea, SWT.LEFT);
	        label.setText(Messages.getString("SendReleaseIssueAction.19")); //$NON-NLS-1$
	        
	        
	        return thisArea;
	    }
		
	}

}
