/***********************************************************************
 * Copyright(C) 2006 Valtech Co.,Ltd.
 * All Rights Reserved. This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.opensource.org/licenses/cpl.php
 ***********************************************************************/
package jp.valtech.bts.ui.editor;

import java.util.ArrayList;

import jp.valtech.bts.data.Issue;

import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.ui.IEditorInput;
import org.eclipse.ui.IPersistableElement;

/**
 * 課題票エディタ用の{@link org.eclipse.ui.IEditorInput}実装クラス
 * 
 * @author		<A href="mailto:m_sugitou@valtech.jp">M.Sugito</A>
 * @version	Ver.0.8
 */
public class IssueMultiPageEditorInput implements IEditorInput {

	/** 新規に課題票を開く */
	public static final int OPEN_NEW_ISSUE = 0;
	
	/** 既存の課題票を開く */
	public static final int OPEN_EXIST_ISSUE = 1;

	/** 課題票のオープンモード(初期値：新規) */
	private int	openMode = OPEN_NEW_ISSUE;

	/** 課題票エディタ表示用のIssueインスタンス */
	private IssueDisplayValue issueDisplayValue = new IssueDisplayValue();
	
	/** オープンする課題票 */
	private Issue issue;
	
	/** 課題票に設定された添付ファイル。ここではクローンを保持する。 */
	private ArrayList saveAttachment ;
	
	/**
	 * 何もしない。
	 */
	public IssueMultiPageEditorInput() {
		super();
	}

	  
	/**
	 * 課題票のオープンモード取得。
	 * 
	 * @return			課題票のオープンモード
	 */
	public int getOpenMode() {
		return openMode;
	}
	/**
	 * 課題票のオープンモードを設定します。
	 * 
	 * @param			openMode		課題票のオープンモード
	 */
	public void setOpenMode(int openMode) {
		this.openMode = openMode;
	}
	
	
	/**
	 * オープンする課題票を設定します。
	 * 
	 * @param			issue				オープンする課題票
	 */
	public void setIssue(Issue issue) {

		// 課題票を設定
		this.issue = issue;
		
		// 課題票添付ファイルのクローンを保管しておく
		ArrayList attachmentList = issue.getAttachmentList();
		if(attachmentList == null) {
			this.saveAttachment = new ArrayList();
		}
		this.saveAttachment = (ArrayList)attachmentList.clone();
		
		// 課題票エディタに表示する形式にして保持する
		issueDisplayValue.setIssue(issue) ;
	}
	
	/**
	 * 保持している情報をリセットします。
	 * 
	 * 更新モードにリセットします。<BR>
	 * データを最新のものに置き換えます。<BR>
	 * 
	 * @param issue
	 */
	public void resetIssue(Issue issue) {
		this.openMode = OPEN_EXIST_ISSUE;
		setIssue(issue);
	}
	
	/**
	 * 課題票を取得します。
	 * 
	 * @return			課題票
	 */
	public Issue getIssue() {
		return this.issue;
	}
		

	/**
	 * 添付ファイルを取得します。
	 * この添付ファイルは、課題票エディタを開いた時点での添付ファイル情報です。
	 * 課題票更新時の比較用に使います。
	 *  
	 * @return			添付ファイル
	 */
	public ArrayList getSavedAttachment() {
		return this.saveAttachment;
	}
	
	/**
	 * 課題票エディタ表示用のIssueインスタンス取得。
	 * 
	 * @return			課題票エディタ表示用のIssueインスタンス
	 */
	public IssueDisplayValue getIssueDisplayValue() {
		return issueDisplayValue;
	}

	
	/**
	 * オブジェクトの比較をします。
	 * 
	 * 課題票エディタオープン時にこのメソッドが呼ばれます。<br>
	 * <code>return=true</code> の場合は当該課題票のエディタが既に開いていると判断され、そこにフォーカスを移します。<br>
	 * <code>return=false</code>の場合は当該課題票のエディタがまだ開いていないと判断され、新しく課題票エディタを開きます。<br>
	 * 
	 * @param			o			当インスタンスと比較するインスタンス
	 * @true			true--既存エディタにフォーカスを移す。false--新規にエディタを開く
	 * @see				Object#equals(java.lang.Object)
	 */
	public boolean equals(Object o) {

		// 自分自身の場合はtrueを返す。
		if(o == this) {
			return true;
		}
		
		// issueがnullの場合はfalseを返す。
		if(issue == null) {
			return false;
		}

		// IssueMultiPageEditorInputインスタンスかどうかを確認
		if(o instanceof IssueMultiPageEditorInput) {
			
			IssueMultiPageEditorInput that = (IssueMultiPageEditorInput)o;
			
			// 課題票を比較する。同じであればtrueを返し、それ以外はfalseを返す。
			return this.issue.equals(that.getIssue()); 
		}
		return false;
	}
	
	
	
	/* (非 Javadoc)
	 * @see org.eclipse.ui.IEditorInput#exists()
	 */
	public boolean exists() {
		return false;
	}

	
	/**
	 * 何もしない。
	 */
	public ImageDescriptor getImageDescriptor() {
		return null;
	}

	
	/**
	 * 課題票エディタのタブに表示する文字列を返します。
	 * 
	 * 既存の課題票の場合は「課題票のタイトル」を返します。<BR>
	 * 新規課題票の場合は"新規課題票"の文字列を返します。<BR>
	 * 
	 * @return 				課題票エディタのタブに表示する文字列
	 */
	public String getName() {

		// 既存の課題票の場合は「課題票のタイトル」を表示する 
		if(openMode == OPEN_EXIST_ISSUE) {
			return issueDisplayValue.getTitle();
		}
		return Messages.getString("IssueMultiPageEditorInput.0"); //$NON-NLS-1$
	}

	/* (非 Javadoc)
	 * @see org.eclipse.ui.IEditorInput#getPersistable()
	 */
	public IPersistableElement getPersistable() {
		return null;
	}

	/**
	 * 課題票エディタのツールチップに表示する文字列を返します。
	 * 
	 * 既存の課題票の場合は「「課題票ID：課題票タイトル」」を返します。<BR>
	 * 新規課題票の場合は"新規課題票"の文字列を返します。<BR>
	 * 
	 * @return 				課題票エディタのツールチップに表示する文字列
	 */
	public String getToolTipText() {
		// 既存の課題票の場合は「課題票ID：課題票タイトル」を表示する 
		if(openMode == OPEN_EXIST_ISSUE) {
			return issueDisplayValue.getDisplayIssueID() + ":" + issueDisplayValue.getTitle();
		}
		return Messages.getString("IssueMultiPageEditorInput.1"); //$NON-NLS-1$
	}

	/* (非 Javadoc)
	 * @see org.eclipse.core.runtime.IAdaptable#getAdapter(java.lang.Class)
	 */
	public Object getAdapter(Class adapter) {
		return null;
	}
}
