/***********************************************************************
 * Copyright(C) 2006 Valtech Co.,Ltd.
 * All Rights Reserved. This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.opensource.org/licenses/cpl.php
 ***********************************************************************/
package jp.valtech.bts.ui.navigator;


import java.io.File;

import jp.valtech.bts.command.client.Join;
import jp.valtech.bts.data.CurrentProject;
import jp.valtech.bts.data.NetworkConfig;
import jp.valtech.bts.facade.DatabaseInitializer;
import jp.valtech.bts.network.ClientInfo;
import jp.valtech.bts.network.ClientList;
import jp.valtech.bts.ui.BtsPlugin;
import jp.valtech.bts.util.BTSUtility;
import jp.valtech.bts.util.Logging;

import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.jface.dialogs.TitleAreaDialog;
import org.eclipse.jface.resource.JFaceResources;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.graphics.Font;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.TabFolder;
import org.eclipse.swt.widgets.TabItem;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.PlatformUI;

/**
 * プロジェクト情報を設定するダイアログです。
 * {@link jp.valtech.bts.ui.navigator.NavigatorView ナビゲーションビュー}から呼ばれるダイアログです。
 * 
 * @author		<A href="mailto:m_sugitou@valtech.jp">M.Sugito</A>
 * @version	Ver.0.8
 */
public class ProjectConfigDialog extends TitleAreaDialog implements Logging {

	/** タブインデックス：「全般」タブ */
	public static final int TAB_INDEX_GENERAL = 0;
	/** タブインデックス：「同期」タブ */
	public static final int TAB_INDEX_SYNC = 1;
	/** タブインデックス：「課題票」タブ */
	public static final int TAB_INDEX_ISSUE = 2;
	/** タブインデックス：「メール」タブ */
	public static final int TAB_INDEX_MAIL = 3;
	/** オープン時のタブインデックス */
	private int tabIndex = TAB_INDEX_GENERAL;
	
	/** 基本レイアウトに使う：テキストボックスのサイズ(大) */
	private GridData textSizeL;
	/** 基本レイアウトに使う：テキストボックスのサイズ(小) */
    private GridData textSizeSS;
	/** 基本レイアウトに使う：ラベルのサイズ(中) */
    private GridData labelSizeM;
	/** 基本レイアウトに使う：ラベルのサイズ(小) */
    private GridData labelSizeS;

    /** 「プロジェクト名」用テキストボックス */
    private Text projectNameTxt;
    /** 「ユーザ名」用テキストボックス */
    private  Text userNameTxt;
    /** 「SSID」用テキストボックス */
    private Text ssidTxt;

    /** カテゴリ入力のテキストボックス */
    private Text categoryTxt;
    /** 優先度入力のテキストボックス */
    private Text priorityTxt;
    /** 担当者入力のテキストボックス */
    private Text assignedTxt;
    /** 投票数入力のテキストボックス */
    private Text voteTxt;
    /** 投票数入力のチェックボックス */
    private Button voteChk;
    /** 完了期限入力のテキストボックス */
    private Text untilDeadLineTxt;
    /** 完了期限入力のチェックボックス */
    private Button untilDeadLineChk;
   
    /** メール通知のチェックボックス */
    private Button mailChk;
    /** SMTPサーバ入力のテキストボックス */
    private Text smtpTxt;
    /** 差出人アドレスのテキストボックス */
    private Text senderTxt;
    /** 送信先アドレスのテキストボックス */
    private Text recipientTxt;
    /** タイトルプレフィックスのテキストボックス */
    private Text prefixTxt;
    
    /** 固定端末との同期処理用チェックボックス */
    private Button fixedSyncChk;
    /** 固定端末のアドレステキストボックス（1番目） */
    private Text address1;
    /** 固定端末のアドレステキストボックス（2番目） */
    private Text address2;
    /** 固定端末のアドレステキストボックス（3番目） */
    private Text address3;
    /** 固定端末のアドレステキストボックス（4番目） */
    private Text address4;
    /** ポート番号テキストボックス */
    private Text port;
    /** 自動同期処理用チェックボックス */
    private Button autoSyncChk;
    /** 「リモート端末を指定して同期」用ラジオボックス */
    private Button appointSyncRdi;
    /** 「リモート端末を指定して同期」用ラジオボックス */
    private Label appointSyncLabel;
    /** 「同期」用ラジオボックス */
    private Button normalSyncRdi;
    /** 「同期」用ラベル */
    private Label normalSyncLabel;
    

    /** 太字を設定するときに使う */
    private Font boldFont = JFaceResources.getFontRegistry().getBold(JFaceResources.DEFAULT_FONT);
    
	/** 旧プロジェクト情報 */
	private ProjectConfig oldConfig;
	

	/**
	 * SSIDと旧プロジェクト情報を設定します。
	 * 
	 * @param		parentShell		ラルクアンシェル
	 * @param		ssid			プロジェクトのSSID
	 */
	public ProjectConfigDialog(Shell parentShell, String ssid){
		super(parentShell);
		this.oldConfig = ProjectConfigManager.getProjectConfig(ssid);
	}
	
	/**
	 * SSIDと旧プロジェクト情報を設定します。
	 * 
	 * @param		parentShell		ラルクアンシェル
	 * @param		ssid			プロジェクトのSSID
	 * @param		openTabIndex	オープン時にアクティブにするタブ
	 */
	public ProjectConfigDialog(Shell parentShell, String ssid, int openTabIndex){
		super(parentShell);
		this.oldConfig = ProjectConfigManager.getProjectConfig(ssid);
		this.tabIndex = openTabIndex;
	}

	/**
	 * SSIDと旧プロジェクト情報を設定します。
	 * 
	 * @param		parentShell		ラルクアンシェル
	 * @param		ssid			プロジェクトのSSID
	 */
	public ProjectConfigDialog(Shell parentShell){
		super(parentShell);
	}

	
	/**
     * 当ダイアログの設定します。タイトルを設定します。
     * 
     * @param			newShell			該ダイアログのシェル
     */
    protected void configureShell(Shell newShell) {
        super.configureShell(newShell);
        newShell.setText(Messages.getString("ProjectConfigDialog.0")); //$NON-NLS-1$
        PlatformUI.getWorkbench().getHelpSystem().setHelp(newShell, "jp.valtech.bts.help.projectContext");
    }


    /**
     * 送信メッセージ作成用のダイアログを生成します。
     * 
     * @param			parent					親コンポジット
     * @return			当ダイアログのコントロールインスタンス
     * @see				TitleAreaDialog#createDialogArea(org.eclipse.swt.widgets.Composite)
     */
    protected Control createDialogArea(Composite parent) {
        Composite composite = (Composite) super.createDialogArea(parent);
        
        try {
        	// 共通レイアウトを生成
        	createCommonLayout();
        	
        	// タイトルエリアのタイトル設定
        	if(oldConfig == null){
        		setTitle(Messages.getString("ProjectConfigDialog.1")); //$NON-NLS-1$
        		setMessage(Messages.getString("ProjectConfigDialog.2")); //$NON-NLS-1$
        	} else {
        		setTitle(Messages.getString("ProjectConfigDialog.3")); //$NON-NLS-1$
        		setMessage(Messages.getString("ProjectConfigDialog.4")); //$NON-NLS-1$
        	}
        	// タイトルエリアのアイコン設定
        	setTitleImage(BtsPlugin.getInstance().getImage("butterfly.gif"));
        	
        	// タブの生成
        	TabFolder tabFolder = new TabFolder (composite, SWT.NONE);
        	tabFolder.setLayoutData(new GridData(GridData.FILL_BOTH));
        	
        	// １つ目のタブ（基本設定）を生成
        	createGeneraTab(tabFolder);
        	// ２つ目のタブ（同期）を生成
        	createSyncTab(tabFolder);
        	// ３つ目のタブ（課題票）を生成
        	createIssueTab(tabFolder);
        	// ４つ目のタブ（メール通知）を生成
        	createMailTab(tabFolder);
        	
        	// オープン時にアクティブにするタブを指定。
        	tabFolder.setSelection(tabIndex);
        	
        }catch (Exception e) {
			String msg = Messages.getString("ProjectConfigDialog.5"); //$NON-NLS-1$
			logger.fatal(msg, e);
			BtsPlugin.getInstance().error(msg, e);
		}
        return composite;
    }
    
    /**
     * この画面で使う共通レイアウトを生成します。
     *
     */
     private void createCommonLayout() {
 		textSizeL = new GridData(GridData.FILL); 
 		textSizeL.widthHint = 290;

 		textSizeSS = new GridData(GridData.FILL); 
 		textSizeSS.widthHint = 30;

 		labelSizeM = new GridData(GridData.FILL); 
 		labelSizeM.widthHint = 70;
 
 		labelSizeS = new GridData(GridData.FILL); 
 		labelSizeS.widthHint = 60;
     }
     
     
    /**
     * 基本設定のタブを生成します。
     * 
     * @param			tabFolder			親タブ
     * @throws			Exception
     */
	private void createGeneraTab(TabFolder tabFolder) throws Exception {

		// 基本設定のタブを生成
		TabItem tab = new TabItem(tabFolder, SWT.NONE);
		tab.setText(Messages.getString("ProjectConfigDialog.15")); //$NON-NLS-1$

		// レイアウト設定
		Composite generalTab = new Composite(tabFolder, SWT.NULL);
		generalTab.setLayout(new GridLayout());

		// タブに部品を貼り付けていく
		createGeneralTabHeaderArea(generalTab);
		createProjectNameArea(generalTab);
		createUserNameArea(generalTab);
		createSSIDArea(generalTab);
		createNotesArea(generalTab);
		tab.setControl(generalTab);
	}

    /**
	 * 課題票設定のタブを生成します。
	 * 
	 * @param			tabFolder			親タブ
	 * @throws			Exception
	 */
    private void createIssueTab(TabFolder tabFolder) throws Exception {
    	// 課題票タブを生成
		TabItem tab = new TabItem (tabFolder, SWT.NONE);
    	tab.setText(Messages.getString("ProjectConfigDialog.6")); //$NON-NLS-1$

    	// レイアウト設定
    	Composite issueTab = new Composite(tabFolder, SWT.NULL);
    	issueTab.setLayout(new GridLayout());
    	
    	// 部品を貼り付けていく
    	createIssueTabHeaderArea(issueTab);
    	createCategoryArea(issueTab);
    	createPriorityArea(issueTab);
    	createAssignedArea(issueTab);
    	createVoteArea(issueTab);
    	createUntilDeadLineArea(issueTab);
        tab.setControl(issueTab);
    }
    
    /**
     * メール通知タブを生成します。
     * 
     * @param			tabFolder			親タブ
     * @throws			Exception
     */
    private void createMailTab(TabFolder tabFolder) throws Exception {
    	
    	// メールタブを生成
    	TabItem tab = new TabItem(tabFolder, SWT.NONE);
    	tab.setText(Messages.getString("ProjectConfigDialog.7")); //$NON-NLS-1$
    	
    	// レイアウト設定
    	Composite mailTab = new Composite(tabFolder, SWT.NULL);
    	mailTab.setLayout(new GridLayout());
    	
    	// 部品を貼り付けていく
    	createMailTabHeaderArea(mailTab);
    	createMailCheckArea(mailTab);
    	createSmtpArea(mailTab);
    	createSenderAddressArea(mailTab);
    	createRecipientAddressArea(mailTab);
    	createPrefixArea(mailTab);
    	tab.setControl(mailTab);
    	
    }
    
    /**
	 * 課題票設定のタブを生成します。
	 * 
	 * @param			tabFolder			親タブ
	 * @throws			Exception
	 */
    private void createSyncTab(TabFolder tabFolder) throws Exception {
    	// 同期タブを生成
		TabItem tab = new TabItem (tabFolder, SWT.NONE);
    	tab.setText(Messages.getString("ProjectConfigDialog.43")); //$NON-NLS-1$

    	// レイアウト設定
    	Composite syncTab = new Composite(tabFolder, SWT.NULL);
    	syncTab.setLayout(new GridLayout());
    	
    	// 部品を貼り付けていく
    	createSyncTabHeaderArea(syncTab);
    	createFixedSyncArea(syncTab);
    	createAutoSyncArea(syncTab);
    	createNotesArea(syncTab);
        tab.setControl(syncTab);
    }
    
    
    /**
     * プロジェクト情報を入力するタブのヘッダエリアを生成します。
     * 
     * @param		parent				親コンポジット
     * @throws		Exception
     */
    private void createGeneralTabHeaderArea(Composite parent) throws Exception {
    	
    	// この部品のコンポジット生成
    	Composite thisArea = new Composite(parent, SWT.NONE);
        thisArea.setLayout(new GridLayout());
    	
        // ラベル生成
        Label label = new Label(thisArea, SWT.NONE);
        label.setText(Messages.getString("ProjectConfigDialog.30")); //$NON-NLS-1$
        label.setFont(boldFont);
    }

    
    /**
     * プロジェクト名を入力するエリアの部品を生成します。
     * 
     * @param		parent				親コンポジット
     * @throws		Exception
     */
    private void createProjectNameArea(Composite parent) throws Exception {

    	// この部品のコンポジット生成
    	Composite thisArea = new Composite(parent, SWT.NONE);

    	// レイアウト設定。2列のグリッドレイアウトを使う
        GridLayout gridLayout = new GridLayout();
        gridLayout.numColumns = 2;
        thisArea.setLayout(gridLayout);

        // プロジェクト名のラベル生成
        Label label = new Label(thisArea, SWT.RIGHT);
        label.setText(Messages.getString("ProjectConfigDialog.31")); //$NON-NLS-1$
        label.setLayoutData(labelSizeM);
        
        // プロジェクト名入力用テキストの生成
        projectNameTxt = new Text(thisArea, SWT.BORDER );
        projectNameTxt.setLayoutData(textSizeL);
        
        // プロジェクト名の文字数制限（20文字）
        projectNameTxt.setTextLimit(20);
        
        // プロジェクト情報の編集時に既存の設定を反映させる
        if(oldConfig != null){
        	projectNameTxt.setText(oldConfig.getProjectName());
        }
        
        // ModifyListenerの追加
        projectNameTxt.addModifyListener(new ModifyListener(){
			public void modifyText(ModifyEvent e) {
				// 入力項目のチェック
				okEnabled();
			}
        });

        // 左側列のダミー
        label = new Label(thisArea, SWT.NONE);
        label.setText("");

        // ラベル
        label = new Label(thisArea, SWT.NONE);
        label.setText(Messages.getString("ProjectConfigDialog.32")); //$NON-NLS-1$
        label.setAlignment(SWT.RIGHT);
    }
    
    
    /**
     * ユーザ名を入力するエリアの部品を生成します。
     * 
     * @param		parent				親コンポジット
     * @throws		Exception
     */
    private void createUserNameArea(Composite parent) throws Exception {

    	// この部品のコンポジット生成
    	Composite thisArea = new Composite(parent, SWT.NONE);

    	// レイアウト設定。2列のグリッドレイアウトを使う
        GridLayout gridLayout = new GridLayout();
        gridLayout.numColumns = 2;
        thisArea.setLayout(gridLayout);

        // ユーザ名ラベル生成
        Label label = new Label(thisArea, SWT.RIGHT);
        label.setText(Messages.getString("ProjectConfigDialog.33")); //$NON-NLS-1$
        label.setLayoutData(labelSizeM);
        
        // ユーザ名入力用テキストの生成
        userNameTxt = new Text(thisArea, SWT.BORDER );
        userNameTxt.setLayoutData(textSizeL);
        
        // ユーザ名の文字数制限（20文字）
        userNameTxt.setTextLimit(20);
        
        // プロジェクト情報の編集時に既存の設定を反映させる
        if(oldConfig != null){
        	userNameTxt.setText(oldConfig.getUserName());
        }
        
        // ModifyListenerの追加
        userNameTxt.addModifyListener(new ModifyListener(){
			public void modifyText(ModifyEvent e) {
				// 入力項目のチェック
				okEnabled();
			}
        });

        // 左側列のダミー
        label = new Label(thisArea, SWT.NONE);
        label.setText("");

        // ラベル
        label = new Label(thisArea, SWT.NONE);
        label.setText(Messages.getString("ProjectConfigDialog.34")); //$NON-NLS-1$
        label.setAlignment(SWT.RIGHT);
    }

    
    /**
     * SSIDを入力するエリアの部品を生成します。
     * 
     * @param		parent				親コンポジット
     * @throws		Exception
     */
    private void createSSIDArea(Composite parent) throws Exception {

    	// この部品のコンポジット生成
    	Composite thisArea = new Composite(parent, SWT.NONE);

    	// レイアウト設定。2列のグリッドレイアウトを使う
        GridLayout gridLayout = new GridLayout();
        gridLayout.numColumns = 2;
        thisArea.setLayout(gridLayout);

        // SSIDラベル生成
        Label label = new Label(thisArea, SWT.RIGHT);
        label.setText(Messages.getString("ProjectConfigDialog.35")); //$NON-NLS-1$
        label.setLayoutData(labelSizeM);
        
        // SSID入力用テキストの生成
        ssidTxt = new Text(thisArea, SWT.BORDER );
        ssidTxt.setLayoutData(textSizeL);
        
        // SSIDの文字数制限（20文字）
        ssidTxt.setTextLimit(20);
        
        // プロジェクト情報の編集時に既存の設定を反映させる
        if(oldConfig != null){
        	ssidTxt.setText(oldConfig.getSsid());
        	// SSIDを非活性にする
        	ssidTxt.setEnabled(false);
        }
        
        // ModifyListenerの追加
        ssidTxt.addModifyListener(new ModifyListener(){
			public void modifyText(ModifyEvent e) {
				// 入力項目のチェック
				okEnabled();
			}
        });
    	
        // 左側列のダミー
        label = new Label(thisArea, SWT.NONE);
        label.setText("");

        // ラベル
        label = new Label(thisArea, SWT.NONE);
        label.setText(Messages.getString("ProjectConfigDialog.36")); //$NON-NLS-1$
        label.setAlignment(SWT.RIGHT);
    }
        
    /**
     * 注釈を表示します。
     * 
     * @param		parent				親コンポジット
     * @throws		Exception
     */
    private void createNotesArea(Composite parent) throws Exception {
    	
    	// この部品のコンポジット生成
    	Composite thisArea = new Composite(parent, SWT.NONE);
        thisArea.setLayout(new GridLayout());
    	
        // ラベル生成
        Label label = new Label(thisArea, SWT.NONE);
        label.setText(Messages.getString("ProjectConfigDialog.40")); //$NON-NLS-1$
        label.setForeground(new Color(Display.getCurrent(), 49, 106, 197));

    }
    
    
    /**
     * 課題票タブのヘッダエリアを生成します。
     * 
     * @param		parent				親コンポジット
     * @throws		Exception
     */        
    private void createIssueTabHeaderArea(Composite parent) throws Exception {

    	// この部品のコンポジット生成
    	Composite thisArea = new Composite(parent, SWT.NONE);
        GridLayout gridLayout = new GridLayout();
        thisArea.setLayout(gridLayout);

        // ラベル
        Label label = new Label(thisArea, SWT.NONE);
        label.setText(Messages.getString("ProjectConfigDialog.29")); //$NON-NLS-1$
        label.setFont(boldFont);
    }
    
    /**
     * カテゴリのプルダウンを設定するエリアを生成します。
     * 
     * @param		parent				親コンポジット
     * @throws		Exception
     */        
    private void createCategoryArea(Composite parent) throws Exception {

    	// この部品のコンポジット生成
    	Composite thisArea = new Composite(parent, SWT.NONE);

    	// レイアウト設定。2列のグリッドレイアウトを使う
        GridLayout gridLayout = new GridLayout();
        gridLayout.numColumns = 2;
        thisArea.setLayout(gridLayout);

        // ラベル
        Label label = new Label(thisArea, SWT.NONE);
        label.setText(Messages.getString("ProjectConfigDialog.22") //$NON-NLS-1$
        		+ Messages.getString("ProjectConfigDialog.23")); //$NON-NLS-1$
        GridData data = new GridData();
        data.horizontalSpan = 2;
        label.setLayoutData(data);

        // ラベル生成
        label = new Label(thisArea, SWT.RIGHT);
        label.setText(Messages.getString("ProjectConfigDialog.24")); //$NON-NLS-1$
        label.setLayoutData(labelSizeS);
        
        // カテゴリ入力用テキストの生成
        categoryTxt = new Text(thisArea, SWT.BORDER );
        categoryTxt.setLayoutData(textSizeL);
        // デフォルト設定
        categoryTxt.setText(Messages.getString("ProjectConfigDialog.25")); //$NON-NLS-1$
        
        // プロジェクト情報の編集時に既存の設定を反映させる
        if(oldConfig != null){
        	categoryTxt.setText(oldConfig.getCategory());
        }
        
        // ModifyListenerの追加
        categoryTxt.addModifyListener(new ModifyListener(){
			public void modifyText(ModifyEvent e) {
				// 入力項目のチェック
				okEnabled();
			}
        });
        
    }
    
    /**
     * 優先度のプルダウンを設定するエリアを生成します。
     * 
     * @param		parent				親コンポジット
     * @throws		Exception
     */        
    private void createPriorityArea(Composite parent) throws Exception {

    	// この部品のコンポジット生成
    	Composite thisArea = new Composite(parent, SWT.NONE);

    	// レイアウト設定。2列のグリッドレイアウトを使う
        GridLayout gridLayout = new GridLayout();
        gridLayout.numColumns = 2;
        thisArea.setLayout(gridLayout);

        // ラベル生成
        Label label = new Label(thisArea, SWT.RIGHT);
        label.setText(Messages.getString("ProjectConfigDialog.26")); //$NON-NLS-1$
        label.setLayoutData(labelSizeS);
        
        // 優先度入力用テキストの生成
        priorityTxt = new Text(thisArea, SWT.BORDER );
        priorityTxt.setLayoutData(textSizeL);
        // デフォルト設定
        priorityTxt.setText(Messages.getString("ProjectConfigDialog.27")); //$NON-NLS-1$
        
        // プロジェクト情報の編集時に既存の設定を反映させる
        if(oldConfig != null){
        	priorityTxt.setText(oldConfig.getPriority());
        }
        
        // ModifyListenerの追加
        priorityTxt.addModifyListener(new ModifyListener(){
			public void modifyText(ModifyEvent e) {
				// 入力項目のチェック
				okEnabled();
			}        	
        });
        
    }
    
    /**
     * 担当者のプルダウンを設定するエリアを生成します。
     * 
     * @param		parent				親コンポジット
     * @throws		Exception
     */        
    private void createAssignedArea(Composite parent) throws Exception {

    	// この部品のコンポジット生成
    	Composite thisArea = new Composite(parent, SWT.NONE);

    	// レイアウト設定。2列のグリッドレイアウトを使う
        GridLayout gridLayout = new GridLayout();
        gridLayout.numColumns = 2;
        thisArea.setLayout(gridLayout);

        // ラベル生成
        Label label = new Label(thisArea, SWT.RIGHT);
        label.setText(Messages.getString("ProjectConfigDialog.28")); //$NON-NLS-1$
        label.setLayoutData(labelSizeS);
        
        // 優先度入力用テキストの生成
        assignedTxt = new Text(thisArea, SWT.BORDER );
        assignedTxt.setLayoutData(textSizeL);
        
        // プロジェクト情報の編集時に既存の設定を反映させる
        if(oldConfig != null){
        	assignedTxt.setText(oldConfig.getAssigned());
        }
        
        // ModifyListenerの追加
        assignedTxt.addModifyListener(new ModifyListener(){
			public void modifyText(ModifyEvent e) {
				// 入力項目のチェック
				okEnabled();
			}        	
        });
        
    }
    
    /**
     * 投票数を設定するエリアを生成します。
     * 
     * @param		parent				親コンポジット
     * @throws		Exception
     */        
    private void createVoteArea(Composite parent) throws Exception {

    	// この部品のコンポジット生成
    	Composite thisArea = new Composite(parent, SWT.NONE);

    	// レイアウト設定。2列のグリッドレイアウトを使う
        GridLayout gridLayout = new GridLayout();
        gridLayout.numColumns = 3;
        thisArea.setLayout(gridLayout);


        // チェックボックス
        voteChk = new Button(thisArea, SWT.CHECK);
        voteChk.setText(Messages.getString("ProjectConfigDialog.19")); //$NON-NLS-1$
        // デフォルト設定
        voteChk.setSelection(true);
        // SelectionListenerの追加
        voteChk.addSelectionListener(new SelectionAdapter(){
			public void widgetSelected(SelectionEvent e) {
				// 投票数テキストの活性化切換え
				if(voteChk.getSelection()){
					voteTxt.setEditable(true);
				} else {
					voteTxt.setEditable(false);
				}
				// 入力項目のチェック
				okEnabled();
			}
        });
        
    	// レイアウト設定
        GridData data = new GridData();
        data.horizontalSpan = 3;
        voteChk.setLayoutData(data);
        
        // ラベル
        Label label = new Label(thisArea, SWT.NONE);
        label.setText(Messages.getString("ProjectConfigDialog.20")); //$NON-NLS-1$
        GridData indent = new GridData();
        indent.horizontalIndent = 20;
        label.setLayoutData(indent);

        
        // 投票数設定用テキストの生成
        voteTxt = new Text(thisArea, SWT.BORDER );
        voteTxt.setLayoutData(textSizeSS);
        // デフォルト設定
        voteTxt.setEditable(true);
        voteTxt.setText("20");
        
        // プロジェクト情報の編集時に既存の設定を反映させる
        if(oldConfig != null){
        	// チェックボックスの反映
        	voteChk.setSelection(oldConfig.isVoteCheck());
        	// テキストの反映
        	voteTxt.setText(oldConfig.getVote());
        	// テキスト活性化の反映
        	if(oldConfig.isVoteCheck()){
        		voteTxt.setEditable(true);
        	} else {
        		voteTxt.setEditable(false);
        	}
        }
        
        // ModifyListenerの追加
        voteTxt.addModifyListener(new ModifyListener(){
			public void modifyText(ModifyEvent e) {
				// 入力項目のチェック
				okEnabled();
			}
        });

        // ラベル
        label = new Label(thisArea, SWT.NONE);
        label.setText(Messages.getString("ProjectConfigDialog.21")); //$NON-NLS-1$
        
    }
    
    /**
     * 完了期限が近い課題票を設定するエリアを生成します。
     * 
     * @param		parent				親コンポジット
     * @throws		Exception
     */        
    private void createUntilDeadLineArea(Composite parent) throws Exception {

    	// この部品のコンポジット生成
    	Composite thisArea = new Composite(parent, SWT.NONE);

    	// レイアウト設定。3列のグリッドレイアウトを使う
        GridLayout gridLayout = new GridLayout();
        gridLayout.numColumns = 3;
        thisArea.setLayout(gridLayout);

        // チェックボックス
        untilDeadLineChk = new Button(thisArea, SWT.CHECK);
        untilDeadLineChk.setText(Messages.getString("ProjectConfigDialog.16")); //$NON-NLS-1$
        // デフォルト設定
        untilDeadLineChk.setSelection(true);
        // SelectionListenerの追加
        untilDeadLineChk.addSelectionListener(new SelectionAdapter(){
			public void widgetSelected(SelectionEvent e) {
				// 完了期限テキストの活性化切換え
				if(untilDeadLineChk.getSelection()){
					untilDeadLineTxt.setEditable(true);
				} else {
					untilDeadLineTxt.setEditable(false);
				}
				// 入力項目のチェック
				okEnabled();
			}
        });
        
    	// レイアウト設定
        GridData data = new GridData();
        data.horizontalSpan = 3;
        untilDeadLineChk.setLayoutData(data);
        
        // ラベル
        Label label = new Label(thisArea, SWT.NONE);
        label.setText(Messages.getString("ProjectConfigDialog.17")); //$NON-NLS-1$
        GridData indent = new GridData();
        indent.horizontalIndent = 20;
        label.setLayoutData(indent);
        
        // 完了期限までの日にちを設定するテキスト
        untilDeadLineTxt = new Text(thisArea, SWT.BORDER );
        untilDeadLineTxt.setLayoutData(textSizeSS);
        // デフォルト設定
        untilDeadLineTxt.setEditable(true);
        untilDeadLineTxt.setText("1");
        
        // プロジェクト情報の編集時に既存の設定を反映させる
        if(oldConfig != null){
        	// チェックボックスの反映
        	untilDeadLineChk.setSelection(oldConfig.isUntilDeadLineCheck());
        	// テキストの反映
        	untilDeadLineTxt.setText(oldConfig.getUntilDeadLine());
        	// テキスト活性化の反映
        	if(oldConfig.isUntilDeadLineCheck()){
        		untilDeadLineTxt.setEditable(true);
        	} else {
        		untilDeadLineTxt.setEditable(false);
        	}
        }
        
        // ModifyListenerの追加
        untilDeadLineTxt.addModifyListener(new ModifyListener(){
			public void modifyText(ModifyEvent e) {
				// 入力項目のチェック
				okEnabled();
			}
        });


        // ラベル
        label = new Label(thisArea, SWT.NONE);
        label.setText(Messages.getString("ProjectConfigDialog.18")); //$NON-NLS-1$
    }
    
    
    /**
     * メールタブのヘッダエリアを生成します。
     * 
     * @param			parent				親コンポジット
     * @throws			Exception
     */        
    private void createMailTabHeaderArea(Composite parent) throws Exception {

    	// この部品のコンポジット生成
    	Composite thisArea = new Composite(parent, SWT.NONE);
        GridLayout gridLayout = new GridLayout();
        thisArea.setLayout(gridLayout);

        // ラベル
        Label label = new Label(thisArea, SWT.NONE);
        label.setText(Messages.getString("ProjectConfigDialog.8")); //$NON-NLS-1$
        label.setFont(boldFont);
    }
    
    /**
     * メール通知チェックボックスエリアを生成します。
     * 
     * @param			parent				親コンポジット			
     * @throws			Exception			
     */
    private void createMailCheckArea(Composite parent) throws Exception {
    	
    	// この部品のコンポジット生成
    	Composite thisArea = new Composite(parent, SWT.NONE);
    	
    	// レイアウト設定
        thisArea.setLayout(new GridLayout());
    	
        // チェックボックス
        mailChk = new Button(thisArea, SWT.CHECK);
        mailChk.setText(Messages.getString("ProjectConfigDialog.9")); //$NON-NLS-1$
        // デフォルト設定
        mailChk.setSelection(false);
        
        // プロジェクト情報の編集時に既存の設定を反映させる
        if(oldConfig != null){
        	mailChk.setSelection(oldConfig.isMailCheck());
        }
        
        // SelectionListenerの追加
        mailChk.addSelectionListener(new SelectionAdapter(){
        	
			public void widgetSelected(SelectionEvent e) {
				
				// メール通知タブ内のテキスト活性化切換え
				if(mailChk.getSelection()){
					smtpTxt.setEditable(true);
					senderTxt.setEditable(true);
					recipientTxt.setEditable(true);
					prefixTxt.setEditable(true);
					
				} else {
					smtpTxt.setEditable(false);
					senderTxt.setEditable(false);
					recipientTxt.setEditable(false);
					prefixTxt.setEditable(false);

				}
				
				// 入力項目のチェック
				okEnabled();
			}
			
        });
    	
    }
    
    /**
     * SMTPサーバを設定するエリアを生成します。
     * 
     * @param			parent				親コンポジット
     * @throws 		Exception
     */
    private void createSmtpArea(Composite parent) throws Exception {
    	
    	// この部品のコンポジット生成
    	Composite thisArea = new Composite(parent, SWT.NONE);

    	// レイアウト設定。2列のグリッドレイアウトを使う
        GridLayout gridLayout = new GridLayout();
        gridLayout.numColumns = 2;
        thisArea.setLayout(gridLayout);
        
        // ラベル設定
        Label label = new Label(thisArea, SWT.NONE);
        label.setText(Messages.getString("ProjectConfigDialog.10")); //$NON-NLS-1$
        
        // テキスト設定
        smtpTxt = new Text(thisArea, SWT.BORDER);
        smtpTxt.setLayoutData(textSizeL);
        smtpTxt.setEditable(false);
        
        // プロジェクト情報の編集時に既存の設定を反映させる
        if(oldConfig != null){
        	
        	// テキストの反映
        	smtpTxt.setText(oldConfig.getSmtp());
        	
        	// テキスト活性化の反映
        	if(oldConfig.isMailCheck()){
        		smtpTxt.setEditable(true);
        	} else {
        		smtpTxt.setEditable(false);
        	}
        	
        }
        
        // ModifyListenerの追加
        smtpTxt.addModifyListener(new ModifyListener(){
			public void modifyText(ModifyEvent e) {
				// 入力項目のチェック
				okEnabled();
			}
        });
    	
    }
    
    /**
     * 差出人アドレスを設定するエリアを生成します。
     * 
     * @param 			parent				親コンポジット
     * @throws 		Exception			
     */
    private void createSenderAddressArea(Composite parent) throws Exception {
    	
    	// この部品のコンポジット生成
    	Composite thisArea = new Composite(parent, SWT.NONE);

    	// レイアウト設定。2列のグリッドレイアウトを使う
        GridLayout gridLayout = new GridLayout();
        gridLayout.numColumns = 2;
        thisArea.setLayout(gridLayout);
        
        // ラベル設定
        Label label = new Label(thisArea, SWT.NONE);
        label.setText(Messages.getString("ProjectConfigDialog.11")); //$NON-NLS-1$
        
        // テキスト設定
        senderTxt = new Text(thisArea, SWT.BORDER);
        senderTxt.setLayoutData(textSizeL);
        senderTxt.setEditable(false);
        
        // プロジェクト情報の編集時に既存の設定を反映させる
        if(oldConfig != null){
        	
        	// テキストの反映
        	senderTxt.setText(oldConfig.getSender());
        	
        	// テキスト活性化の反映
        	if(oldConfig.isMailCheck()){
        		senderTxt.setEditable(true);
        	} else {
        		senderTxt.setEditable(false);
        	}
        	
        }
        
        // ModifyListenerの追加
        senderTxt.addModifyListener(new ModifyListener(){
			public void modifyText(ModifyEvent e) {
				// 入力項目のチェック
				okEnabled();
			}
        });
    	
    }
    
    /**
     * 送信先アドレスを設定するエリアを生成します。
     * 
     * @param			parent				親コンポジット
     * @throws 		Exception
     */
    private void createRecipientAddressArea(Composite parent) throws Exception {
    	
    	// この部品のコンポジット生成
    	Composite thisArea = new Composite(parent, SWT.NONE);

    	// レイアウト設定。2列のグリッドレイアウトを使う
        GridLayout gridLayout = new GridLayout();
        gridLayout.numColumns = 2;
        thisArea.setLayout(gridLayout);
        
        // ラベル設定
        Label label = new Label(thisArea, SWT.NONE);
        label.setText(Messages.getString("ProjectConfigDialog.12")); //$NON-NLS-1$

        // テキスト設定
        recipientTxt = new Text(thisArea, SWT.BORDER);
        recipientTxt.setLayoutData(textSizeL);
        recipientTxt.setEditable(false);
        
        // プロジェクト情報の編集時に既存の設定を反映させる
        if(oldConfig != null){
        	
        	// テキストの反映
        	recipientTxt.setText(oldConfig.getRecipient());
        	
        	// テキスト活性化の反映
        	if(oldConfig.isMailCheck()){
        		recipientTxt.setEditable(true);
        	} else {
        		recipientTxt.setEditable(false);
        	}
        	
        }
        
        // ModifyListenerの追加
        recipientTxt.addModifyListener(new ModifyListener(){
			public void modifyText(ModifyEvent e) {
				// 入力項目のチェック
				okEnabled();
			}
        });
        
        // 左側列のダミー
        label = new Label(thisArea, SWT.NONE);
        label.setText("");

        // ラベル
        label = new Label(thisArea, SWT.NONE);
        label.setText(Messages.getString("ProjectConfigDialog.13")); //$NON-NLS-1$
        label.setAlignment(SWT.RIGHT);
    	
    }
    
    /**
     * タイトルプレフィックスを設定するエリアを生成します。
     * 
     * @param 			parent				親コンポジット
     * @throws 		Exception
     */
    private void createPrefixArea(Composite parent) throws Exception {
    	
    	// この部品のコンポジット生成
    	Composite thisArea = new Composite(parent, SWT.NONE);

    	// レイアウト設定。2列のグリッドレイアウトを使う
        GridLayout gridLayout = new GridLayout();
        gridLayout.numColumns = 2;
        gridLayout.marginLeft = 16;
        thisArea.setLayout(gridLayout);
        
        // タイトルラベル設定
        Label label = new Label(thisArea, SWT.NONE);
        label.setText(Messages.getString("ProjectConfigDialog.14")); //$NON-NLS-1$
        GridData data = new GridData();
        data.heightHint = 37;
        label.setLayoutData(data);

        // テキスト設定
        prefixTxt = new Text(thisArea, SWT.BORDER);
        prefixTxt.setLayoutData(textSizeL);
        prefixTxt.setText("[BTS]");
        prefixTxt.setEditable(false);
        
        // プロジェクト情報の編集時に既存の設定を反映させる
        if(oldConfig != null){
        	
        	// テキストの反映
        	prefixTxt.setText(oldConfig.getPrefix());
        	
        	// テキスト活性化の反映
        	if(oldConfig.isMailCheck()){
        		prefixTxt.setEditable(true);
        	} else {
        		prefixTxt.setEditable(false);
        	}
        	
        }
        
        // ModifyListenerの追加
        prefixTxt.addModifyListener(new ModifyListener(){
			public void modifyText(ModifyEvent e) {
				// 入力項目のチェック
				okEnabled();
			}
        });
    	
    }

    
    /**
     * 同期タブのヘッダエリアを生成します。
     * 
     * @param		parent				親コンポジット
     * @throws		Exception
     */
    private void createSyncTabHeaderArea(Composite parent) throws Exception {
    	
    	// この部品のコンポジット生成
    	Composite thisArea = new Composite(parent, SWT.NONE);
        thisArea.setLayout(new GridLayout());
    	
        // ラベル生成
        Label label = new Label(thisArea, SWT.NONE);
        label.setText(Messages.getString("ProjectConfigDialog.44")); //$NON-NLS-1$
        label.setFont(boldFont);
    }
    
    /**
     * 固定端末への同期を設定するエリアを生成します。
     * 
     * @param			parent			親コンポジット
     * @throws 		Exception
     */
    private void createFixedSyncArea(Composite parent) throws Exception {
    	
    	// この部品のコンポジット生成
    	Composite thisArea = new Composite(parent, SWT.NONE);
        
    	// レイアウト設定。8列のグリッドレイアウトを使う
        GridLayout gridLayout = new GridLayout();
        gridLayout.numColumns = 8;
        thisArea.setLayout(gridLayout);
        
        // 固定同期処理チェックボックスの設定
        fixedSyncChk = new Button(thisArea, SWT.CHECK);
        fixedSyncChk.setText(Messages.getString("ProjectConfigDialog.45")); //$NON-NLS-1$
        fixedSyncChk.setSelection(false);
        GridData data = new GridData();
        data.horizontalSpan = 8;
        fixedSyncChk.setLayoutData(data);
        
        // プロジェクト情報の編集時に既存の設定を反映させる
        if(oldConfig != null){
        	fixedSyncChk.setSelection(oldConfig.isFixedSyncCheck());
        }
        
        // 固定同期処理チェックボックスにSelectionListenerを追加
        fixedSyncChk.addSelectionListener(new SelectionAdapter() {

        	/*
        	 *  (非 Javadoc)
        	 * @see org.eclipse.swt.events.SelectionListener#widgetSelected(org.eclipse.swt.events.SelectionEvent)
        	 */
			public void widgetSelected(SelectionEvent e) {
				
				boolean flag = fixedSyncChk.getSelection();
				
				// アドレステキストとポートテキストの活性化切換え
				address1.setEditable(flag);
				address2.setEditable(flag);
				address3.setEditable(flag);
				address4.setEditable(flag);
				port.setEditable(flag);
				
				// 「同期端末への同期」の活性化切換え
				normalSyncRdi.setEnabled(flag && autoSyncChk.getSelection());
				normalSyncLabel.setEnabled(flag && autoSyncChk.getSelection());
				
				// 固定同期処理チェックボックスがチェックされていない場合
				if(!flag) {
					
					// 「リモート端末を指定して同期」ラジオを選択する
					normalSyncRdi.setSelection(false);
					appointSyncRdi.setSelection(true);	
					
				}
				
				// 入力項目のチェック
				okEnabled();
				
			}
        	
        });
        
        // 「アドレス」ラベルの設定
        Label label = new Label(thisArea, SWT.NONE);
        label.setText(Messages.getString("ProjectConfigDialog.46")); //$NON-NLS-1$
        GridData labelData = new GridData();
        labelData.horizontalIndent = 15;
        label.setLayoutData(labelData);
        
        // アドレステキストボックスの設定　1番目
        address1 = new Text(thisArea, SWT.BORDER);
        address1.setTextLimit(3);
        address1.setEditable(false);
        address1.setLayoutData(textSizeSS);
        
        // プロジェクト情報の編集時に既存の設定を反映させる
        if(oldConfig != null){
        	address1.setText(oldConfig.getAddress1());
        	address1.setEditable(oldConfig.isFixedSyncCheck());
        }
        
        // ModifyListener追加
        address1.addModifyListener(new ModifyListener(){
        	
        	/*
        	 *  (非 Javadoc)
        	 * @see org.eclipse.swt.events.ModifyListener#modifyText(org.eclipse.swt.events.ModifyEvent)
        	 */
			public void modifyText(ModifyEvent e) {
				// 入力項目のチェック
				okEnabled();
			}
        	
        });
        
        // 「.」ラベルの設定
        label = new Label(thisArea, SWT.NONE);
        label.setText(".");
        
        // アドレステキストボックスの設定　2番目
        address2 = new Text(thisArea, SWT.BORDER);
        address2.setTextLimit(3);
        address2.setEditable(false);
        address2.setLayoutData(textSizeSS);
        
        // プロジェクト情報の編集時に既存の設定を反映させる
        if(oldConfig != null){
        	address2.setText(oldConfig.getAddress2());
        	address2.setEditable(oldConfig.isFixedSyncCheck());
        }
        
        // ModifyListener追加
        address2.addModifyListener(new ModifyListener(){
        	
        	/*
        	 *  (非 Javadoc)
        	 * @see org.eclipse.swt.events.ModifyListener#modifyText(org.eclipse.swt.events.ModifyEvent)
        	 */
			public void modifyText(ModifyEvent e) {
				// 入力項目のチェック
				okEnabled();
			}
        	
        });
        
        // 「.」ラベルの設定
        label = new Label(thisArea, SWT.NONE);
        label.setText(".");
        
        // アドレステキストボックスの設定　3番目
        address3 = new Text(thisArea, SWT.BORDER);
        address3.setTextLimit(3);
        address3.setEditable(false);
        address3.setLayoutData(textSizeSS);
        
        // プロジェクト情報の編集時に既存の設定を反映させる
        if(oldConfig != null){
        	address3.setText(oldConfig.getAddress3());
        	address3.setEditable(oldConfig.isFixedSyncCheck());
        }
        
        // ModifyListener追加
        address3.addModifyListener(new ModifyListener(){
        	
        	/*
        	 *  (非 Javadoc)
        	 * @see org.eclipse.swt.events.ModifyListener#modifyText(org.eclipse.swt.events.ModifyEvent)
        	 */
			public void modifyText(ModifyEvent e) {
				// 入力項目のチェック
				okEnabled();
			}
        	
        });
        
        // 「.」ラベルの設定
        label = new Label(thisArea, SWT.NONE);
        label.setText(".");
        
        // アドレステキストボックスの設定　4番目
        address4 = new Text(thisArea, SWT.BORDER);
        address4.setTextLimit(3);
        address4.setEditable(false);
        address4.setLayoutData(textSizeSS);
        
        // プロジェクト情報の編集時に既存の設定を反映させる
        if(oldConfig != null){
        	address4.setText(oldConfig.getAddress4());
        	address4.setEditable(oldConfig.isFixedSyncCheck());
        }
        
        // ModifyListener追加
        address4.addModifyListener(new ModifyListener(){
        	
        	/*
        	 *  (非 Javadoc)
        	 * @see org.eclipse.swt.events.ModifyListener#modifyText(org.eclipse.swt.events.ModifyEvent)
        	 */
			public void modifyText(ModifyEvent e) {
				// 入力項目のチェック
				okEnabled();
			}
        	
        });
        
        
        // 「ポート」ラベルの設定
        label = new Label(thisArea, SWT.NONE);
        label.setText(Messages.getString("ProjectConfigDialog.47")); //$NON-NLS-1$
        label.setLayoutData(labelData);
        
        // ポート番号テキストボックスの設定
        port = new Text(thisArea, SWT.BORDER);
        port.setTextLimit(5);
        port.setText("4344");
        port.setEditable(false);
 		GridData portData = new GridData(); 
 		portData.widthHint = 30;
 		portData.horizontalSpan = 7;
        port.setLayoutData(portData);
        
        // プロジェクト情報の編集時に既存の設定を反映させる
        if(oldConfig != null){
        	port.setText(oldConfig.getPort());
        	port.setEditable(oldConfig.isFixedSyncCheck());
        }
        
        // ModifyListener追加
        port.addModifyListener(new ModifyListener(){
        	
        	/*
        	 *  (非 Javadoc)
        	 * @see org.eclipse.swt.events.ModifyListener#modifyText(org.eclipse.swt.events.ModifyEvent)
        	 */
			public void modifyText(ModifyEvent e) {
				// 入力項目のチェック
				okEnabled();
			}
        	
        });
        
        // ラベルの設定
        label = new Label(thisArea, SWT.NONE);
        label.setText(Messages.getString("ProjectConfigDialog.48")); //$NON-NLS-1$
        GridData noteData = new GridData();
        noteData.horizontalSpan = 8;
        noteData.horizontalIndent = 15;
        label.setLayoutData(noteData);

        
    }
    
    /**
     * 自動同期処理を設定するエリアを生成します。
     * 
     * @param			parent			親コンポジット
     * @throws 		Exception
     */
    private void createAutoSyncArea(Composite parent) throws Exception {
    	
    	// この部品のコンポジット生成
    	Composite thisArea = new Composite(parent, SWT.NONE);
        thisArea.setLayout(new GridLayout());
        
        // 自動同期処理チェックボックスの設定
        autoSyncChk = new Button(thisArea, SWT.CHECK);
        autoSyncChk.setText(Messages.getString("ProjectConfigDialog.37")); //$NON-NLS-1$
        autoSyncChk.setSelection(true);
        GridData data = new GridData();
        autoSyncChk.setLayoutData(data);
        
        // プロジェクト情報の編集時に既存の設定を反映させる
        if(oldConfig != null){
        	autoSyncChk.setSelection(oldConfig.isAutoSyncCheck());
        }
        
        // 自動同期処理チェックボックスにSelectionListenerを追加
        autoSyncChk.addSelectionListener(new SelectionAdapter(){

        	/*
        	 *  (非 Javadoc)
        	 * @see org.eclipse.swt.events.SelectionListener#widgetSelected(org.eclipse.swt.events.SelectionEvent)
        	 */
			public void widgetSelected(SelectionEvent e) {
				
				// 活性化切換え
				normalSyncRdi.setEnabled(autoSyncChk.getSelection() && fixedSyncChk.getSelection());
				normalSyncLabel.setEnabled(autoSyncChk.getSelection() && fixedSyncChk.getSelection());
				appointSyncRdi.setEnabled(autoSyncChk.getSelection());
				appointSyncLabel.setEnabled(autoSyncChk.getSelection());
				
				// 入力項目のチェック
				okEnabled();
				
			}
        	
        });
    	
        
        // 「同期」ラジオボタンの設定
        normalSyncRdi = new Button(thisArea, SWT.RADIO);
        normalSyncRdi.setText(Messages.getString("ProjectConfigDialog.38")); //$NON-NLS-1$
        normalSyncRdi.setSelection(false);
        normalSyncRdi.setEnabled(false);
        data = new GridData();
        data.horizontalIndent = 15;
        normalSyncRdi.setLayoutData(data);
        
        // プロジェクト情報の編集時に既存の設定を反映させる
        if(oldConfig != null){
        	normalSyncRdi.setSelection(oldConfig.isNormalSyncRadio());
        	normalSyncRdi.setEnabled(oldConfig.isAutoSyncCheck() && oldConfig.isFixedSyncCheck());
        }
        
        // 「同期」ラジオボタンにSelectionListenerを追加
        normalSyncRdi.addSelectionListener(new SelectionAdapter(){

        	/*
        	 *  (非 Javadoc)
        	 * @see org.eclipse.swt.events.SelectionListener#widgetSelected(org.eclipse.swt.events.SelectionEvent)
        	 */
			public void widgetSelected(SelectionEvent e) {
				
				// 入力項目チェック
				okEnabled();
				
			}
        	
        });
        
        // ラベル設定
        normalSyncLabel = new Label(thisArea, SWT.NONE);
        normalSyncLabel.setText(Messages.getString("ProjectConfigDialog.41")); //$NON-NLS-1$
        normalSyncLabel.setEnabled(false);
        data = new GridData();
        data.horizontalIndent = 33;
        normalSyncLabel.setLayoutData(data);
        
        // プロジェクト情報の編集時に既存の設定を反映させる
        if(oldConfig != null){
        	normalSyncLabel.setEnabled(oldConfig.isAutoSyncCheck() && oldConfig.isFixedSyncCheck());
        }
        
        // 「リモート端末を指定して同期」ラジオボタンの設定
        appointSyncRdi = new Button(thisArea, SWT.RADIO);
        appointSyncRdi.setText(Messages.getString("ProjectConfigDialog.39")); //$NON-NLS-1$
        appointSyncRdi.setSelection(true);
        data = new GridData();
        data.horizontalIndent = 15;
        appointSyncRdi.setLayoutData(data);
        
        // プロジェクト情報の編集時に既存の設定を反映させる
        if(oldConfig != null){
        	appointSyncRdi.setSelection(oldConfig.isAppointSyncRadio());
        	appointSyncRdi.setEnabled(oldConfig.isAutoSyncCheck());
        }
        
        // 「リモート端末を指定して同期」ラジオボタンにSelectionListenerを追加
        appointSyncRdi.addSelectionListener(new SelectionAdapter(){

        	/*
        	 *  (非 Javadoc)
        	 * @see org.eclipse.swt.events.SelectionListener#widgetSelected(org.eclipse.swt.events.SelectionEvent)
        	 */
			public void widgetSelected(SelectionEvent e) {
				
				// 入力項目チェック
				okEnabled();
				
			}
        	
        });
        
        // ラベル設定
        appointSyncLabel = new Label(thisArea, SWT.NONE);
        appointSyncLabel.setText(Messages.getString("ProjectConfigDialog.42")); //$NON-NLS-1$
        data = new GridData();
        data.horizontalIndent = 33;
        appointSyncLabel.setLayoutData(data);
        
        // プロジェクト情報の編集時に既存の設定を反映させる
        if(oldConfig != null){
        	appointSyncLabel.setEnabled(oldConfig.isAutoSyncCheck());
        }
    }
    
    
	/*
	 *  (非 Javadoc)
	 * @see org.eclipse.jface.dialogs.Dialog#createButtonsForButtonBar(org.eclipse.swt.widgets.Composite)
	 */
	protected void createButtonsForButtonBar(Composite parent) {
		
        // 「OK」ボタンをデフォルトで非活性にする
        createButton(parent, IDialogConstants.OK_ID, IDialogConstants.OK_LABEL, true).setEnabled(false);
        
        // キャンセルボタン
        createButton(parent, IDialogConstants.CANCEL_ID, IDialogConstants.CANCEL_LABEL, false);
	}
    
    
    /*
     *  (非 Javadoc)
     * @see org.eclipse.jface.dialogs.Dialog#okPressed()
     */
	protected void okPressed() {
		
		// プロジェクト情報をXMLに保存する
		ProjectConfig newconf = save();

		// カレントプロジェクトの場合には、カレントプロジェクト管理クラスにも反映します
		CurrentProject current = CurrentProject.getInsance();
		if( current.isCurrentProject(newconf) ) {
			current.setProjectConfig(newconf);
		}

		// プロジェクトノードのツリー表示を最新にします。
		refreshView(newconf);
		
		// 新規プロジェクトの場合。新規プロジェクトの情報を生成する。
		if(oldConfig==null) {
			createNewProject(newconf);
		}
		
		super.okPressed();
	}
	
	/**
	 * 新規プロジェクトの情報を生成します。
	 * 新規にプロジェクトを作成した場合に呼ばれます。新規プロジェクトは以下の処理を行います。
	 * <pre>
	 * ・プロジェクト情報を格納するディレクトリを生成
	 * ・プロジェクト用DBのテーブルを生成
	 * </pre>
	 * 
	 * @param		newconf		新規プロジェクトの情報
	 */
	private void createNewProject(ProjectConfig newconf) {
		
		// ワークスペースのディレクトリ取得
		String ssidDir = BTSUtility.getBaseDir(newconf.getSsid());
		
		File dir =new File(ssidDir);
		if( dir.exists() ){
			// 既にディレクトリがあった場合は中身を全部消す
			File[] files = dir.listFiles();
			for(int i=0; i<files.length; i++) {
				files[i].delete();		
			}
		} else {
			// ディレクトリがない場合は作る
			dir.mkdirs();
		}

		// 新規にDBを生成。
		DatabaseInitializer.Initialize(newconf.getSsid());
	}

	
	/**
	 * プロジェクトノードのツリー表示を最新更新にします。
	 * 
	 * プロジェクト新規追加の場合はプロジェクトノードを追加します。<br>
	 * プロジェクト編集の場合は表示を更新します。ただしプロジェクト名を変更してないときはなにもしません。<br>
	 * 
	 * @param			newConfig			新プロジェクト情報
	 */
	private void refreshView(final ProjectConfig newConfig) {

		// プロジェクト新規追加の場合
		if(oldConfig==null) {
			// ビューを操作するので、メインスレッドでの処理を強制します。
			Display.getDefault().syncExec(new Runnable() {
				public void run() {
					// ツリーにプロジェクト情報を追加します。
					TreeHelper.addProjectNode(newConfig.getSsid(), newConfig.getProjectName());
				}
			});
			
			return ;
		}
		
		//（ 以降、プロジェクト情報更新の場合。）
		
		// 新旧プロジェクト名取得
		String oldName = oldConfig.getProjectName();
		String newName = newConfig.getProjectName();
		
		// プロジェクト名を変更したかどうか確認
		if(!oldName.equals(newName)) {
			Display.getDefault().syncExec(new Runnable() {
				public void run() {
					// ツリー表示を更新します。
					TreeHelper.modifyProjectNode(newConfig.getSsid(), newConfig.getProjectName());
				}
			});
		}
		
		//（ 以降、ユーザ名更新の場合。）
		
		// 新旧ユーザ名首都k
		String oldUserName = oldConfig.getUserName();
		final String newUserName = newConfig.getUserName();
		
		// ユーザ名を変更したかどうか確認
		if(!newUserName.equals(oldUserName)) {
			
			Display.getDefault().syncExec(new Runnable() {
				public void run() {
					
					// ローカル端末の表示更新
					ClientInfo clientInfo = ClientList.getMyInfo();
					if(clientInfo != null) {

						// ツリー表示を更新します。
						clientInfo.setClientName(newUserName);
						TreeHelper.modifyMemberName(newConfig.getSsid(), clientInfo);

						// 他の端末にユーザ名変更を知らせる
						Join join = new Join( new NetworkConfig());
						join.execute();
					}
				}
			});
			

		}
		
	}
	
	
	/**
	 * プロジェクト情報をXMLに保存します。
	 * 
	 * @return			保存したプロジェクト情報と同じインスタンス
	 */
	private ProjectConfig save() {
		
		// プロジェクト情報を生成する
		ProjectConfig newConfig = new ProjectConfig();
		
		// テキストの入力をプロジェクト情報にセット
		newConfig.setProjectName(projectNameTxt.getText().trim());			// プロジェクト名
		newConfig.setUserName(userNameTxt.getText().trim());				// ユーザ名
		newConfig.setSsid(ssidTxt.getText().trim());						// SSID
		newConfig.setMailCheck(mailChk.getSelection());						// メール通知（チェックボックス）
		newConfig.setSmtp(smtpTxt.getText().trim());						// SMTPサーバ
		newConfig.setSender(senderTxt.getText().trim());					// 差出人アドレス
		newConfig.setRecipient(recipientTxt.getText().trim());				// 送信先アドレス
		newConfig.setPrefix(prefixTxt.getText().trim());					// タイトルプレフィックス
		newConfig.setCategory(categoryTxt.getText().trim());				// カテゴリ
		newConfig.setPriority(priorityTxt.getText().trim());				// 優先度
		newConfig.setAssigned(assignedTxt.getText().trim());				// 担当者
		newConfig.setVoteCheck(voteChk.getSelection());					 	// 投票（チェックボックス）
		newConfig.setVote(voteTxt.getText().trim());						// 投票
		newConfig.setUntilDeadLineCheck(untilDeadLineChk.getSelection());	// 期限（チェックボックス）
		newConfig.setUntilDeadLine(untilDeadLineTxt.getText().trim());		// 期限
		newConfig.setFixedSyncCheck(fixedSyncChk.getSelection());			// 固定同期処理（チェックボックス）
		newConfig.setAddress1(address1.getText().trim());					// アドレス1番目
		newConfig.setAddress2(address2.getText().trim());					// アドレス2番目
		newConfig.setAddress3(address3.getText().trim());					// アドレス3番目
		newConfig.setAddress4(address4.getText().trim());					// アドレス4番目
		newConfig.setPort(port.getText().trim());							// ポート番号
		newConfig.setAutoSyncCheck(autoSyncChk.getSelection());				// 自動同期処理（チェックボックス）
		newConfig.setNormalSyncRadio(normalSyncRdi.getSelection());			// 同期（ラジオボックス）
		newConfig.setAppointSyncRadio(appointSyncRdi.getSelection());		// 端末指定同期（ラジオボックス）
		
		// 旧プロジェクト情報の有無を判定
		if (oldConfig == null) {
			// 旧プロジェクト情報がなければ新規作成
			ProjectConfigManager.save(newConfig);
		} else {
			// 旧プロジェクト情報があれば編集
			ProjectConfigManager.modify(oldConfig, newConfig);
		}
		return newConfig;
	}

	
	/**
	 * 当ダイアログの入力項目のチェックを行います。
	 * エラーがない場合は、「OK」ボタンを活性化させます。
	 */
	private void okEnabled() {
		
		ProjectConfigDialogValidate validate = new ProjectConfigDialogValidate(this);
		
		// 全ての入力項目がエラーなしの場合に「OK」ボタンを活性化
		if(oldConfig == null){
			// 新規作成時
			getButton(IDialogConstants.OK_ID).setEnabled
				(  validate.validateProjectName(projectNameTxt) 
				&& validate.validateUserName(userNameTxt) 
				&& validate.validateSsid(ssidTxt)
				&& validate.validatePriority(priorityTxt) 
				&& validate.validateVote(voteChk, voteTxt) 
				&& validate.validateUntilDeadLine(untilDeadLineChk, untilDeadLineTxt)
				&& validate.validateSmtp(mailChk, smtpTxt)
				&& validate.validateSender(mailChk, senderTxt)
				&& validate.validateRecipient(mailChk, recipientTxt)
				&& validate.validateAddress(fixedSyncChk, address1)
				&& validate.validateAddress(fixedSyncChk, address2)
				&& validate.validateAddress(fixedSyncChk, address3)
				&& validate.validateAddress(fixedSyncChk, address4)
				&& validate.validatePort(fixedSyncChk, port));
		} else {
			// 編集時はSSIDを無視
			getButton(IDialogConstants.OK_ID).setEnabled
				(  validate.validateProjectName(projectNameTxt) 
				&& validate.validateUserName(userNameTxt)
				&& validate.validatePriority(priorityTxt) 
				&& validate.validateVote(voteChk, voteTxt) 
				&& validate.validateUntilDeadLine(untilDeadLineChk, untilDeadLineTxt)
				&& validate.validateSmtp(mailChk, smtpTxt)
				&& validate.validateSender(mailChk, senderTxt)
				&& validate.validateRecipient(mailChk, recipientTxt)
				&& validate.validateAddress(fixedSyncChk, address1)
				&& validate.validateAddress(fixedSyncChk, address2)
				&& validate.validateAddress(fixedSyncChk, address3)
				&& validate.validateAddress(fixedSyncChk, address4)
				&& validate.validatePort(fixedSyncChk, port));
		}
	}
}


/*
 * テキスト入力項目のエラー処理
 * 
 * ・参考サイト
 * 　http://www13.plala.or.jp/observe/JFace/jface_dialog.html
 *   http://www-06.ibm.com/jp/developerworks/opensource/030808/j_os-ecjfw.html
 */
