/***********************************************************************
 * Copyright(C) 2006 Valtech Co.,Ltd.
 * All Rights Reserved. This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.opensource.org/licenses/cpl.php
 ***********************************************************************/
package jp.valtech.bts.util;

import java.util.ArrayList;
import java.util.Iterator;

import jp.valtech.bts.data.Attachment;
import jp.valtech.bts.data.CommentHistory;
import jp.valtech.bts.data.CurrentProject;
import jp.valtech.bts.data.Issue;
import jp.valtech.bts.data.IssueAttribute;
import jp.valtech.bts.data.IssueHistory;
import jp.valtech.bts.data.IssueStatus;
import jp.valtech.bts.data.IssueType;
import jp.valtech.bts.ui.navigator.ProjectConfig;

/**
 * 課題票関連のよく使う処理を集めたくらすです。
 * 
 * @author		<A href="mailto:m_sugitou@valtech.jp">M.Sugito</A>
 * @version	Ver.0.8
 */
public class IssueUtility {
	
	/**
	 * 2つの課題票を比較して課題票履歴を生成します。
	 * 
	 * @param			oldIssue			比較対照の課題票１
	 * @param			newIssue			比較対照の課題票２
	 * @return			比較して生成した課題票履歴
	 */
	public static IssueHistory[] createIssueHistory(Issue oldIssue , Issue newIssue, CommentHistory newComment) {
		ArrayList historyList = new ArrayList();
		
		// 新規登録の課題票の場合
		if(oldIssue.getVersion().intValue() == 0) {
			return createInitialHistory(newIssue);
		}
		
		// カテゴリ
		if( !equals(oldIssue.getCategory(), newIssue.getCategory()) ) {
			historyList.add(new IssueHistory( newIssue
									 , IssueAttribute.CATEGORY_VALUE
									 , oldIssue.getCategory()
									 , newIssue.getCategory()) );
		}
		
		// タイトル
		if( !equals(oldIssue.getTitle(), newIssue.getTitle()) ) {
			historyList.add(new IssueHistory( newIssue
									 , IssueAttribute.TITLE_VALUE
									 , oldIssue.getTitle()
									 , newIssue.getTitle())	);
		}
		
		// 詳細
		if( !equals(oldIssue.getDescription(), newIssue.getDescription()) ) {
			historyList.add(new IssueHistory( newIssue
									 , IssueAttribute.DESCRIPTION_VALUE
									 , oldIssue.getDescription()
									 , newIssue.getDescription())	);
		}
		
		// 優先度
		if( !equals(oldIssue.getPriority(), newIssue.getPriority()) ) {
			historyList.add(new IssueHistory( newIssue
									 , IssueAttribute.PRIORITY_VALUE
									 , oldIssue.getPriority()
									 , newIssue.getPriority())	);
		}
		
		// ステータス
		if( !equals(oldIssue.getStatus(), newIssue.getStatus()) ) {
			historyList.add(new IssueHistory( newIssue
									 , IssueAttribute.STATUS_VALUE
									 , IssueStatus.getEnumFromValue( oldIssue.getStatus()).getDescription() 
									 , IssueStatus.getEnumFromValue( newIssue.getStatus()).getDescription()) );
		}
		
		// 担当者
		if( !equals(oldIssue.getAssigned(), newIssue.getAssigned()) ) {
			historyList.add(new IssueHistory( newIssue
									 , IssueAttribute.ASSIGNED_VALUE
									 , oldIssue.getAssigned()
									 , newIssue.getAssigned())	);
		}

		
		// 完了期限
		String date1  = BTSUtility.formatDate(oldIssue.getDeadline(), "yyyy/MM/dd"); 
		String date2  = BTSUtility.formatDate(newIssue.getDeadline(), "yyyy/MM/dd"); 
		if( !equals(date1, date2) ) {
			historyList.add(new IssueHistory( newIssue
									 , IssueAttribute.DEADLINE_VALUE
									 , date1
									 , date2)	 );
		}
		
		// 内緒のバグ
		if( !equals(oldIssue.getSecretBug().toString(), newIssue.getSecretBug().toString()) ) {
			
			// 更新後の表示を変換
			String oldSecretBug = Messages.getString("IssueUtility.3"); //$NON-NLS-1$
			if(oldIssue.getSecretBug().booleanValue()) {
				oldSecretBug = Messages.getString("IssueUtility.4"); //$NON-NLS-1$
			}
			
			// 更新後の表示を変換
			String newSecretBug = Messages.getString("IssueUtility.3"); //$NON-NLS-1$
			if(newIssue.getSecretBug().booleanValue()) {
				newSecretBug = Messages.getString("IssueUtility.4"); //$NON-NLS-1$
			}
			
			historyList.add(new IssueHistory( newIssue
									 , IssueAttribute.SECRETBUG_VALUE
									 , oldSecretBug
									 , newSecretBug)	);
		}
		
		// 一時保留
		if( !equals(oldIssue.getSuspend().toString(), newIssue.getSuspend().toString()) ) {
			
			// 更新後の表示を変換
			String oldSuspend = Messages.getString("IssueUtility.3"); //$NON-NLS-1$
			if(oldIssue.getSuspend().booleanValue()) {
				oldSuspend = Messages.getString("IssueUtility.4"); //$NON-NLS-1$
			}
			
			// 更新後の表示を変換
			String newSuspend = Messages.getString("IssueUtility.3"); //$NON-NLS-1$
			if(newIssue.getSuspend().booleanValue()) {
				newSuspend = Messages.getString("IssueUtility.4"); //$NON-NLS-1$
			}
			
			historyList.add(new IssueHistory( newIssue
									 , IssueAttribute.SUSPEND_VALUE
									 , oldSuspend
									 , newSuspend)	);
		}
		
		// 添付ファイル
		IssueHistory attachHistory = createAttachmentHistory(oldIssue, newIssue);
		if(attachHistory !=null) {
			historyList.add(attachHistory);
		}
		
		// コメント
		IssueHistory commentHistory = createCommentHistory(newComment);
		if(commentHistory != null) {
			historyList.add(commentHistory);
		}
		
		
		// 更新が無かった場合
		if( historyList.size() ==0 ) {
			return createNonUpdateHistory(newIssue);
		}
		
		 
		return (IssueHistory[])historyList.toArray(new IssueHistory[0]);
	}

	


	
	/**
	 * 添付ファイルの更新履歴を生成して返します。
	 * 戻り値がnullの場合は添付ファイルの追加／削除が無かったということです。
	 * 
	 * @param			oldIssue			比較対照の課題票１
	 * @param			newIssue			比較対照の課題票２
	 * @return			比較して生成した添付の更新履歴。nullの場合は変更無し
	 */
	private static IssueHistory createAttachmentHistory(Issue oldIssue , Issue newIssue) {
	
		
		ArrayList oldList = oldIssue.getAttachmentList();
		ArrayList newList = newIssue.getAttachmentList();
		
		// 添付が両方無い場合は差分なしとする
		if( (oldList ==null || oldList.size()==0) && (newList ==null || newList.size()==0)) {
			return null;
		}

		// 添付が両方有る場合、差分があるかどうか調べる
		if( oldList !=null && newList !=null ) {

			// まずサイズが同じかどうか調べる 
			if(oldList.size() == newList.size()) {

				// ファイルの表示名のArrayListを作る（新しいほうの添付情報）
				ArrayList newAttachFileNames = new ArrayList();
				Iterator newIte = newList.iterator();
				while (newIte.hasNext()) {
					Attachment element = (Attachment) newIte.next();
					newAttachFileNames.add( element.getDisplayName() );
				}
				
				// ファイルの表示名のArrayListを作る（古いほうの添付情報）
				ArrayList oldAttachFileNames = new ArrayList();
				Iterator oldIte = oldList.iterator();
				while (oldIte.hasNext()) {
					Attachment element = (Attachment) oldIte.next();
					oldAttachFileNames.add( element.getDisplayName() );
				}
				
				boolean equals = true;

				// 新・旧比較してお互いに全て同じファイルかどうかを確認する
				if(!newAttachFileNames.containsAll(oldAttachFileNames)) {
					equals = false;
				}
				if(!oldAttachFileNames.containsAll(newAttachFileNames)) {
					equals = false;
				}
				
				if(equals) {
					return null;
				}

			}
		}
		
		// 履歴インスタンス生成
		IssueHistory history = new IssueHistory
				( newIssue
				, IssueAttribute.ATTACHMENT_VALUE
				, Messages.getString("IssueUtility.0") //$NON-NLS-1$
				, Messages.getString("IssueUtility.1")); //$NON-NLS-1$
		

		// 添付ファイルの更新前情報生成
		if(oldList !=null && oldList.size()!=0) {
			StringBuffer msg = new StringBuffer("");
			for (int i = 0; i < oldList.size(); i++) {
				// ファイル名称を履歴に表示する
				Attachment attachment = (Attachment)oldList.get(i);
				msg.append(attachment.getDisplayName()).append("\n");
			}
			
			// 履歴に更新前情報を加える
			history.setUpdateBefore(msg.toString());
		}
		
		// 添付ファイルの更新後情報生成
		if(newList !=null && newList.size()!=0) {
			StringBuffer msg = new StringBuffer("");
			for (int i = 0; i < newList.size(); i++) {
				// ファイル名称を履歴に表示する
				Attachment attachment = (Attachment)newList.get(i);
				msg.append(attachment.getDisplayName()).append("\n");
			}

			// 履歴に更新後情報を加える
			history.setUpdateAfter(msg.toString());
		}

		return history;
	}
	
	
	/**
	 * コメント追記の履歴を生成して返します。
	 * コメント追記が無い場合はnullを返します。
	 * 
	 * @param		newComment		追記したコメント		
	 * @return		追記したコメントの履歴。追記がない場合はnull
	 */
	public static IssueHistory createCommentHistory(CommentHistory newComment) {
		if(newComment ==null) {
			return null;
		}
		
		IssueHistory history = new IssueHistory();
		history.setFingerPrint	  (newComment.getFingerPrint());
		history.setType			  (IssueType.RELEASE_VALUE);
		history.setUpdateAttribute(IssueAttribute.COMMENT_VALUE);
		history.setUpdateBefore	  ("");
		history.setUpdateAfter	  (Messages.getString("IssueUtility.2")); //$NON-NLS-1$
		history.setUpdateUser	  (newComment.getUpdateUser());
		history.setUpdateDate	  (newComment.getUpdateDate());

		return history;
	}

	/**
	 * 一番最初に登録した時の履歴を生成します。
	 * {@link IssueHistory#setUpdateAttribute(String) 変更した属性}に{@link IssueHistory#ISSUE_INITIAL_REGIST}
	 * を設定します。
	 *
	 * @param			issue			課題票情報
	 * @return			生成した履歴
	 */
	public static IssueHistory[] createInitialHistory(Issue issue) {
		
		IssueHistory history = new IssueHistory();
		history.setFingerPrint	  (issue.getFingerPrint());
		history.setType			  (IssueType.RELEASE_VALUE);
		history.setUpdateAttribute(IssueHistory.ISSUE_INITIAL_REGIST);
		history.setUpdateBefore	  ("");
		history.setUpdateAfter	  ("");
		history.setUpdateUser	  (CurrentProject.getInsance().getProjectConfig().getUserName());
		history.setUpdateDate	  (issue.getUpdateDate());

		return new IssueHistory[]{history};
	}


	/**
	 * 何も更新をせずに配信したときの履歴を生成します。
	 * {@link IssueHistory#setUpdateAttribute(String) 変更した属性}に{@link IssueHistory#ISSUE_NON_UPDATE}
	 * を設定します。
	 * 
	 * @param			issue			課題票情報
	 * @return			生成した履歴
	 */
	public static IssueHistory[] createNonUpdateHistory(Issue issue) {
		
		IssueHistory history = new IssueHistory();
		history.setFingerPrint	  (issue.getFingerPrint());
		history.setType			  (IssueType.RELEASE_VALUE);
		history.setUpdateAttribute(IssueHistory.ISSUE_NON_UPDATE);
		history.setUpdateBefore	  ("");
		history.setUpdateAfter	  ("");
		history.setUpdateUser	  (CurrentProject.getInsance().getProjectConfig().getUserName());
		history.setUpdateDate	  (issue.getUpdateDate());

		return new IssueHistory[]{history};
	}
	
	
	
	/**
	 * ごみ箱に移動したときの履歴を生成します。
	 * {@link IssueHistory#setUpdateAttribute(String) 変更した属性}に{@link IssueHistory#ISSUE_MOVE_GARBAGE}
	 * を設定します。
	 * 
	 * @param			issue			課題票情報
	 * @param			fromUser		ごみ箱にした人
	 * @return			生成した履歴
	 */
	public static IssueHistory createGarbageHistory(Issue issue, String fromUser) {
		
		IssueHistory history = new IssueHistory();
		history.setFingerPrint	  (issue.getFingerPrint());
		history.setType			  (IssueType.GARBAGE_VALUE);
		history.setUpdateAttribute(IssueHistory.ISSUE_MOVE_GARBAGE);
		history.setUpdateBefore	  ("");
		history.setUpdateAfter	  ("");
		history.setUpdateUser	  (fromUser);
		history.setUpdateDate	  (issue.getUpdateDate());

		return history;
	}

	/**
	 * <DL>
	 *  <DT>課題票のFingerPrintを取得します。</DT>
	 *  <DD></DD>
	 * </DL>
	 * @param			issue				課題票
	 * @return			FingerPrint			生成したFingerPrint
	 */
	public static String createFingerPrint( Issue issue ){
		StringBuffer buf = new StringBuffer();
		
		buf.append( issue.getType() );
		buf.append( issue.getIssueID() );
		buf.append( issue.getCategory() );
		buf.append( issue.getTitle() );
		buf.append( issue.getDescription() );
		buf.append( issue.getPriority() );
		buf.append( issue.getStatus() );
		buf.append( issue.getDeadline() );
		buf.append( issue.getAssigned() );
		buf.append( issue.isSecretBug() );
		buf.append( issue.isSuspend() );
		buf.append( issue.getVersion() );
		buf.append( issue.getCreateUser() );
		buf.append( issue.getCreateDate() );
		buf.append( System.currentTimeMillis() );
		
		issue.setFingerPrint( CipherUtility.hash( buf.toString() ) );
		return issue.getFingerPrint();
	}


	/**
	 * ２つの添付情報を比較し、削除された添付ファイルのリストを生成して返します。
	 * 
	 * @param		oldIssue		更新前課題票
	 * @param		newIssue		更新後課題票
	 * @return		課題票更新時に削除された課題票情報のリスト
	 */
	public static Attachment[] getAttachmentDeleteList(ArrayList oldList, ArrayList newList) {
		
		if(oldList==null || oldList.size()==0) {
			return null;
		}
		Attachment[] oldAttachments = (Attachment[])oldList.toArray(new Attachment[0]);
		
		ArrayList deleteList = new ArrayList();

		
		// 「削除された添付ファイル」の情報を生成する。
		for (int i = 0; i < oldAttachments.length; i++) {
			
			if(!newList.contains(oldAttachments[i])) {
				deleteList.add(oldAttachments[i]);
			}
		}
		
		// 無い場合はNULLを返す。
		if(deleteList.size()==0) {
			return null;
		}
		
		return (Attachment[])deleteList.toArray(new Attachment[0]);
	}

	
	/**
	 * 「優先度」配列取得。
	 * 「プロジェクト設定ダイアログ」にて設定した「優先度」情報をString配列にて取得します。
	 * 
	 * @return			「優先度」情報の配列
	 */
	public static String[] getPriorities() {

		// プロジェクトに設定した優先度情報を取得する
		ProjectConfig config = CurrentProject.getInsance().getProjectConfig();
		String priority = config.getPriority();
		
		// 設定が無いとき
		if(priority==null || priority.equals("")) {
			return new String[]{""};
		}
	
		// String配列にして返す。
		String[] priorities = priority.split("[|]");
		for (int i = 0; i < priorities.length; i++) {
			// 空白はtrimしておく
			priorities[i] = priorities[i].trim();
		}
		return priorities;
	}
	
	/**
	 * 「カテゴリ」配列取得。
	 * 「プロジェクト設定ダイアログ」にて設定した「カテゴリ」情報をString配列にて取得します。
	 * 
	 * @return			「カテゴリ」情報の配列
	 */
	public static String[] getCategories() {

		// プロジェクトに設定した優先度情報を取得する
		ProjectConfig config = CurrentProject.getInsance().getProjectConfig();
		String category = config.getCategory();
		
		// 設定が無いとき
		if(category==null || category.equals("")) {
			return new String[]{""};
		}
	
		// String配列にして返す。
		String[] categories = category.split("[|]");
		for (int i = 0; i < categories.length; i++) {
			// 左右の空白はtrimしておく
			categories[i] = categories[i].trim();
		}
		return categories;
	}
	
	
	/**
	 * 「担当者」配列取得。
	 * 「プロジェクト設定ダイアログ」にて設定した「担当者」情報をString配列にて取得します。
	 * 
	 * @return			「担当者」情報の配列
	 */
	public static String[] getAssigned() {

		// プロジェクトに設定した担当者情報を取得する
		ProjectConfig config = CurrentProject.getInsance().getProjectConfig();
		String assigned = config.getAssigned();
		
		// 設定が無いとき
		if(assigned==null || assigned.equals("")) {
			return new String[]{""};
		}
	
		// String配列にして返す。
		String[] allAssigned = assigned.split("[|]");
		for (int i = 0; i < allAssigned.length; i++) {
			// 左右の空白はtrimしておく
			allAssigned[i] = allAssigned[i].trim();
		}
		return allAssigned;
	}
	
	
	/**
	 * 2つのStringを比較します。
	 * <code>diff1</code>がnullでも結果を返します。
	 * 
	 * @param		diff1			比較対照文字列１
	 * @param		diff2			比較対照文字列２
	 * @return		{diff1=null,diff2=null}の場合--true。
	 * 				{diff1=null,diff2=notNull}の場合--false。
	 * 				{diff1=notNll}の場合--diff1.equals(diff2)の結果
	 */
	private static boolean equals(String diff1, String diff2) {
		
		if(diff1 ==null && diff2 == null) {
			return true;
		} else if(diff1 == null && diff2 !=null) {
			return false;
		} else {
			return diff1.equals(diff2);
		}
	}
}
