/*
 * Paraselene
 * Copyright (c) 2009  Akira Terasaki
 * このファイルは同梱されているLicense.txtに定めた条件に同意できる場合にのみ
 * 利用可能です。
 */
package paraselene.tag;

import java.util.*;
import java.io.*;
import java.net.*;
import paraselene.*;
import paraselene.supervisor.*;
import paraselene.tag.attr.*;

/**
 * HTML汎用タグ。
 */
public class Tag extends HTMLPart {
	private static final long serialVersionUID = 1L;
	private String name;
	protected HashMap<String,Attribute>	attr = new HashMap<String,Attribute>();
	private ArrayList<HTMLPart>		doc = new ArrayList<HTMLPart>();
	private ArrayList<Tag>		tag_only = new ArrayList<Tag>();
	private boolean nest_f = false;
	private boolean visible_f = true;

	private static String	VALUE = "value";

	protected Tag() {}

	private static final String[]	plain_tag = new String[] {
		"code",
		"comment",
		"plaintext",
		"xmp",
		"script",
		"style",
		"listing",
	};
	private static HashMap<String,String>	plain_map = new HashMap<String,String>();

	static {
		for ( int i = 0; i < plain_tag.length; i++ ) {
			plain_map.put( plain_tag[i], plain_tag[i] );
		}
	}

	/**
	 * エスケープしないタグであるか？
	 * @param name タグ名。
	 * @return true:エスケープしない、false:エスケープする。
	 */
	public static boolean isPlainTag( String name ) {
		return plain_map.get( name.toLowerCase( Locale.ENGLISH ) ) != null;
	}

	/**
	 * 閉じタグの有無。
	 * @return true:閉じタグなし、false:閉じタグあり。
	 */
	public boolean isSimpleTag() {
		return !nest_f;
	}

	/**
	 * 可視であるか？
	 * @return true:可視、false:不可視。
	 */
	public boolean isVisible() {
		return visible_f;
	}

	/**
	 * 可視設定。
	 * @param flag true:可視、false:不可視。
	 */
	public void setVisible( boolean flag ) {
		visible_f = flag;
	}

	/**
	 * コンストラクタ。
	 * @param n タグ名。
	 * @param simple_f true:閉じタグ無し、false:閉じタグ有り。
	 */
	public Tag( String n, boolean simple_f ) {
		name = n.toLowerCase( Locale.ENGLISH );
		nest_f = !simple_f;
	}

	/**
	 * name属性の取得。
	 * @return name属性に設定されている名前。
	 */
	public String getNameAttribute() {
		try {
			Attribute	n = getAttribute( "name" );
			if ( n == null )	return null;
			Text	t = n.get();
			if ( t == null )	return null;
			return t.toString();
		}
		catch( Exception e ) {
			return null;
		}
	}

	/**
	 * 属性の全取得。
	 * @return 属性配列。属性が無い場合、0個の配列を返します。
	 */
	public Attribute[] getAllAttribute() {
		ArrayList<Attribute>	list = new ArrayList<Attribute>();
		for ( String k : attr.keySet() ) {
			list.add( attr.get( k ) );
		}
		return list.toArray( new Attribute[0] );
	}

	/**
	 * 複製の作成。<br>
	 * 複製したものは、name属性値も引き継がれるので
	 * addHTMLPart()する時に配列化にされる場合があります。
	 * @return 複製。
	 */
	public HTMLPart getReplica() {
		Tag	tag = newReplica();
		tag.visible_f = this.visible_f;
		for ( String k : attr.keySet() ) {
			tag.setAttribute( attr.get( k ).getReplica() );
		}
		int	size = doc.size();
		for ( int i = 0; i < size; i++ ) {
			tag.addHTMLPart( doc.get( i ).getReplica() );
		}
		return tag;
	}

	/**
	 * 複製用インスタンスの生成。
	 * @return 自身と同等のインスタンス。
	 */
	protected Tag newReplica() {
		return new Tag( name, !nest_f );
	}

	/**
	 * タグ名の取得。小文字で返す。
	 * @return タグ名。
	 */
	public String getName() {
		return name;
	}

	/**
	 * 属性の追加/置換。
	 * @param at 属性。
	 */
	public void setAttribute( Attribute at ) {
		attr.put( at.getName(), at );
	}

	/**
	 * 複数の属性の追加/置換。
	 * @param at 属性。
	 */
	public void setAttribute( Attribute ... at ) {
		for ( int i = 0; i < at.length; i++ ) {
			setAttribute( at[i] );
		}
	}

	/**
	 * 属性の削除。
	 * @param at_name 属性名。
	 */
	public void removeAttribute( String at_name ) {
		attr.remove( at_name.toLowerCase( Locale.ENGLISH ) );
	}

	/**
	 * 属性の取得。無ければnullを返す。
	 * @param at_name 属性名。
	 */
	public Attribute getAttribute( String at_name ) {
		return attr.get( at_name.toLowerCase( Locale.ENGLISH ) );
	}

	/**
	 * HTML文書の追加。name属性値が衝突すると、それを変更します。
	 * 末尾に追加されます。
	 * @param d 文書。
	 */
	public void addHTMLPart( HTMLPart d ) {
		addHTMLPart( doc.size(), d );
	}

	/**
	 * HTML文書の追加。name属性値が衝突すると、それを変更します。
	 * @param idx 追加位置。
	 * @param d 文書。
	 */
	public void addHTMLPart( int idx, HTMLPart d ) {
		doc.add( idx, d );
		if ( d instanceof Tag ) {
			Tag tag = (Tag)d;
			if ( embed != null ) {
				embed.nameTag( (Tag)d );
			}
			int	cnt = doc.size();
			idx = 0;
			for ( int i = 0; i < cnt; i++ ) {
				HTMLPart	p = doc.get( i );
				if ( p == d )	break;
				if ( p instanceof Tag ) {
					idx++;
				}
			}
			tag_only.add( idx, tag );
		}
		else {
			d.embed = this.embed;
		}
	}

	/**
	 * 複数のHTML文書の追加。name属性値が衝突すると、それを変更します。
	 * 末尾に追加されます。
	 * @param d 文書。
	 */
	public void addHTMLPart( HTMLPart ... d ) {
		for ( int i = 0; i < d.length; i++ ) {
			addHTMLPart( d[i] );
		}
	}

	/**
	 * 複数のHTML文書の追加。name属性値が衝突すると、それを変更します。
	 * 指定位置から連続挿入されます。
	 * @param idx 追加位置。
	 * @param d 文書。
	 */
	public void addHTMLPart( int idx, HTMLPart ... d ) {
		for ( int i = 0; i < d.length; i++ ) {
			addHTMLPart( idx + i, d[i] );
		}
	}

	/**
	 * タグのインクルード。<br>
	 * タグで挟まれた部分全てを取り込む。そのタグ自体は取り込まない。
	 * @param tag 取り込むタグ。
	 */
	public void include( Tag tag ) {
		if ( tag == null )	return;
		int	cnt = tag.getHTMLPartCount();
		for ( int i = 0; i < cnt; i++ ) {
			addHTMLPart( tag.getHTMLPart( i ) );
		}
	}

	/**
	 * HTMLのインクルード。<br>
	 * bodyタグの内側部分全てを取り込む。
	 * @param page 取り込むページ。
	 */
	public void include( Page page ) {
		Tag[]	tag = page.getAllTagByType( "body" );
		if ( tag == null )	return;
		if ( tag.length == 0 )	return;
		include( tag[0] );
	}

	/**
	 * HTML文書の取得。
	 * @param idx 取得位置。
	 */
	public HTMLPart getHTMLPart( int idx ) {
		return doc.get( idx );
	}

	/**
	 * HTML文書件数の取得。
	 * @return 件数。
	 */
	public int getHTMLPartCount() {
		return doc.size();
	}

	/**
	 * HTML文書の削除。
	 * @param idx 削除位置。
	 */
	public void removeHTMLPart( int idx ) {
		HTMLPart	p = doc.remove( idx );
		if ( p == null )	return;
		removeHTMLPart( p );
	}

	/**
	 * HTML文書の削除。最初に登場した指定インスタンスを削除します。
	 * @param part 削除対象。
	 */
	public void removeHTMLPart( HTMLPart part ) {
		if ( part == null )	return;
		if ( part instanceof Tag ) {
			Tag	tag = (Tag)part;
			tag_only.remove( tag );
			if ( embed != null ) {
				embed.removeNameEntry( tag );
			}
		}
		else {
			part.embed = null;
		}
		doc.remove( part );
	}

	/**
	 * HTML文書の全削除。
	 */
	public void removeHTMLPart() {
		int	cnt = doc.size();
		for ( cnt--; cnt >= 0; cnt-- ) {
			removeHTMLPart( cnt );
		}
	}

	/**
	 * タグの除外。
	 * removeHTMLPartと異なり、除去対象タグの内部のデータが残存します。
	 * @param tag 除去対象。
	 */
	public void exclude( Tag tag ) {
		int	idx = indexOf( tag );
		removeHTMLPart( tag );
		int	cnt = tag.getHTMLPartCount();
		for ( int i = cnt - 1; i >= 0; i-- ) {
			addHTMLPart( idx, tag.getHTMLPart( i ) );
		}
	}

	/**
	 * 位置を返す。最初に登場した指定インスタンスの位置を返します。
	 * @param part 検出対象。無ければ-1を返す。
	 */
	public int indexOf( HTMLPart part ) {
		if ( part == null ) return -1;
		return doc.indexOf( part );
	}

	/**
	 * 開始タグの文字列。
	 * @return 開始タグ。
	 */
	protected String getFirstTag() {
		boolean	xml = false;
		if ( embed != null ) {
			xml = embed.isXML();
		}
		StringBuffer	buf = new StringBuffer( "<" );
		buf = buf.append( getName() );
		for ( String k : attr.keySet() ) {
			buf = buf.append( " " );
			buf = buf.append( attr.get( k ).toHtmlString( xml ) );
		}
		if ( xml && isSimpleTag() ) {
			buf = buf.append( " /" );
		}
		buf = buf.append( ">" );
		return buf.toString();
	}

	/**
	 * 終了タグの文字列。
	 * @return 終了タグ。
	 */
	protected String getLastTag() {
		StringBuffer	buf = new StringBuffer( "</" );
		buf = buf.append( name );
		buf = buf.append( ">" );
		return buf.toString();
	}

	/**
	 * モードの決定。
	 * @param tag_name タグ名。
	 * @return モード。
	 */
	public static StringMode selectMode( String tag_name ) {
		tag_name = tag_name.toLowerCase( Locale.ENGLISH );
		StringMode	mode = StringMode.BODY;

		if ( "textarea".equals( tag_name ) || "pre".equals( tag_name ) ) {
			mode = StringMode.TEXTAREA;
		}
		else if ( "td".equals( tag_name ) || "th".equals( tag_name ) ) {
			mode = StringMode.TABLE;
		}
		else if ( isPlainTag( tag_name ) ) {
			mode = StringMode.PLAIN;
		}
		return mode;
	}

	/**
	 * モードの決定。
	 * @retrun モード。
	 */
	private StringMode selectMode() {
		return selectMode( name );
	}

	/**
	 * 文書の出力。
	 * @retrun 文書の文字列化。
	 */
	private String getHTMLParts() {

		int	cnt = getHTMLPartCount();
		StringBuffer	buf = new StringBuffer();
		StringMode	mode = selectMode();
		for ( int i = 0; i < cnt; i++ ) {
			buf = buf.append( getHTMLPart( i ).toString( mode ) );
		}
		return buf.toString();
	}

	/**
	 * 文字列化。
	 * @param mode 文字列種別。
	 * @return 文字列。
	 */
	public String toString( StringMode mode ) {
		if ( !visible_f )	return "";
		StringBuffer	buf = new StringBuffer( "\n" );
		buf = buf.append( getFirstTag() );
		buf = buf.append( getHTMLParts() );
		if ( nest_f ) {
			buf = buf.append( getLastTag() );
		}
		return buf.toString();
	}

	/**
	 * 出力。
	 * @param w ライター。
	 * @param mode 文字列種別。
	 */
	public void write( PrintWriter w, StringMode mode ) {
		if ( !visible_f )	return;
		w.print( getFirstTag() );
		int	cnt = getHTMLPartCount();
		if ( mode != StringMode.TEXTAREA )	mode = selectMode();
		for ( int i = 0; i < cnt; i++ ) {
			getHTMLPart( i ).write( w, mode );
		}
		if ( nest_f ) {
			w.print( getLastTag() );
		}
	}

	/**
	 * value属性値の取得。
	 * @return 文字列。
	 */
	protected String getValueAttribute() {
		Attribute	attr = getAttribute( VALUE );
		if ( attr == null )	return null;
		Text	text = attr.get();
		if ( text == null )	return null;
		return text.toString( StringMode.PLAIN );
	}

	/**
	 * 値の取得。<br><br>
	 * 閉じタグを持たないものであれば、value属性値を返します。<br>
	 * ただし、value属性値がなければnullを返します。<br><br>
	 * 閉じタグを持つものであれば、タグで括られた最初の文字列を返します。<br>
	 * タグで括られた内部にText要素が無ければnullを返します。
	 * @return 文字列。
	 */
	public String getValueString() {
		if  ( nest_f ) {
			int	cnt = getHTMLPartCount();
			for ( int i = 0; i < cnt; i++ ) {
				HTMLPart	pt = getHTMLPart( i );
				if ( pt instanceof Text ) {
					return ((Text)pt).toString( StringMode.PLAIN );
				}
			}
			return null;
		}
		return getValueAttribute();
	}

	/**
	 * value属性への値設定。
	 * @param v 設定値。
	 */
	protected void setValueAttribute( String v ) {
		Attribute	attr = getAttribute( VALUE );
		if ( attr == null ) {
			try {
				setAttribute( new Attribute( VALUE, v ) );
			}
			catch( Exception e ) {}
			return;
		}
		attr.set( v );
	}

	/**
	 * 値の設定。<br>
	 * 閉じタグを持たないものであれば、value属性値に設定します。<br>
	 * 閉じタグを持つものであれば、タグ内部に持たせます。
	 * @param v 設定値。
	 */
	public void setValueString( String v ) {
		if ( nest_f ) {
			int	cnt = getHTMLPartCount();
			for ( int i = 0; i < cnt; i++ ) {
				HTMLPart	pt = getHTMLPart( i );
				if ( pt instanceof Text ) {
					((Text)pt).setText( v );
					return;
				}
			}
			addHTMLPart( new Text( v ) );
			return;
		}
		setValueAttribute( v );
	}

	/**
	 * 値を持っているか？
	 * @return true:空文字列かnull値である、false:値を持っている。
	 */
	public boolean isEmptyValue() {
		String	str = getValueString();
		if ( str == null )	return true;
		return str.isEmpty();
	}

	/**
	 * 内包タグの列挙。得られるタグはthisが直接保持しているものだけです。
	 * 再帰的に全てのタグを探すわけではありません。
	 * @return タグ。無ければ0個の配列を返す。
	 */
	public Tag[] getTagArray() {
		return tag_only.toArray( new Tag[0] );
	}

	/**
	 * 内包タグの列挙。
	 * 指定クラスと同一、または指定クラスから派生したクラスを取り出し
	 * 配列にして返します。再帰的に探します。
	 * @param cls 探すクラス。Tagの派生クラスを指定します。
	 * @return タグ。無ければ0個の配列を返す。
	 * ただし、clsがnullなら、nullを返す。
	 */
	public Tag[] getAllTagByClass( Class<?> cls ) {
		if ( cls == null )	return null;
		ArrayList<Tag>	list = new ArrayList<Tag>();
		Tag[]	crt = getTagArray();
		for ( int i = 0; i < crt.length; i++ ) {
			Tag[]	sub = crt[i].getAllTagByClass( cls );
			for ( int j = 0; j < sub.length; j++ ) {
				list.add( sub[j] );
			}
			if ( cls.isAssignableFrom( crt[i].getClass() ) ) {
				list.add( crt[i] );
			}
		}
		return list.toArray( new Tag[0] );
	}

	/**
	 * タグの取得。AやTABLEのような、タグの種別で探します。再帰的に探します。
	 * @param name タグの種別。
	 * @return タグの配列。無ければ0個の配列を返す。
	 * ただし、nameがnullなら、nullを返す。
	 */
	public Tag[] getAllTagByType( String name ) {
		if ( name == null )	return null;
		name = name.toLowerCase( Locale.ENGLISH );
		ArrayList<Tag>	list = new ArrayList<Tag>();
		Tag[]	crt = getTagArray();
		for ( int i = 0; i < crt.length; i++ ) {
			Tag[]	sub = crt[i].getAllTagByType( name );
			for ( int j = 0; j < sub.length; j++ ) {
				list.add( sub[j] );
			}
			if ( name.equals( crt[i].getName() ) ) {
				list.add( crt[i] );
			}
		}
		return list.toArray( new Tag[0] );
	}

	/**
	 * タグの取得。AやTABLEのような、タグの種別で探します。
	 * 再帰的に探し、最初に見つかったインスタンスを返します。
	 * @param name タグの種別。
	 * @return タグ。見つからなければnull。
	 */
	public Tag getFirstTagByType( String name ) {
		if ( name == null )	return null;
		name = name.toLowerCase( Locale.ENGLISH );
		Tag[]	crt = getTagArray();
		for ( int i = 0; i < crt.length; i++ ) {
			if ( name.equals( crt[i].getName() ) )	return crt[i];
			Tag	sub = crt[i].getFirstTagByType( name );
			if ( sub != null )	return sub;
		}
		return null;
	}

	/**
	 * インスタンスを内包しているか？
	 * @param target 検索対象。
	 * @return true:持っている、false:持っていない。
	 */
	public boolean isInner( HTMLPart target ) {
		return getDirectInnerTag( target ) != null;
	}

	/**
	 * 指定インスタンスを直接保持しているタグを返す。
	 * @param target 検索対象。
	 * @return 直接保持しているタグ。無ければnull。
	 */
	public Tag getDirectInnerTag( HTMLPart target ) {
		if ( this == target )	return null;
		int	cnt = doc.size();
		for ( int i = 0; i < cnt; i++ ) {
			HTMLPart	part = doc.get( i );
			if ( part == target )	return this;
			if ( part instanceof Tag ) {
				Tag	tag = ((Tag)part).getDirectInnerTag( target );
				if ( tag != null )	return tag;
			}
		}
		return null;
	}

	/**
	 * 配列の中にthisを含むか？
	 * @param tag 検索する配列。
	 * @return true:含む、false:含まない。
	 */
	public boolean isContain( Tag[] tag ) {
		if ( tag == null )	return false;
		for ( int i = 0; i < tag.length; i++ ) {
			if ( this == tag[i] )	return true;
		}
		return false;
	}

	/**
	 * テキストの生成。
	 */
	protected Text makeText( String str ) {
		return new Text( str );
	}

	/**
	 * BRタグを改行コードに置換する。
	 */
	public void replaceBR() {
		int	cnt = getHTMLPartCount();
		for ( int i = 0; i < cnt; i++ ) {
			HTMLPart	p = getHTMLPart( i );
			if ( !(p instanceof Tag) )	continue;
			Tag	tag = (Tag)p;
			if ( !"br".equals( tag.getName() ) )	continue;
			if ( tag.getAllAttribute().length > 0 )	continue;
			removeHTMLPart( i );
			addHTMLPart( i, makeText( "\n" ) );
		}

		for ( int i = cnt - 1; i > 0; i-- ) {
			HTMLPart	p = getHTMLPart( i );
			if ( !(p instanceof Text) )	continue;
			HTMLPart	p2 = getHTMLPart( i - 1 );
			if ( !(p2 instanceof Text) )	continue;
			Text	next = (Text)p, pre = (Text)p2;
			pre.append( next );
			removeHTMLPart( i );
		}
	}

}

