/*
 * Paraselene
 * Copyright (c) 2009-2012  Akira Terasaki
 * このファイルは同梱されているLicense.txtに定めた条件に同意できる場合にのみ
 * 利用可能です。
 */
package paraselene.dyna;

import paraselene.*;
import paraselene.supervisor.*;
import java.net.*;
import java.util.*;
import java.io.*;


/**
 * HTML相対パスと PageID の対応表。
 */
public class Dominion {
	/**
	 * 相対パス対応表。パスとページの対応を表します。
	 */
	public static class RelativePage {
		private String uri;
		private PageID page_id;
		private boolean dl_f;
		/**
		 * コンストラクタ。
		 * @param path htmlとしての相対パス。
		 * @param id 対応するページID。
		 * @param download true:Downloadable、false:通常ページ。
		 */
		public RelativePage( String path, PageID id, boolean download ) {
			uri = path;
			page_id = id;
			dl_f = download;
		}
	}

	private HashMap<String, RelativePage>	map_f2p = new HashMap<String, RelativePage>();
	private HashMap<Integer, String>	map_p2f = new HashMap<Integer, String>();

	/**
	 * コンストラクタ。
	 * @param path 登録する相対パス。
	 */
	public Dominion( RelativePage ... path ) {
		for ( RelativePage r: path ) {
			map_f2p.put( r.uri, r );
			map_p2f.put( r.page_id.getID(), r.uri );
		}
	}

	/**
	 * パス解決。
	 */
	public static class Path {
		/**
		 * パス種別。
		 */
		public enum Type {
			/** 
			 * 絶対パス。
			 * /で始まるものや、完全修飾URIです。
			 */
			ABSOLUTE,
			/**
			 * アプリケーションのページ。
			 * PageID へ解決できたページへのURIです。
			 */
			PAGE,
			/**
			 * Downloadable。
			 * PageID は判明していますが、返すURIはダミーです。<br>
			 * 実行時にURIを正しいものに差し替える必要があります。
			 */
			DOWNLOADABLE,
			/**
			 * アプリケーション別名ページへの相対パス。
			 * .na で終わっている相対パスです。
			 */
			ALIAS,
			/**
			 * 相対パス。その他の相対パスです。
			 * css や 画像等になります。
			 */
			RELATIVE
		}
		private Type type;
		private String uri;
		private Path( Type t, String s, String dec ) throws Exception {
			type = t;
			uri = URLDecoder.decode( s, dec );
		}
		/**
		 * パス種別取得。
		 * @return パス種別。
		 */
		public Type getType() { return type; }
		/**
		 * パスの取得。
		 * @return パス文字列。
		 */
		public String getPath() { return uri; }
	}


	/**
	 * HTMLファイルのパス生成。
	 * @param parent 親ディレクトリの絶対パス。
	 * これがnull以外ならば親ディレクトリに連結したパスを返します。
	 * @param id ページID。
	 * @return HTMLのパス。
	 * parentの有無により、絶対パス、相対パスのどちらかを返します。<br>
	 * 絶対パスの場合はOSに合わせたパス表記、
	 * 相対パスの場合はURLでのパス表記になります。
	 */
	public String getHTMLPath( String parent, PageID id ) {
		String	child = map_p2f.get( id.getID() );
		if ( parent == null )	return child;
		return fix( parent, child, false );
	}

	private static boolean isParaselene( String path ) {
		for ( String s: path.split( "/" ) ) {
			int	len = s.length();
			if ( len < 4 )	continue;
			if ( s.substring( len - 3 ).equals( TransactionSequencer.EXTENSION ) )	return true;
		}
		return false;
	}

	/**
	 * フレームワークが使用します。
	 */
	public static String fix( String top, String path, boolean url_f ) {
		if ( new File( path ).isAbsolute() )	return path;
		if ( top == null )	return path;
		ArrayList<String>	dir = new ArrayList<String>();
		for ( String s: top.split( "[/\\\\]" ) )	dir.add( s );
		int	size;
		for ( String s: path.split( "[/\\\\]" ) ) {
			if ( s.length() == 0 )	continue;
			if ( ".".equals( s ) )	continue;
			if ( "..".equals( s ) ) {
				size = dir.size();
				if ( size == 0 )	return null;
				dir.remove( size - 1 );
				continue;
			}
			dir.add( s );
		}
		size = dir.size();
		if ( size == 0 )	return null;
		String	sep = url_f?	"/":	File.separator;
		StringBuilder	buf = new StringBuilder( dir.get( 0 ) );
		for ( int  i = 1; i < size; i++ ) {
			buf = buf.append( sep ).append( dir.get( i ) );
		}
		return buf.toString();
	}

	/**
	 * パス解決。
	 * 入力パスを判定し、適切なパス文字列へ置換して返します。
	 * @param id 起点となるページ。
	 * @param path 判定するパス。
	 * @param dec デコード文字列。
	 * @return パス情報。
	 * @exception Exception パス指定が不正。
	 */
	public Path resolve( PageID id, String path, String dec ) throws Exception {
		URI	uri = new URI( path );
		if ( uri.getScheme() != null )	return new Path( Path.Type.ABSOLUTE, path, dec );
		if ( path.charAt( 0 ) == '/' )	return new Path( Path.Type.ABSOLUTE, path, dec );
		path = fix( new File( getHTMLPath( null, id ) ).getParent(), uri.getPath(), true );
		if ( path == null )	throw new Exception( getHTMLPath( null, id ) + "  /  " + uri.getPath() + "  -> failed" );
		RelativePage	rp = map_f2p.get( path );
		if ( rp != null ) {
			if ( rp.dl_f ) {
				return new Path( Path.Type.DOWNLOADABLE,
					URIValue.pageToDownloadURI( rp.page_id, "data" ),
				dec );
			}
			else {
				QueryItem[]	item = new URIValue( URLDecoder.decode( path, dec ) ).getQuery();
				return new Path( Path.Type.PAGE,
					URIValue.pageToURI( id, uri.getFragment(), item ),
				dec );
			}
		}
		return new Path( isParaselene( path )?
			Path.Type.ALIAS: Path.Type.RELATIVE, path, dec );
	}
}


