/*
 * Paraselene
 * Copyright (c) 2009-2012  Akira Terasaki
 * このファイルは同梱されているLicense.txtに定めた条件に同意できる場合にのみ
 * 利用可能です。
 */
package paraselene.util;

/**
 * IPアドレス。
 */
public abstract class IP implements java.io.Serializable {
	private static final long serialVersionUID = 1L;
	/**
	 * IPアドレス表記エラー。
	 */
	public static class IPFormatException extends Exception {
		IPFormatException( String str ) {
			super( makeStr( str ) );
		}
		private static String makeStr( String str ) {
			StringBuilder	buf = new StringBuilder( "Bad format[" );
			buf = buf.append( str );
			buf = buf.append( "]" );
			return buf.toString();
		}
	}

	/**
	 * IPアドレスインスタンス生成。
	 * @param ip IPアドレス文字列。v4かv6かは自動判別します。
	 * @return インスタンス。ip がIPアドレス文字列でない場合、null。
	 */
	public static IP toIP( String ip ) {
		IP	ret = null;
		try { ret = new IPv4( ip ); }
		catch( IP.IPFormatException e4 ) {
			try { ret = new IPv6( ip ); }
			catch( IP.IPFormatException e6 ) {}
		}
		return ret;
	}

	/**
	 * IPアドレスとネットマスク。<br>
	 * 次のような文字列をIPアドレスとネットマスクに分割し、
	 * その値を保持します。
	 * <ul>
	 * <li>192.168.1.1/24
	 * <li>2001:d0b::1/48
	 * </ul>
	 * IPアドレスがv4かv6かは自動判別します。
	 * /以降は、10進数のビット長を表すネットマスクとみなします。<br>
	 * /が与えられてない場合、アドレス全体を表すビット長(IPv4なら32)と
	 * なります。
	 */
	public static class IPMask implements java.io.Serializable {
		private static final long serialVersionUID = 1L;
		private IP	ip;
		private int	mask = -1;
		private IPMask(){}
		/**
		 * コンストラクタ。
		 * @param ip_addr IPアドレス。
		 * @param net_mask ネットマスク。
		 */
		public IPMask( IP ip_addr, int net_mask ) {
			ip = ip_addr;
			mask = net_mask;
		}
		/**
		 * コンストラクタ。
		 * @param str IPアドレス/ネットマスク 形式の文字列。
		 */
		public IPMask( String str ) throws IP.IPFormatException {
			String[]	sep = str.split( "/" );
			if ( sep.length > 2 || sep.length < 1 )	throw new IP.IPFormatException( str );
			ip = IP.toIP( sep[0] );
			if ( ip == null )	throw new IP.IPFormatException( str );
			if ( sep.length == 2 ) {
				try {
					mask = Integer.parseInt( sep[1] );
				}
				catch( NumberFormatException e ) {
					throw new IP.IPFormatException( str );
				}
			}
			else	mask = ip.addr.length * 8;
		}
		/**
		 * IPアドレスの取得。
		 * @return IPアドレス。
		 */
		public IP getIP() { return ip; }
		/**
		 * ネットマスクの取得。
		 * @return ネットマスク。
		 */
		public int getMask() { return mask; }
		/**
		 * 文字列化。
		 * @return 文字列。
		 */
		public String toString() {
			StringBuilder	buf = new StringBuilder( ip.toString() );
			buf = buf.append( "/" ).append( mask );
			return buf.toString();
		}
	}

	int[]	addr;
	IP(){}
	IP( int[] ip ) { addr = ip; }
	IP( IP ip ) {
		this( ip.addr.clone() );
	}

	abstract IP getReplica();

	/**
	 * ネットマスク適用。指定ビット長をアドレスに適用します。<br>
	 * 例えば、192.168.1.255 に ビット長 24 を与えると、
	 * 192.168.1.0 のアドレスを返します。
	 * @param bit_length ビット長。
	 * @return ネットマスク適用後のアドレス。
	 */
	public IP mask( int bit_length ) {
		IP	ip = getReplica();
		int	max_length = ip.addr.length * 8;
		if ( bit_length < 0 )	bit_length = 0;
		if ( bit_length > max_length )	bit_length = max_length;
		for ( int no = 0; no < ip.addr.length; no++ ) {
			if ( bit_length > 0 ) {
				if ( bit_length < 8 ) {
					int	bit = 0x80;
					int	result = 0;
					for ( ; bit_length > 0; bit_length--, bit >>= 1 ) {
						result |= (bit & ip.addr[no]);
					}
					ip.addr[no] = result;
				}
				else	bit_length -= 8;
			}
			else	ip.addr[no] = 0;
		}
		return ip;
	}

	/**
	 * 比較。ビット長が合わない場合も不一致とみなされます。
	 * @param obj 比較対象。
	 * @return true:アドレス一致、false:アドレス不一致。
	 */
	public boolean equals( Object obj ) {
		if ( !(obj instanceof IP) )	return false;
		IP	cmp = (IP)obj;
		if ( cmp.addr.length != addr.length )	return false;
		return equalsLowBit( cmp );
	}

	/**
	 * 比較。<br>
	 * equals と異なり、ビット長の異なるアドレス同士でも比較を行います。<br>
	 * ビット長が異なる場合、ビット長が長い方のアドレスの下位ビットと、
	 * ビット長が短い方のアドレス全体を比較します。
	 * @param ip 比較対象。
	 * @return true:アドレス一致、false:アドレス不一致。
	 */
	public boolean equalsLowBit( IP ip ) {
		int	ip_start = 0;
		int	my_start = 0;
		if ( ip.addr.length > addr.length ) {
			ip_start = ip.addr.length - addr.length;
		}
		else if ( ip.addr.length < addr.length ) {
			my_start = addr.length - ip.addr.length;
		}
		for ( ; my_start < addr.length; ip_start++, my_start++ ) {
			if ( addr[my_start] != ip.addr[ip_start] )	return false;
		}
		return true;
	}

	/**
	 * ハッシュコード取得。
	 * @return ハッシュコード。
	 */
	public int hashCode() {
		int	ret = 0;
		for ( int i = 0; i < addr.length; i += 4 ) {
			int	tmp = 0;
			for ( int j = 0; j < 4; j++ ) {
				tmp <<= 8;
				tmp |= addr[i * 4 + j];
			}
			ret ^= tmp;
		}
		return ret;
	}
}

