<?php
/* vim: set tabstop=4 shiftwidth=4: */

/*
 * PEAR Log Property Configurator
 *
 *
 * PHP version 5
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 * 	http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * @category   PHP
 * @package    Commons
 * @author     Yomei Komiya
 * @copyright  2008 the original author or authors.
 * @license    http://www.apache.org/licenses/LICENSE-2.0  Apache License 2.0
 * @version    SVN: $Id: PEARLogPropertyConfigurator.php 240 2008-02-24 06:03:59Z whitestar $
 * @link       http://phpcommons.sourceforge.jp/
 * @see        
 * @since      File available since Release 0.9.0
 */

//namespace Commons::Logging::Impl;

require_once 'Commons/Lang/Object.php';
require_once 'Commons/Lang/StdErr.php';
require_once 'Commons/Lang/StdOut.php';
require_once 'Commons/Lang/StringUtils.php';
require_once 'Commons/Util/Properties.php';
require_once 'Commons/Logging/LogConfigurationException.php';
require_once 'Commons/Logging/Impl/LogFactoryImpl.php';
require_once 'Commons/Logging/Impl/PEARLogCategory.php';
/*
use Commons::Lang::Object;
use Commons::Lang::StdErr;
use Commons::Lang::StdOut;
use Commons::Lang::StringUtils;
use Commons::Util::Properties;
use Commons::Logging::LogConfigurationException;
use Commons::Logging::Impl::LogFactoryImpl;
use Commons::Logging::Impl::PEARLogCategory;
*/


/**
 * PEARLogPropertyConfigurator
 *
 *
 * @category   PHP
 * @package    Commons.Logging.Impl
 * @author     Yomei Komiya
 * @copyright  2008 the original author or authors.
 * @license    http://www.apache.org/licenses/LICENSE-2.0  Apache License 2.0
 * @version    Release: 1.0.1.1
 * @link       http://phpcommons.sourceforge.jp/
 * @see        
 * @since      Class available since Release 0.9.0
 */
class Commons_Logging_Impl_PEARLogPropertyConfigurator extends Commons_Lang_Object {
	
	/**
	 * PEARLog priority level inheritance key.
	 */
	const INHERIT = -1;

	/**
	 * The property name used to identify the PEARLog root logger.
	 */
	const ROOT_LOGGER_PREFIX = 'pearlog.rootLogger';
	
	/**
	 * The property name used to identify the PEARLog logger.
	 */
	const LOGGER_PREFIX = 'pearlog.logger';
	
	/**
	 * The property name used to identify the PEARLog additivity.
	 */
	const ADDITIVITY_PREFIX = 'pearlog.additivity';

	/**
	 * The property name used to identify the PEARLog log handler.
	 */
	const HANDLER_PREFIX = 'pearlog.handler';
	
	const INTERNAL_ROOT_NAME = 'pearlog.root';
	
	/**
	 * Configuration propertis
	 *
	 * @var Commons_Util_Properties
	 */
	private static $_confProperties = null;
	
	/**
	 * Handler configuration properties
	 *
	 * @var array
	 */
	private static $_handlerProperties = null;
	
	
	/**
	 * Private constructor
	 */
	private function Commons_Logging_Impl_PEARLogPropertyConfigurator() {
		parent::__construct();
	}
	
	
	public function __destruct() {
	}
	
	
	/**
	 * Builds configuration.
	 *
	 * @param string $confPath configuration file path.
	 * @throws Commons_Logging_LogConfigurationException
	 */
	public static function configure($confPath) {
		$confProps = self::getConfProperties($confPath);
		self::parseCategoryProperties($confProps);
		self::parseHandlerProperties($confProps);
	}
	
	
	public static function getHandlerProperties() {
		return self::$_handlerProperties;
	}
	
	
	/**
	 * Gets configuration properties.
	 *
	 * @return Commons_Util_Properties
	 * @throws Commons_Logging_LogConfigurationException
	 */
	protected static function getConfProperties($confPath) {
		if (is_null(self::$_confProperties)) {
			if (is_null($confPath)) {
				throw new Commons_Logging_LogConfigurationException(
					'No configuration file path setting ('
					. Commons_Logging_Impl_LogFactoryImpl::PEARLOG_CONF_PROPERTY . ').');
			}
			$stream = @fopen($confPath, 'r', true);
			if ($stream === false) {
				throw new Commons_Logging_LogConfigurationException(
					'PEAR Log - Invalid configuration file path ('
					. $confPath . ').');
			}
			$props = new Commons_Util_Properties();
			$props->load($stream);
			fclose($stream);
			
			self::$_confProperties = $props;
		}
		
		return self::$_confProperties;
	}
	
	
	/**
	 * Gets the priority level int-parsed value from the string value.
	 *
	 * @param string $levelStr level string.
	 * @return int priority level.
	 * @throws Commons_Logging_LogConfigurationException
	 */
	protected static function getLevelValue($levelStr) {
		$levelStr = Commons_Lang_StringUtils::upperCase($levelStr);
		switch ($levelStr) {
			case 'EMERG':
			case 'PEAR_LOG_EMERG':
				return PEAR_LOG_EMERG;
				break;
			case 'ALERT':
			case 'PEAR_LOG_ALERT':
				return PEAR_LOG_ALERT;
				break;
			case 'CRIT':
			case 'PEAR_LOG_CRIT':
				return PEAR_LOG_CRIT;
				break;
			case 'ERR':
			case 'PEAR_LOG_ERR':
				return PEAR_LOG_ERR;
				break;
			case 'WARNING':
			case 'PEAR_LOG_WARNING':
				return PEAR_LOG_WARNING;
				break;
			case 'NOTICE':
			case 'PEAR_LOG_NOTICE':
				return PEAR_LOG_NOTICE;
				break;
			case 'INFO':
			case 'PEAR_LOG_INFO':
				return PEAR_LOG_INFO;
				break;
			case 'DEBUG':
			case 'PEAR_LOG_DEBUG':
				return PEAR_LOG_DEBUG;
				break;
			case 'INHERIT':
				return self::INHERIT;
				break;
			default:
				throw new Commons_Logging_LogConfigurationException(
					'PEAR Log - Invalid priority level: ' . $levelStr
					. ' (available level: EMERG, ALERT, CRIT, ERR, WARNING, NOTICE, INFO, DEBUG)');
		}
	}
	
	
	/**
	 * Parses category's configuration properties.
	 *
	 * @param Commons_Util_Properties $confProps
	 */
	protected static function parseCategoryProperties(Commons_Util_Properties $confProps) {
		$loggerPrefix = self::LOGGER_PREFIX . '.';
		$loggerPrefixLength = Commons_Lang_StringUtils::strlen($loggerPrefix);
		$additivityPrefix = self::ADDITIVITY_PREFIX . '.';
		$additivityPrefixLength = Commons_Lang_StringUtils::strlen($additivityPrefix);
		
		foreach ($confProps->propertyNames() as $name) {
			if (Commons_Lang_StringUtils::startsWith($name, $loggerPrefix)
				|| $name === self::ROOT_LOGGER_PREFIX) {
				$key = null;
				if ($name === self::ROOT_LOGGER_PREFIX) {
					$key = self::INTERNAL_ROOT_NAME;
				}
				else {
					$key = Commons_Lang_StringUtils::substr($name, $loggerPrefixLength);
				}
				
				$loggerProp = $confProps->getProperty($name);
				$loggerHandlers = array_map('trim', explode(',', $loggerProp));
				// PEAR_LOG_EMERG(0) < PEAR_LOG_DEBUG(7)
				$level = self::getLevelValue(array_shift($loggerHandlers));
				
				$category
					= Commons_Logging_Impl_PEARLogCategory::getCategoryByConfName($key);
				$category->setHandlerNames($loggerHandlers);
				$category->setPriority($level);
			}
			elseif (Commons_Lang_StringUtils::startsWith($name, $additivityPrefix)) {
				$key = Commons_Lang_StringUtils::substr($name, $additivityPrefixLength);
				$additive = self::boolValue($confProps->getProperty($name));

				$category
					= Commons_Logging_Impl_PEARLogCategory::getCategoryByConfName($key);
				$category->setAdditivity($additive);
			}
			else {
				// do nothing.
			}
		}
	}
	
	
	/**
	 * Parses handler's configuration properties.
	 *
	 * @param Commons_Util_Properties $confProps properties.
	 * @return array parsed handler configuration.
	 * 	$handlerProps = array(
	 * 		'name1' => array('type' => 'cosole', 'resource' => '', 'conf' => array()),
	 * 		'name2' => array('type' => 'file', 'resource' => 'out.log', 'conf' => array()),
	 * 		...
	 * 		);
	 * @throws Commons_Logging_LogConfigurationException
	 */
	protected static function parseHandlerProperties(Commons_Util_Properties $confProps) {
		if (is_null(self::$_handlerProperties)) {
			$handlerProps = array();
			$prefix = self::HANDLER_PREFIX . '.';
			$prefixLength = Commons_Lang_StringUtils::strlen($prefix);
			foreach ($confProps->propertyNames() as $name) {
				if (Commons_Lang_StringUtils::startsWith($name, $prefix)) {
					$key = Commons_Lang_StringUtils::substr($name, $prefixLength);
					$key = explode('.', $key);
					$keySize = count($key);
					if ($keySize === 1) {
						$handlerProps[$key[0]] = array(
							'type' => $confProps->getProperty($name),
							'resource' => '');
					}
					elseif ($keySize === 2) {
						if ($key[1] === 'resource') {	// string type
							$res = $confProps->getProperty($name);
							$handlerProps[$key[0]]['resource']
								 = defined($res) ? constant($res) : $res;
						}
						else {	// mixed type
							$handlerProps[$key[0]]['conf'][$key[1]]
								= self::getHandlerParameterValue(
									$handlerProps[$key[0]]['type'],
									$key[1],
									$confProps->getProperty($name));
						}
					}
					else {
						throw new Commons_Logging_LogConfigurationException(
							'PEAR Log - Invalid property name: ' . $name);
					}
				}
			}
			self::$_handlerProperties = $handlerProps;
		}
		
		return $handlerProps;
	}
	
	
	/**
	 * Gets the handler configuration parameter's typed value from the value string
	 *
	 * @param string $type handler type.
	 * @param string $paramName parameter name.
	 * @param string $valueStr value string.
	 * @return mixed type-parsed value.
	 * @throws Commons_Logging_LogConfigurationException
	 */
	protected static function getHandlerParameterValue($type, $paramName, $valueStr) {
		switch ($type) {
			case 'console':
				return self::getConsoleParameterValue($paramName, $valueStr);
				break;
			case 'display':
				return self::getDisplayParameterValue($paramName, $valueStr);
				break;
			case 'error_log':
				return self::getErrorLogParameterValue($paramName, $valueStr);
				break;
			case 'file':
				return self::getFileParameterValue($paramName, $valueStr);
				break;
			case 'firebug':
				return self::getFirebugParameterValue($paramName, $valueStr);
				break;
			case 'mail':
				return self::getMailParameterValue($paramName, $valueStr);
				break;
			case 'sql':
				return self::getSQLParameterValue($paramName, $valueStr);
				break;
			case 'sqlite':
				return self::getSQLiteParameterValue($paramName, $valueStr);
				break;
			case 'syslog':
				return self::getSyslogParameterValue($paramName, $valueStr);
				break;
			case 'win':
				return self::getWinParameterValue($paramName, $valueStr);
				break;
			default:
				throw new Commons_Logging_LogConfigurationException(
					'PEAR Log - Invalid handler type: ' . $type
					. ' (available type: console, display, error_log, file, firebug, mail, sql, sqlite, syslog, win)');
		}
	}
	
	
	/**
	 * Getes the console handler's configuration parameter value from the string value.
	 *
	 * @param string $name parameter name.
	 * @param string $valueStr value string.
	 * @return mixed type-parsed value.
	 * @throws Commons_Logging_LogConfigurationException
	 * 	if the parameter or the value string is invalid. 
	 */
	protected static function getConsoleParameterValue($name, $valueStr) {
		switch ($name) {
			case 'stream':
				if ($valueStr === 'STDOUT' || $valueStr === 'php://stdout') {
					return Commons_Lang_StdOut::getPrintStream();
				}
				elseif ($valueStr === 'STDERR' || $valueStr === 'php://stderr') {
					return Commons_Lang_StdErr::getPrintStream();
				}
				else {
					throw new Commons_Logging_LogConfigurationException(
						'PEAR Log - Invalid console handler\'s stream value: ' . $valueStr);
				}
				break;
			case 'buffering':
				return self::boolValue($valueStr);
				break;
			case 'lineFormat':
			case 'timeFormat':
				return $valueStr;
				break;
			default:
				throw new Commons_Logging_LogConfigurationException(
					'PEAR Log - Invalid console handler parameter name: ' . $name
					. ' (available parameters: stream, buffering, lineFormat, timeFormat)');
		}
	}
	
	
	/**
	 * Getes the display handler's configuration parameter value from the string value.
	 *
	 * @param string $name parameter name.
	 * @param string $valueStr value string.
	 * @return mixed type-parsed value.
	 * @throws Commons_Logging_LogConfigurationException
	 * 	if the parameter or the value string is invalid. 
	 */
	protected static function getDisplayParameterValue($name, $valueStr) {
		switch ($name) {
			// all string parameter.
			case 'lineFormat':
			case 'timeFormat':
			case 'error_prepend':
			case 'error_append':
			case 'linebreak':
				return $valueStr;
				break;
			default:
				throw new Commons_Logging_LogConfigurationException(
					'PEAR Log - Invalid display handler parameter name: ' . $name
					. ' (available parameters: lineFormat, timeFormat, error_prepend, error_append, linebreak)');
		}
	}
	
	
	/**
	 * Getes the error_log handler's configuration parameter value from the string value.
	 *
	 * @param string $name parameter name.
	 * @param string $valueStr value string.
	 * @return mixed type-parsed value.
	 * @throws Commons_Logging_LogConfigurationException
	 * 	if the parameter or the value string is invalid. 
	 */
	protected static function getErrorLogParameterValue($name, $valueStr) {
		switch ($name) {
			// all string parameter.
			case 'destination':
			case 'extra_headers':
				return $valueStr;
				break;
			default:
				throw new Commons_Logging_LogConfigurationException(
					'PEAR Log - Invalid error_log handler parameter name: ' . $name
					. ' (available parameters: destination, extra_headers)');
		}
	}
	
	
	/**
	 * Getes the file handler's configuration parameter value from the string value.
	 *
	 * @param string $name parameter name.
	 * @param string $valueStr value string.
	 * @return mixed type-parsed value.
	 * @throws Commons_Logging_LogConfigurationException
	 * 	if the parameter or the value string is invalid. 
	 */
	protected static function getFileParameterValue($name, $valueStr) {
		switch ($name) {
			case 'append':
			case 'locking':
				return self::boolValue($valueStr);
				break;
			case 'mode':
			case 'dirmode':
				return intval($valueStr, 8);
				break;
			case 'eol':
				return stripcslashes($valueStr);
				//return self::eolValue($valueStr);
				break;
			case 'lineFormat':
			case 'timeFormat':
				return $valueStr;
				break;
			default:
				throw new Commons_Logging_LogConfigurationException(
					'PEAR Log - Invalid file handler parameter name: ' . $name
					. ' (available parameters: append, locking, mode, dirmode, eol, lineFormat, timeFormat)');
		}
	}
	
	
	/**
	 * Getes the Firebug handler's configuration parameter value from the string value.
	 *
	 * @param string $name parameter name.
	 * @param string $valueStr value string.
	 * @return mixed type-parsed value.
	 * @throws Commons_Logging_LogConfigurationException
	 * 	if the parameter or the value string is invalid. 
	 */
	protected static function getFirebugParameterValue($name, $valueStr) {
		switch ($name) {
			case 'buffering':
				return self::boolValue($valueStr);
				break;
			case 'lineFormat':
			case 'timeFormat':
				return $valueStr;
				break;
			default:
				throw new Commons_Logging_LogConfigurationException(
					'PEAR Log - Invalid firebug handler parameter name: ' . $name
					. ' (available parameters: buffering, lineFormat, timeFormat)');
		}
	}
	

	/**
	 * Getes the mail handler's configuration parameter value from the string value.
	 *
	 * @param string $name parameter name.
	 * @param string $valueStr value string.
	 * @return mixed type-parsed value.
	 * @throws Commons_Logging_LogConfigurationException
	 * 	if the parameter or the value string is invalid. 
	 */
	protected static function getMailParameterValue($name, $valueStr) {
		switch ($name) {
			// all string parameter.
			case 'from':
			case 'subject':
			case 'preamble':
			case 'lineFormat':
			case 'timeFormat':
				return $valueStr;
				break;
			default:
				throw new Commons_Logging_LogConfigurationException(
					'PEAR Log - Invalid mail handler parameter name: ' . $name
					. ' (available parameters: from, subject, preamble, lineFormat, timeFormat)');
		}
	}
	
	
	/**
	 * Getes the SQL handler's configuration parameter value from the string value.
	 *
	 * @param string $name parameter name.
	 * @param string $valueStr value string.
	 * @return mixed type-parsed value.
	 * @throws Commons_Logging_LogConfigurationException
	 * 	if the parameter or the value string is invalid. 
	 */
	protected static function getSQLParameterValue($name, $valueStr) {
		switch ($name) {
			case 'dsn':
			case 'sql':
				return $valueStr;
				break;
			case 'options':
				eval('$value = ' . $valueStr . ';');
				return $value;
				break;
			//case 'db':	// object
			//	break;
			case 'sequence':
				return $valueStr;
				break;
			case 'identLimit':
				return intval($valueStr);
				break;
			default:
				throw new Commons_Logging_LogConfigurationException(
					'PEAR Log - Invalid sql handler parameter name: ' . $name
					. ' (available parameters: dsn, sql, options, sequence, identLimit)');
		}
	}
	
	
	/**
	 * Getes the SQLite handler's configuration parameter value from the string value.
	 *
	 * @param string $name parameter name.
	 * @param string $valueStr value string.
	 * @return mixed type-parsed value.
	 * @throws Commons_Logging_LogConfigurationException
	 * 	if the parameter or the value string is invalid. 
	 */
	protected static function getSQLiteParameterValue($name, $valueStr) {
		switch ($name) {
			case 'filename':
				return $valueStr;
				break;
			case 'mode':
				return intval($valueStr, 8);
				break;
			case 'persistent':
				return self::boolValue($valueStr);
				break;
			default:
				throw new Commons_Logging_LogConfigurationException(
					'PEAR Log - Invalid sqlite handler parameter name: ' . $name
					. ' (available parameters: filename, mode, persistent)');
		}
	}
	
	
	/**
	 * Getes the syslog handler's configuration parameter value from the string value.
	 *
	 * @param string $name parameter name.
	 * @param string $valueStr value string.
	 * @return mixed type-parsed value.
	 * @throws Commons_Logging_LogConfigurationException
	 * 	if the parameter or the value string is invalid. 
	 */
	protected static function getSyslogParameterValue($name, $valueStr) {
		switch ($name) {
			case 'inherit':
				return self::boolValue($valueStr);
				break;
			default:
				throw new Commons_Logging_LogConfigurationException(
					'PEAR Log - Invalid syslog handler parameter name: ' . $name
					. ' (available parameter: inherit)');
		}
	}
	
	
	/**
	 * Getes the Window handler's configuration parameter value from the string value.
	 *
	 * @param string $name parameter name.
	 * @param string $valueStr value string.
	 * @return mixed type-parsed value.
	 * @throws Commons_Logging_LogConfigurationException
	 * 	if the parameter or the value string is invalid. 
	 */
	protected static function getWinParameterValue($name, $valueStr) {
		switch ($name) {
			case 'title':
				return $valueStr;
				break;
			case 'styles':
				eval('$value = ' . $valueStr . ';');
				return $value;
				break;
			default:
				throw new Commons_Logging_LogConfigurationException(
					'PEAR Log - Invalid win handler parameter name: ' . $name
					. ' (available parameter: title, styles)');
		}
	}
	
	
	/**
	 * Paeses the string to the boolean value.
	 *
	 * @param string $boolStr
	 * @return bool
	 * @throws Commons_Logging_LogConfigurationException
	 */
	protected static function boolValue($boolStr) {
		$tmp = Commons_Lang_StringUtils::upperCase($boolStr);
		if ($tmp === 'TRUE') {
			return true;
		}
		elseif ($tmp === 'FALSE') {
			return false;
		}
		else {
			throw new Commons_Logging_LogConfigurationException(
				'PEAR Log - Invalid boolean value: ' . $boolStr
				. ' (available value: true, false)');
		}
	}
	
	
	/**
	 * Parses the string to the end of line
	 *
	 * @param string $eolStr
	 * @return string
	 */
	protected static function eolValue($eolStr) {
		if ($eolStr === '\n') {
			return "\n";
		}
		elseif ($eolStr === '\r') {
			return "\r";
		}
		elseif ($eolStr === '\r\n') {
			return "\r\n";
		}
		else {
			return Commons_Lang_SystemUtils::getLineSeparator();
		}
	}

}

/*
 * Local variables:
 * tab-width: 4
 * c-basic-offset: 4
 * c-hanging-comment-ender-p: nil
 * End:
 */
?>