#include "config.h"
#include "pipes/pipes.h"
#include <errno.h>
#include <stdlib.h>
#include <fcntl.h>
#include <glob.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <pwd.h>
#include <limits.h>

// skip spaces
static void skip_spaces(char** p)
{
    while(**p == ' ' || **p == '\t') {
        (*p)++;
    }
}

static void skip_to_nextline(char** p, int* sline)
{
    while(**p) {
        if(**p == '\n') {
            (*p)++;
            (*sline)++;
            break;
        }
        else {
            (*p)++;
        }
    }
}

#define PIPES_ENV_MAX 32

typedef struct {
    char* mBuf;
    int mLen;
    int mSize;

    BOOL mEnv;
    BOOL mGlob;
    BOOL mTilda;
    BOOL mMessagePassing;
    BOOL mNullString;
} sBuf;

static void add_char_to_buf(sBuf* buf, char c)
{
    if(buf->mSize <= buf->mLen + 1) {
        buf->mSize *= 1.8;
        buf->mBuf = REALLOC(buf->mBuf, buf->mSize);
    }
    (buf->mBuf)[buf->mLen] = c;
    (buf->mBuf)[buf->mLen+1] = 0;
    buf->mLen++;
}

static void add_str_to_buf(sBuf* buf, char* str)
{
    const int len = strlen(str);
    if(buf->mSize <= buf->mLen + len) {
        buf->mSize = (buf->mSize + len) * 1.8;
        buf->mBuf = REALLOC(buf->mBuf, buf->mSize);
    }
    memcpy(buf->mBuf + buf->mLen, str, len+1);
    buf->mLen += len;
}

static BOOL read_quote(char**p, sBuf* buf, char* sname, int* sline)
{
    (*p)++;

    switch(**p) {
        case 0:
            err_msg("unexpected end. can't quote null.", sname, *sline, buf->mBuf);
            return FALSE;

        case '\n':
            add_char_to_buf(buf, '\n');
            (*sline)++;
            break;

        case 't':
            add_char_to_buf(buf, '\t');
            break;

        case 'n':
            add_char_to_buf(buf, '\n');
            break;

        case 'r':
            add_char_to_buf(buf, '\r');
            break;

        case 'a':
            add_char_to_buf(buf, '\a');
            break;

        case '\\':
            add_char_to_buf(buf, '\\');
            break;

        default:
            add_char_to_buf(buf, **p);
    }

    (*p)++;

    return TRUE;
}

static BOOL block_parse(char** p, char* sname, int* sline, sObject* block, sObject** current_object);
static BOOL read_one_argument(char** p, sBuf* buf, char* sname, int* sline, BOOL expand_quote, sCommand* command, sObject* block, sObject** current_object);

static BOOL read_env(char** p, sBuf* buf, char* sname, int* sline, BOOL expand_quote, sCommand* command, sObject* block, sObject** current_object)
{
    (*p)++;

    enum eLineField lf = gLineField;
    if(**p == '\\') {
        (*p)++;

        switch(**p) {
            case 'a':
                lf = kBel;
                (*p)++;
                break;

            case 'r':
                (*p)++;

                if(**p == '\\') {
                    (*p)++;
                    if(**p == 'n') {
                        (*p)++;

                        lf = kCRLF;
                    }
                    else {
                        err_msg("invalid linefield", sname, *sline, buf->mBuf);
                        return FALSE;
                    }
                }
                else {
                    lf = kCR;
                }
                break;

            case 'n':
                lf = kLF;
                (*p)++;
                break;
            
            default:
                err_msg("invalid linefield", sname, *sline, buf->mBuf);
                return FALSE;
        }
    }

    if(**p == '(') {
        (*p)++;

        sObject* block2 = BLOCK_NEW_STACK();
        if(!block_parse(p, sname, sline, block2, current_object)) {
            sCommand_add_env_block(command, block2, lf);

            SBLOCK(block).mCompletionFlags |= COMPLETION_FLAGS_ENV_BLOCK;
            SBLOCK(block).mCompletionFlags |= command->mEnvsNum & 0xFF;
            return FALSE;
        }

        buf->mEnv = TRUE;

        add_char_to_buf(buf, PARSER_MAGIC_NUMBER_ENV);
        char buf2[128];
        snprintf(buf2, 128, "%d", command->mEnvsNum);
        add_str_to_buf(buf, buf2);
        add_char_to_buf(buf, PARSER_MAGIC_NUMBER_ENV);

        sCommand_add_env_block(command, block2, lf);
    }
    else {
        SBLOCK(block).mCompletionFlags |= COMPLETION_FLAGS_ENV;

        sBuf name;
        memset(&name, 0, sizeof(sBuf));
        name.mBuf = MALLOC(64);
        name.mBuf[0] = 0;
        name.mSize = 64;

        sBuf key;
        memset(&key, 0, sizeof(sBuf));
        key.mBuf = MALLOC(64);
        key.mBuf[0] = 0;
        key.mSize = 64;

        if(**p == '{') {
            (*p)++;

            while(1) {
                if(**p == 0) {
                    err_msg("close } which is used by expand variable", sname, *sline, name.mBuf);
                    FREE(name.mBuf);
                    FREE(key.mBuf);
                    return FALSE;
                }
                else if(**p == '}') {
                    (*p)++;
                    break;
                }
                else {
                    add_char_to_buf(&name, **p);
                    (*p)++;
                }
            }
        }
        else {
            while(**p >= 'a' && **p <= 'z' || **p >= 'A' && **p <= 'Z' || **p == '_' || **p >= '0' && **p <= '9')
            {
                add_char_to_buf(&name, **p);
                (*p)++;
            }
        }

        if(**p == '[') {
            (*p)++;

            if(!read_one_argument(p, &key, sname, sline, TRUE, command, block, current_object)) 
            {
                FREE(name.mBuf);
                if(key.mEnv) {  // for readline
                    sCommand_add_arg_to_command(command, MANAGED key.mBuf, TRUE);
                }
                else {
                    FREE(key.mBuf);
                }
                return FALSE;
            }

            skip_spaces(p);
            if(**p != ']') {
                err_msg("require ] character", sname, *sline, name.mBuf);
                FREE(name.mBuf);
                FREE(key.mBuf);
                return FALSE;
            }
            (*p)++;
        }

        buf->mEnv = TRUE;

        add_char_to_buf(buf, PARSER_MAGIC_NUMBER_ENV);
        char buf2[128];
        snprintf(buf2, 128, "%d", command->mEnvsNum);
        add_str_to_buf(buf, buf2);
        add_char_to_buf(buf, PARSER_MAGIC_NUMBER_ENV);

        sCommand_add_env(command, MANAGED name.mBuf, MANAGED key.mBuf, key.mEnv);
    }

    return TRUE;
}

/// ñɤ߹
static BOOL read_one_argument(char** p, sBuf* buf, char* sname, int* sline, BOOL expand_quote, sCommand* command, sObject* block, sObject** current_object)
{
    /// argument separated character table
    static unsigned char table[] = { 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        1, 0, 0, 1, 0, 0, 1, 0, 1, 1, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0 
    };

    BOOL squote = FALSE;
    BOOL dquote = FALSE;
    while(**p) {
        if(!dquote && **p == '\'') {
            (*p)++;

            if(**p == '\'') {
                (*p)++;
                buf->mNullString = TRUE;
                break;
            }
            else {
                squote = !squote;
            }
        }
        else if(!squote && **p == '"') {
            (*p)++;

            if(**p == '"') {
                (*p)++;
                buf->mNullString = TRUE;
                break;
            }
            else {
                dquote = !dquote;
            }
        }
        /// env ///
        else if(!squote && **p == '$') {
            if(!read_env(p, buf, sname,sline, expand_quote, command, block, current_object)) {
                return FALSE;
            }
        }
        else if(squote || dquote) {
            if(**p == 0) {
                err_msg("close single quote or double quote", sname, *sline, buf->mBuf);
                return FALSE;
            }
            else if(**p == '\n') {
                (*sline)++;
                add_char_to_buf(buf, **p);
                (*p)++;
            }
            else {
                add_char_to_buf(buf, **p);
                (*p)++;
            }
        }
        /// quote ///
        else if(**p == '\\') {
            if(!read_quote(p, buf, sname, sline)) {
                return FALSE;
            }
        }
        /// here document ///
        else if(**p == '<' && *(*p+1) == '<' && *(*p+2) == '<') {
            (*p)+=3;

            skip_spaces(p);

            BOOL squote;
            if(**p == '\'') {
                squote = TRUE;
                (*p)++;
            }
            else {
                squote = FALSE;
            }

            skip_spaces(p);

            sObject* name = STRING_NEW_STACK("");
            while(**p) {
                if(**p >='A' && ** p <= 'Z' || **p >='a' && **p <='z' || **p == '_' || **p >= '0' && **p <= '9')
                {
                    string_push_back2(name, **p);
                    (*p)++;
                }
                else {
                    break;
                }
            }

            skip_spaces(p);

            if(squote) {
                if(**p == '\'') {
                    (*p)++;
                }
                else {
                    err_msg("invalid here document name", sname, *sline, buf->mBuf);
                    return FALSE;
                }
            }

            skip_spaces(p);

            if(**p == '\n') {
                (*sline)++;
                (*p)++;
            }
            else {
                err_msg("invalid here document name", sname, *sline, buf->mBuf);
                return FALSE;
            }

            if(string_c_str(name)[0] == 0) {
                err_msg("invalid here document name", sname, *sline, buf->mBuf);
                return FALSE;
            }

            sObject* line = STRING_NEW_STACK("");
            while(**p) {
                /// env ///
                if(!squote && **p == '$') {
                    if(!read_env(p, buf, sname,sline, expand_quote, command, block, current_object)) {
                        return FALSE;
                    }
                }
                else if(**p == '\n') {
                    add_char_to_buf(buf, **p);
                    (*p)++;
                    (*sline)++;

                    sObject* line = STRING_NEW_STACK("");
                    while(**p) {
                        if(**p >='A' && ** p <= 'Z' || **p >='a' && **p <='z' || **p == '_' || **p >= '0' && **p <= '9')
                        {
                            string_push_back2(line, **p);
                            (*p)++;
                        }
                        else {
                            break;
                        }
                    }

                    if(strcmp(string_c_str(line), string_c_str(name)) == 0) {
                        break;
                    }
                    else {
                        add_str_to_buf(buf, string_c_str(line));
                    }
                }
                else {
                    add_char_to_buf(buf, **p);
                    (*p)++;
                }
            }
        }
        /// option ///
        else if(command->mArgsNum > 0 && buf->mLen == 0 && **p == '-' && (isalpha(*(*p+1)) || *(*p+1) == '-' || *(*p+1) == '_'))
        {
            add_char_to_buf(buf, PARSER_MAGIC_NUMBER_OPTION);
            (*p)++;
        }
        /// tilda ///
        else if(buf->mLen == 0 && **p == '~') {
            SBLOCK(block).mCompletionFlags |= COMPLETION_FLAGS_TILDA;

            add_char_to_buf(buf, **p);
            (*p)++;

            buf->mTilda = TRUE;
        }
        /// glob ///
        else if(command->mArgsNum > 0 && **p == '[' || **p == '?' || **p == '*') {
            add_char_to_buf(buf, **p);
            (*p)++;

            buf->mGlob = TRUE;
        }
        /// message passing ///
        else if(command->mArgsNum == 0 && **p == '-' && *(*p+1) == '>') {
            (*p)+=2;
            buf->mMessagePassing = TRUE;
            SBLOCK(block).mCompletionFlags &= ~(COMPLETION_FLAGS_NEXT_STATMENT|COMPLETION_FLAGS_NEXT_PIPE);
            break;
        }
        /// utf-8 character
        else if(((unsigned char)**p) > 127) {
            add_char_to_buf(buf, **p);
            (*p)++;
        }
        else if(table[**p]) {
            break;
        }
        else {
            add_char_to_buf(buf, **p);
            (*p)++;
        }
    }

    if(squote || dquote) {
        err_msg("require to close ' or \"", sname, *sline, buf->mBuf);
        return FALSE;
    }

    return TRUE;
}

static void tilda_expasion(char* file, ALLOC char** file2)
{
    char* p = file;
    p++; // ~

    char result[PATH_MAX];
    
    char* user = MALLOC(16);
    int user_size = 16;
    int user_len = 0;
    user[0] = 0;

    /// get user name ///
    while(*p && *p != '/') {
        if(user_len >= user_size) {
            user_size *= 1.8;
            user = REALLOC(user, user_size);
        }
        user[user_len++] = *p++;
    }
    user[user_len] = 0;

    /// expand HOME evironment
    if(user[0] == 0) {
        FREE(user);

        char* home = getenv("HOME");
        if(home) {
            xstrncpy(result, home, PATH_MAX);
        }
        else {
            struct passwd* pw = getpwuid(getuid());
            if(pw) {
                xstrncpy(result, pw->pw_dir, PATH_MAX);
            }
            else {
                xstrncpy(result, "~", PATH_MAX);
            }
        }
    }
    /// expand user name to home directory
    else {
        struct passwd* pw = getpwnam(user);
        FREE(user);

        if(pw) {
            xstrncpy(result, pw->pw_dir, PATH_MAX);
        }
        else {
            xstrncpy(result, "~", PATH_MAX);
            xstrncat(result, user, PATH_MAX);
        }
    }

    xstrncat(result, p, PATH_MAX);

    *file2 = STRDUP(result);
}

static BOOL glob_expasion(sBuf* buf, sCommand* command, char* sname, int* sline)
{
    glob_t result;
    int rc;
    if(buf->mTilda) {  // tilda expasion
        char* buf2;
        tilda_expasion(buf->mBuf, ALLOC &buf2);

        rc = glob(buf2, 0, NULL, &result);
        FREE(buf2);
    }
    else {
        rc = glob(buf->mBuf, 0, NULL, &result);
    }

    if(rc == GLOB_NOSPACE) {
        err_msg("read_one_argument: out of space during glob operation", sname, *sline, buf->mBuf);
        return FALSE;
    }
    else if(rc == GLOB_NOMATCH) {
        sCommand_add_arg_to_command(command, MANAGED STRDUP(buf->mBuf), 0);
    }
    else {
        int i;
        for(i=0; i<result.gl_pathc; i++) {
            sCommand_add_arg_to_command(command, MANAGED STRDUP(result.gl_pathv[i]), 0);
        }
    }

    return TRUE;
}

static BOOL read_statment(char**p, sStatment* statment, sObject* block, char* sname, int* sline, sObject** current_object);

// result --> TRUE: success
// result --> FALSE: There is a error message on gErrMsg. Please output the error message and stop running
static BOOL block_parse(char** p, char* sname, int* sline, sObject* block, sObject** current_object)
{
    char* source_head = *p;

    while(**p != ')') {
        /// skip spaces at head of statment ///
        while(**p == ' ' || **p == '\t' || **p == '\n' && (*sline)++) { (*p)++; }

        /// skip a comment ///
        if(**p == '#') skip_to_nextline(p, sline);

        /// go parsing ///
        sStatment* statment = sStatment_new(block, *sline, sname);

        if(!read_statment(p, statment, block, sname, sline, current_object))
        {
            return FALSE;
        }

        /// skip comment ///
        if(**p == '#') skip_to_nextline(p, sline);

        /// there is no command, delete last statment
        if(statment->mCommandsNum == 0) {
            sStatment_delete(statment);
            memset(statment, 0, sizeof(sStatment));
            SBLOCK(block).mStatmentsNum--;
        }

        if(**p == 0) {
            err_msg("close block. require ) character before the end of statments", sname, *sline, NULL);
            return FALSE;
        }
    }
    (*p)++;

    const int len = *p - source_head;
    char* source = MALLOC(len);
    memcpy(source, source_head, len-1);
    source[len-1] = 0;

    SBLOCK(block).mSource = source;

    return TRUE;
}

static BOOL read_command(char** p, sCommand* command, sStatment* statment, sObject* block, char* sname, int* sline, sObject** current_object)
{
    while(**p) {
        SBLOCK(block).mCompletionFlags &= ~COMPLETION_FLAGS_WORDBREAK_CHARACTER;

        sBuf buf;
        memset(&buf, 0, sizeof(sBuf));
        buf.mBuf = MALLOC(64);
        buf.mBuf[0] = 0;
        buf.mSize = 64;

        if(!read_one_argument(p, &buf, sname, sline, TRUE, command, block, current_object)) 
        {
            if(buf.mEnv) {  // for readline
                sCommand_add_arg_to_command(command, MANAGED buf.mBuf, TRUE);
            }
            else {
                FREE(buf.mBuf);
            }
            return FALSE;
        }

        /// env ///
        if(buf.mEnv) {
            sCommand_add_arg_to_command(command, MANAGED buf.mBuf, TRUE);
        }
        /// message passing ///
        else if(buf.mMessagePassing) {
            if(!sCommand_add_message(command, MANAGED buf.mBuf)) {
                err_msg("too many message", sname, *sline, buf.mBuf);
                FREE(buf.mBuf);
                return FALSE;
            }

            continue;
        }
        /// expand glob ///
        else if(buf.mGlob && command->mArgsNum > 0) {
            if(!glob_expasion(&buf, command, sname, sline)) {
                FREE(buf.mBuf);
                return FALSE;
            }
            FREE(buf.mBuf);
        }
        /// tilda expression ///
        else if(buf.mTilda) {
            char* buf2;
            tilda_expasion(buf.mBuf, ALLOC &buf2);
            sCommand_add_arg_to_command(command, MANAGED buf2, 0);
            FREE(buf.mBuf);
        }
        /// add buf to command ///
        else if(buf.mNullString || buf.mBuf[0] != 0) {
            sCommand_add_arg_to_command(command, MANAGED buf.mBuf, buf.mEnv);
        }
        else {
            FREE(buf.mBuf);
        }

        if(**p != 0) {
            SBLOCK(block).mCompletionFlags &= ~COMPLETION_FLAGS_ENV;
            SBLOCK(block).mCompletionFlags &= ~COMPLETION_FLAGS_TILDA;
        }

        /// subshell, block ///
        if(**p == '(') {
            (*p)++;

            if(command->mArgsNum == 0) {
                sObject* block2 = BLOCK_NEW_STACK();
                if(!block_parse(p, sname, sline, block2, current_object)) {
                    sCommand_add_block_to_command(command, block2);

                    SBLOCK(block).mCompletionFlags |= COMPLETION_FLAGS_BLOCK;
                    SBLOCK(block).mCompletionFlags |= command->mBlocksNum & 0xFF;
                    return FALSE;
                }

                sCommand_add_arg_to_command(command, MANAGED STRDUP("subshell"), 0);
                sCommand_add_block_to_command(command, block2);
            }
            else {
                sObject* current_object_before;

                /// if user call run method, change current_object
                if(current_object) {
                    current_object_before = *current_object;

                    if(strcmp(command->mArgs[0], "run") == 0 && command->mMessagesNum > 0) {
                        sObject* current_object2 = *current_object;
                        sObject* object = access_object(command->mMessages[0], &current_object2, NULL);

                        if(object && TYPE(object) == T_UOBJECT) {
                            int i;
                            for(i=1; i<command->mMessagesNum; i++) {
                                object = hash_item(object, command->mMessages[i]);

                                if(object == NULL || TYPE(object) != T_UOBJECT) {
                                    break;
                                }
                            }

                            if(object && TYPE(object) == T_UOBJECT) {
                                *current_object = object;
                            }
                        }
                    }
                }

                sObject* block2 = BLOCK_NEW_STACK();
                if(!block_parse(p, sname, sline, block2, current_object)) {
                    sCommand_add_block_to_command(command, block2);

                    SBLOCK(block).mCompletionFlags |= COMPLETION_FLAGS_BLOCK;
                    SBLOCK(block).mCompletionFlags |= command->mBlocksNum & 0xFF;
                    return FALSE;
                }

                /// if user call run method, change current_object
                if(current_object) {
                    *current_object = current_object_before;
                }

                sCommand_add_block_to_command(command, block2);
            }
        }
        /// spaces ///
        else if(**p == ' ' || **p == '\t') {
            SBLOCK(block).mCompletionFlags |= COMPLETION_FLAGS_WORDBREAK_CHARACTER;
            skip_spaces(p);
        }
        else {
            break;
        }
    }

    return TRUE;
}

/// read statment
/// TRUE: OK
/// FALSE: indicate to occure error
static BOOL read_statment(char**p, sStatment* statment, sObject* block, char* sname, int* sline, sObject** current_object)
{
    /// a reversing code ///
    if(**p == '!') {
        (*p)++;
        skip_spaces(p);

        statment->mFlags |= STATMENT_REVERSE;
    }

    /// a global pipe ///
    if(**p == '|' && *(*p+1) == '>') {
        (*p)+=2;
        skip_spaces(p);

        statment->mFlags |= STATMENT_GLOBALPIPEIN;
    }
    /// a context pipe ///
    else if(**p == '|' && *(*p+1) != '|') {
        (*p)++;
        skip_spaces(p);

        statment->mFlags |= STATMENT_CONTEXTPIPE;
    }

    while(**p) {
        if(statment->mCommandsNum >= STATMENT_COMMANDS_MAX) {
            err_msg("There are max number of commands on this statment", sname, *sline, NULL);
            return FALSE;
        }

        SBLOCK(block).mCompletionFlags &= ~COMPLETION_FLAGS_NEXT_PIPE;

        sCommand* command = sCommand_new(statment);
        if(!read_command(p, command, statment, block, sname, sline, current_object)) {
            return FALSE;
        }

        /// There is no argument, so delete last command in block ///
        if(command->mArgsNum == 0)
        {
            if(command->mMessagesNum > 0) {
                err_msg("require command name", sname, *sline, NULL);
                return FALSE;
            }
            else {
                sCommand_delete(command);
                memset(command, 0, sizeof(sCommand));
                statment->mCommandsNum--;
            }
        }

        if(**p == '|' && *(*p+1) != '>' && *(*p+1) != '|') // chains a next command with pipe
        {
            (*p)++;

            SBLOCK(block).mCompletionFlags |= COMPLETION_FLAGS_NEXT_PIPE;
            skip_spaces(p);
        }
        else {
            break;
        }
    }

    /// global pipe
    if(**p == '|' && *(*p+1) == '>') {
        (*p)+=2;
        skip_spaces(p);

        statment->mFlags |= STATMENT_GLOBALPIPEOUT;
    }

    /// the termination of a statment
    if(**p == '&' && *(*p+1) == '&') {
        (*p)+=2;
        skip_spaces(p);

        statment->mFlags |= STATMENT_ANDAND;

        SBLOCK(block).mCompletionFlags |= COMPLETION_FLAGS_NEXT_STATMENT;
    }
    else if(**p == '|' && *(*p+1) == '|') {
        (*p)+=2;
        skip_spaces(p);

        statment->mFlags |= STATMENT_OROR;

        SBLOCK(block).mCompletionFlags |= COMPLETION_FLAGS_NEXT_STATMENT;
    }
    else if(**p == '&') {
        (*p)++;
        skip_spaces(p);

        statment->mFlags |= STATMENT_BACKGROUND;

        SBLOCK(block).mCompletionFlags |= COMPLETION_FLAGS_NEXT_STATMENT;
    }
    else if(**p == '\n') {
        (*p)++; 
        skip_spaces(p);

        statment->mFlags |= STATMENT_NORMAL;
        (*sline)++;

        SBLOCK(block).mCompletionFlags |= COMPLETION_FLAGS_NEXT_STATMENT;
    }
    else if(**p == ';') {
        (*p)++;
        skip_spaces(p);

        statment->mFlags |= STATMENT_NORMAL;

        SBLOCK(block).mCompletionFlags |= COMPLETION_FLAGS_NEXT_STATMENT;
    }
    else if(**p == 0) {
        statment->mFlags |= STATMENT_NORMAL;
    }
    else if(**p == ')' || **p == '#') {
    }
    else {
        char buf[128];
        snprintf(buf, 128, "unexpected token3 -->(%c)\n", **p);
        err_msg(buf, sname, *sline, NULL);
        return FALSE;
    }

    return TRUE;
}

// result --> TRUE: success
// result --> FALSE: There is a error message on gErrMsg. Please output the error message and stop running
BOOL parse(char* p, char* sname, int* sline, sObject* block, sObject** current_object)
{
    SBLOCK(block).mSource = STRDUP(p);

    while(1) {
        /// skip spaces at head of statment ///
        while(*p == ' ' || *p == '\t' || *p == '\n' && (*sline)++) { p++; }

        /// skip comment ///
        if(*p == '#') skip_to_nextline(&p, sline);

        /// go parsing ///
        SBLOCK(block).mCompletionFlags &= ~(COMPLETION_FLAGS_NEXT_STATMENT);

        sStatment* statment = sStatment_new(block, *sline, sname);

        if(!read_statment(&p, statment, block, sname, sline, current_object))
        {
            return FALSE;
        }

        /// There is not a command, so delete last statment in block
        if(statment->mCommandsNum == 0) {
            sStatment_delete(statment);
            memset(statment, 0, sizeof(sStatment));
            SBLOCK(block).mStatmentsNum--;
        }

        if(*p == ')') {
            char buf[128];
            snprintf(buf, 128, "unexpected token3 -->(%c)\n", *p);
            err_msg(buf, sname, *sline, NULL);
            return FALSE;
        }
        /// skip comment ///
        else if(*p == '#') {
            skip_to_nextline(&p, sline);
        }
        else if(*p == 0) {
            break;
        }
    }

    return TRUE;
}
