package org.postgresforest.tool.tests;

import java.io.*;
import java.sql.*;
import junit.framework.*;

import org.postgresforest.tool.Logger;
import org.postgresforest.tool.lib.Config;
import org.postgresforest.tool.lib.Instance;
import org.postgresforest.tool.lib.GSCdata;
import org.postgresforest.tool.lib.ForestToolException;

/*
 * 試験内容
 *  - GSC作成（初期化）
 *  - GSC接続
 *  - Config作成
 *  - Config名一覧取得
 *  - Config設定・取得
 *  - Config削除
 */
public class TestGSCdata extends ForestToolTestCase {

	public void setUp()
	{
	}

	public void tearDown()
	{
		//		dropdb("gsctest", "9998");
	}

	/* ------------------------------------------------
	 * GSCを作成
	 * ------------------------------------------------
	 */
	public void testInitialize001()
	{
		GSCdata gsc = null;

		try {
			dropdb("gsctest", "9998");

			gsc = GSCdata.initialize("localhost", "9998", "gsctest",
									 user, pass);
			assertTrue(gsc!=null);
			gsc.close();

			dropdb("gsctest", "9998");
		}
		catch (ForestToolException e)
		{
			System.out.println(e.getMessage());
			fail();
		}
	}

	/*
	 * GSCを作成できない（ホスト名違い）
	 */
	public void testInitialize002()
	{
		GSCdata gsc = null;

		try {
			gsc = GSCdata.initialize("hoge", "9998", "gsctest",
									 user, pass);
			gsc.close();
		}
		catch (ForestToolException e)
		{
			System.out.println(e.getMessage());
			assertTrue(true);
			return;
		}
		fail();
	}

	/*
	 * GSCを作成できない（ポート番号違い）
	 */
	public void testInitialize003()
	{
		GSCdata gsc = null;

		try {
			gsc = GSCdata.initialize("localhost", "pppp", "gsctest",
									 user, pass);
			gsc.close();
		}
		catch (ForestToolException e)
		{
			System.out.println(e.getMessage());
			assertTrue(true);
			return;
		}
		fail();
	}

	/*
	 * GSCを作成できない（DB名違い）
	 */
	public void testInitialize004()
	{
		GSCdata gsc = null;

		try {
			gsc = GSCdata.initialize("localhost", "9998", "",
									 user, pass);
			gsc.close();
		}
		catch (ForestToolException e)
		{
			System.out.println(e.getMessage());
			assertTrue(true);
			return;
		}
		fail();
	}

	/*
	 * GSCを作成できない（ユーザ名違い）
	 */
	public void testInitialize005()
	{
		GSCdata gsc = null;

		try {
			gsc = GSCdata.initialize("localhost", "9998", "gsctest",
									 "nosuchuser", pass);
			gsc.close();
		}
		catch (ForestToolException e)
		{
			System.out.println(e.getMessage());
			assertTrue(true);
			return;
		}
		fail();
	}

	/*
	 * FIXME: GSCを作成できない（パスワード違い）
	 */
	public void _testInitialize006()
	{
		GSCdata gsc = null;

		try {
			gsc = GSCdata.initialize("localhost", "9998", "gsctest",
									 user, "nosuchpass");
			gsc.close();
		}
		catch (ForestToolException e)
		{
			System.out.println(e.getMessage());
			assertTrue(true);
			return;
		}
		fail();
	}

	/*
	 * GSCを作成、接続
	 */
	public void testNew001()
	{
		GSCdata gsc = null;

		try {
			gsc = GSCdata.initialize("localhost", "9998", "gsctest",
									 user, pass);
			gsc.close();
		}
		catch (ForestToolException e)
		{
			System.out.println(e.getMessage());
			fail();
		}

		try {
			gsc = new GSCdata("localhost", "9998", "gsctest",
							  user, pass);
			assertTrue(gsc!=null);
			gsc.close();
		}
		catch (ForestToolException e)
		{
			System.out.println(e.getMessage());
			fail();
		}
	}

	/*
	 * GSCに接続（ホスト名間違い）
	 */
	public void testNew002()
	{
		GSCdata gsc = null;

		try {
			gsc = new GSCdata("localhsot", "9998", "gsctest",
							  user, pass);
			fail();
		}
		catch (ForestToolException e)
		{
			System.out.println(e.getMessage());
			assertTrue(true);
		}
	}

	/*
	 * GSCに接続（ポート番号間違い）
	 */
	public void testNew003()
	{
		GSCdata gsc = null;

		try {
			gsc = new GSCdata("localhost", "19998", "gsctest",
							  user, pass);
			fail();
		}
		catch (ForestToolException e)
		{
			System.out.println(e.getMessage());
			assertTrue(true);
		}
	}

	/*
	 * GSCに接続（DB名間違い）
	 */
	public void testNew004()
	{
		GSCdata gsc = null;

		try {
			gsc = new GSCdata("localhost", "19998", "gsctest_",
							  user, pass);
			fail();
		}
		catch (ForestToolException e)
		{
			System.out.println(e.getMessage());
			assertTrue(true);
		}
	}

	/*
	 * GSCに接続（ユーザ名間違い）
	 */
	public void testNew005()
	{
		GSCdata gsc = null;

		try {
			gsc = new GSCdata("localhost", "19998", "gsctest",
							  "nosuchuser", pass);
			fail();
		}
		catch (ForestToolException e)
		{
			System.out.println(e.getMessage());
			assertTrue(true);
		}
	}

	/*
	 * FIXME: GSCに接続（パスワード間違い）
	 */
	public void _testNew006()
	{
		GSCdata gsc = null;

		try {
			gsc = new GSCdata("localhost", "19998", "gsctest",
							  "nosuchuser", pass);
			fail();
		}
		catch (ForestToolException e)
		{
			System.out.println(e.getMessage());
			assertTrue(true);
		}
	}

	/*
	 * GSCに接続（nullチェック）
	 */
	public void testNew007()
	{
		GSCdata gsc = null;

		try {
			gsc = new GSCdata(null, "9998", "gsctest",
							  user, pass);
			fail();
		}
		catch (ForestToolException e)
		{
			System.out.println(e.getMessage());
			assertTrue(true);
		}

		try {
			gsc = new GSCdata("localhost", null, "gsctest",
							  user, pass);
			fail();
		}
		catch (ForestToolException e)
		{
			System.out.println(e.getMessage());
			assertTrue(true);
		}

		try {
			gsc = new GSCdata("localhost", "9998", null,
							  user, pass);
			fail();
		}
		catch (ForestToolException e)
		{
			System.out.println(e.getMessage());
			assertTrue(true);
		}

		try {
			gsc = new GSCdata("localhost", "9998", "gsctest",
							  null, pass);
			fail();
		}
		catch (ForestToolException e)
		{
			System.out.println(e.getMessage());
			assertTrue(true);
		}

		/* FIXME: パスワードがnullでも通る */
		/*
		try {
			gsc = new GSCdata("localhost", "9998", "gsctest",
							  user, null);
			fail();
		}
		catch (ForestToolException e)
		{
			System.out.println(e.getMessage());
			assertTrue(true);
		}
		*/
	}

	/* -------------------------------------------------------
	 * Config作成
	 * ------------------------------------------------------
	 */
	/*
	 * DEFAULT_CONFIGをテンプレにConfigを作成（暗黙に）
	 */
    public void testCreateConfig001()
    {
        GSCdata gsc = null;

        try {
            gsc = new GSCdata("localhost", "9998", "gsctest",
                              user, pass);
            assertTrue(gsc!=null);

            Config config = gsc.createConfig("TEST_CONFIG_001");
            assertTrue(config!=null);

            gsc.close();
        }
        catch (ForestToolException e)
        {
            System.out.println(e.getMessage());
            fail();
        }
    }

	/*
	 * DEFAULT_CONFIGをテンプレにConfigを作成（明示的に）
	 */
    public void testCreateConfig002()
    {
        GSCdata gsc = null;

        try {
            gsc = new GSCdata("localhost", "9998", "gsctest",
                              user, pass);
            assertTrue(gsc!=null);

			Config config = gsc.createConfig("TEST_CONFIG_002", Config.DEFAULT_CONFIG_NAME);
            assertTrue(config!=null);

            gsc.close();
        }
        catch (ForestToolException e)
        {
            System.out.println(e.getMessage());
            fail();
        }
    }

	/*
	 * DEFAULT_CONFIGをテンプレにConfigを作成（明示的に）
	 */
    public void testCreateConfig003()
    {
        GSCdata gsc = null;

        try {
            gsc = new GSCdata("localhost", "9998", "gsctest",
                              user, pass);
            assertTrue(gsc!=null);

			Config template = gsc.getConfig(Config.DEFAULT_CONFIG_NAME);

            Config config = gsc.createConfig("TEST_CONFIG_003", template);
            assertTrue(config!=null);

            gsc.close();
        }
        catch (ForestToolException e)
        {
            System.out.println(e.getMessage());
            fail();
        }
    }

	/*
	 * CONFIG名一覧を取得
	 */
	public void testGetConfigNames001()
	{
        GSCdata gsc = null; 

        try {
            gsc = new GSCdata("localhost", "9998", "gsctest",
                              user, pass);
            assertTrue(gsc!=null);

			String[] s = gsc.getConfigNames();
			assertTrue(s.length == 4);
			assertTrue( s[0].equals("FOREST_DEFAULT_CONFIG") );
			assertTrue( s[1].equals("TEST_CONFIG_001") );
			assertTrue( s[2].equals("TEST_CONFIG_002") );
			assertTrue( s[3].equals("TEST_CONFIG_003") );

            gsc.close();
        }
        catch (ForestToolException e)
        {
            System.out.println(e.getMessage());
            fail();
        }
	}

	/*
	 * CONFIGを取得
	 */
	public void testGetConfig001()
	{
        GSCdata gsc = null; 

        try {
            gsc = new GSCdata("localhost", "9998", "gsctest",
                              user, pass);
            assertTrue(gsc!=null);

			Config c = gsc.getConfig("TEST_CONFIG_001");
			assertTrue(c!=null);

            gsc.close();
        }
        catch (ForestToolException e)
        {
            System.out.println(e.getMessage());
            fail();
        }
	}

	/*
	 * CONFIGを取得（存在しないCONFIG名）
	 */
	public void testGetConfig002()
	{
        GSCdata gsc = null; 

        try {
            gsc = new GSCdata("localhost", "9998", "gsctest",
                              user, pass);
            assertTrue(gsc!=null);

			Config c = gsc.getConfig("TEST_CONFIG_xxx");
			assertTrue(c==null);

            gsc.close();
        }
        catch (ForestToolException e)
        {
            System.out.println(e.getMessage());
            fail();
        }
	}

	/*
	 * CONFIGを取得（null値チェック）
	 */
	public void testGetConfig003()
	{
        GSCdata gsc = null; 

        try {
            gsc = new GSCdata("localhost", "9998", "gsctest",
                              user, pass);
            assertTrue(gsc!=null);

			Config c = gsc.getConfig(null);
			assertTrue(c==null);

            gsc.close();
        }
        catch (ForestToolException e)
        {
            System.out.println(e.getMessage());
            fail();
        }
	}

	/*
	 * CONFIGの値を設定
	 */
    public void testSetAndGet001()
    {
        GSCdata gsc = null;

        try {
            gsc = new GSCdata("localhost", "9998", "gsctest",
                              user, pass);
            assertTrue(gsc!=null);

			Config config = gsc.createConfig("TEST_CONFIG_101");
			assertTrue( config!=null );

            gsc.close();
        }
        catch (ForestToolException e)
        {
            System.out.println(e.getMessage());
            fail();
        }

        try {
            gsc = new GSCdata("localhost", "9998", "gsctest",
                              user, pass);
            assertTrue(gsc!=null);

			Config config = gsc.getConfig("TEST_CONFIG_101");
			assertTrue( config!=null );

			assertTrue( config.setDescription("description test") );
			assertTrue( config.setCacheRefresh(8) );
			assertTrue( config.setRetryCount(16) );
			assertTrue( config.setDefectTimeout(32) );
			assertTrue( config.setDistributedConnection(0) );
			assertTrue( config.setPartitionMode(0) );
			assertTrue( config.setSynchronizeMode(0) );

            gsc.close();
        }
        catch (ForestToolException e)
        {
            System.out.println(e.getMessage());
            fail();
        }

        try {
            gsc = new GSCdata("localhost", "9998", "gsctest",
                              user, pass);
            assertTrue(gsc!=null);

			Config config = gsc.getConfig("TEST_CONFIG_101");
			assertTrue( config!=null );

			assertTrue( config.getDescription().equals("description test") );
			assertTrue( config.getCacheRefresh()==8 );
			assertTrue( config.getRetryCount()==16 );
			assertTrue( config.getDefectTimeout()==32 );
			assertTrue( config.getDistributedConnection()==0 );
			assertTrue( config.getPartitionMode()==0 );
			assertTrue( config.getSynchronizeMode()==0 );

            gsc.close();
        }
        catch (ForestToolException e)
        {
            System.out.println(e.getMessage());
            fail();
        }

        try {
            gsc = new GSCdata("localhost", "9998", "gsctest",
                              user, pass);
            assertTrue(gsc!=null);

			Config config = gsc.getConfig("TEST_CONFIG_101");
			assertTrue( config!=null );

			assertTrue( config.setDescription("description test 2") );
			config.setCacheRefresh(24);
			config.setRetryCount(48);
			config.setDefectTimeout(96);
			config.setDistributedConnection(1);
			config.setPartitionMode(1);
			config.setSynchronizeMode(1);

            gsc.close();
        }
        catch (ForestToolException e)
        {
            System.out.println(e.getMessage());
            fail();
        }

        try {
            gsc = new GSCdata("localhost", "9998", "gsctest",
                              user, pass);
            assertTrue(gsc!=null);

			Config config = gsc.getConfig("TEST_CONFIG_101");
			assertTrue( config!=null );

			assertTrue( config.getDescription().equals("description test 2") );
			assertTrue( config.getCacheRefresh()==24 );
			assertTrue( config.getRetryCount()==48 );
			assertTrue( config.getDefectTimeout()==96 );
			assertTrue( config.getDistributedConnection()==1 );
			assertTrue( config.getPartitionMode()==1 );
			assertTrue( config.getSynchronizeMode()==1 );

            gsc.close();
        }
        catch (ForestToolException e)
        {
            System.out.println(e.getMessage());
            fail();
        }
    }

	/*
	 * 指定したCONFIGを削除
	 */
    public void testDropConfig001()
    {
        GSCdata gsc = null;

        try {
            gsc = new GSCdata("localhost", "9998", "gsctest",
                              user, pass);
            assertTrue(gsc!=null);

			boolean rc = gsc.dropConfig("TEST_CONFIG_003");
			assertTrue( rc==true );

            gsc.close();
        }
        catch (ForestToolException e)
        {
            Logger.error(e.getMessage());
            Logger.trace(e);
            fail();
        }
    }


	/*
	 * 指定したCONFIGを削除（Configオブジェクト指定）
	 */
    public void testDropConfig002()
    {
        GSCdata gsc = null;

        try {
            gsc = new GSCdata("localhost", "9998", "gsctest",
                              user, pass);
            assertTrue(gsc!=null);

			Config config = gsc.getConfig("TEST_CONFIG_002");

			boolean rc = gsc.dropConfig(config);
			assertTrue( rc==true );

            gsc.close();
        }
        catch (ForestToolException e)
        {
            System.out.println(e.getMessage());
            fail();
        }
    }

	/*
	 * 存在しないCONFIGを削除
	 */
    public void testDropConfig003()
    {
        GSCdata gsc = null;

        try {
            gsc = new GSCdata("localhost", "9998", "gsctest",
                              user, pass);
            assertTrue(gsc!=null);

			boolean rc = gsc.dropConfig("TEST_CONFIG_xxx");
			assertTrue( rc!=true );

            gsc.close();
        }
        catch (ForestToolException e)
        {
            System.out.println(e.getMessage());
            fail();
        }
    }

	/*
	 * 存在しないCONFIGを削除（null値チェック）
	 */
    public void testDropConfig004()
    {
        GSCdata gsc = null;

        try {
            gsc = new GSCdata("localhost", "9998", "gsctest",
                              user, pass);
            assertTrue(gsc!=null);

			boolean rc = gsc.dropConfig((Config)null);
			assertTrue( rc!=true );

            gsc.close();
        }
        catch (ForestToolException e)
        {
            System.out.println(e.getMessage());
            fail();
        }
    }

	/*
	 * 存在しないCONFIGを削除（null値チェック）
	 */
    public void testDropConfig005()
    {
        GSCdata gsc = null;

        try {
            gsc = new GSCdata("localhost", "9998", "gsctest",
                              user, pass);
            assertTrue(gsc!=null);

			boolean rc = gsc.dropConfig((String)null);
			assertTrue( rc!=true );

            gsc.close();
        }
        catch (ForestToolException e)
        {
            System.out.println(e.getMessage());
            fail();
        }
    }

	public static void main(String[] args)
	{
		Logger.setLevel(Logger.LOG_TRACE);

		junit.textui.TestRunner.run(TestGSCdata.class);
	}
}
