package org.postgresforest.tool.cli.action;

import java.util.ArrayList;
import java.util.Iterator;

import org.postgresforest.tool.Logger;
import org.postgresforest.tool.lib.Database;
import org.postgresforest.tool.lib.ForestToolException;
import org.postgresforest.tool.lib.GSCdata;
import org.postgresforest.tool.lib.GT;
import org.postgresforest.tool.lib.Instance;
import org.postgresforest.tool.lib.Table;
import org.postgresforest.tool.util.CommandParser;
import org.postgresforest.tool.util.MessagesCommandLine;
import org.postgresforest.tool.util.Misc;
import org.postgresforest.tool.util.Parser;
import org.postgresforest.tool.util.Table2StringUtil;

/**
 * データベースへのコマンドラインツール用のアクションを提供するクラス
 */
public class DbActions {

	static private final String[] SHOW_DB_HEADER = { "DBNAME", "SERVERID", "HOST", "PORT"}; 

	
	/**
	 * DBの作成
	 * @param cp コマンド解析結果
	 * @param gsc GSCヘの接続情報
	 */	
	public static void create(CommandParser cp, GSCdata gsc) {
		//DB名取得
		String dbName = cp.getDB();
		if(dbName == null || dbName.equals("")){
			Logger.error( GT.tr("データベース名が入力されていません。") );
			return;
		}

		//オプション取得
		String option = ""; //$NON-NLS-1$
		Iterator it = cp.getOption().iterator();
		while(it.hasNext()){
			option += (String)it.next();
		}
		
		
		ArrayList chkServerList = cp.getServer();
		// 表示用のサーバリストの作成
		String serverListStr = "";
		int ids[] = new int[chkServerList.size()];
		for (int i=0; i<ids.length;i++) {
			String idStr = (String)chkServerList.get(i);
			ids[i] =  Integer.parseInt(idStr);
			serverListStr += idStr + ",";
		}


		//DB作成処理
		try {
			Database db;

			if ( gsc.getDatabase(dbName)!=null )
			{
				if ( cp.isForce() )
				{
					gsc.dropDatabase(dbName);
				}
				else
				{
					Logger.error( GT.tr("データベース \"{0}\" は既に存在しています。",
										dbName) );
					return;
				}
			}

			if (option.equals("")) {
				db = gsc.createDatabase(ids,dbName);
			} else {
				db = gsc.createDatabase(ids,dbName,option);
			}
			if(db == null){
				Logger.error(MessagesCommandLine.getString("cui.error.db.create",
														   new Object[] {dbName, serverListStr}));
				return;
			}
			
		} catch (ForestToolException e) {
			Logger.error( GT.tr("データベース \"{0}\" をサーバID \"{1}\" 上に作成できませんでした。",
								dbName, serverListStr) );
			return;
		}

		Logger.notice( GT.tr("データベース \"{0}\" をサーバID \"{1}\" 上に作成しました。",
							 dbName, serverListStr) );
	}
	
	/**
	 * DBの削除
	 * @param cp コマンド解析結果
	 * @param gsc GSCヘの接続情報
	 */	
	public static void drop(CommandParser cp, GSCdata gsc) {

		//DB名取得
		String dbName = cp.getDB();
		if(dbName == null || dbName.equals("")){
			Logger.error( GT.tr("データベース名が入力されていません。") );
			return;
		}
		
		try {
			if ( gsc.getDatabase(dbName)==null )
			{
				Logger.error( GT.tr("データベース \"{0}\" が見つかりません。",
									dbName) );
				return;
			}
		}
		catch (ForestToolException e)
		{
			Logger.error( GT.tr("データベース \"{0}\" の情報を取得できません。",
								dbName) );

			Logger.detail( e.getDetail().getMessage() );
			Logger.trace( e.getDetail() );
		}

		//コマンドラインから強制フラグを取得
		boolean force = cp.isForce();

		//データベース削除
		if( !gsc.dropDatabase(dbName, force) )
		{
			Logger.error( GT.tr("データベース \"{0}\" の削除に失敗しました。",
								dbName) );
			return;
		}

		Logger.notice( GT.tr("データベース \"{0}\" を削除しました。",
							 dbName) );
	}

	/**
	 * DBのリストア
	 * @param cp コマンド解析結果
	 * @param gsc GSCヘの接続情報
	 */	
	public static void restore(CommandParser cp, GSCdata gsc) {
		Logger.warning( GT.tr("restore dbコマンドは現バージョンではサポートされていません。") );

		/**
		//DB名取得
		String dbName = cp.getDB();
		if(dbName == null || dbName.equals("")){
			//引数が不正
			// FIXME:
			Logger.error("invalid argument.");
			return;
		}

		//オプション取得
		String option = ""; //$NON-NLS-1$
		Iterator it = cp.getOption().iterator();
		while(it.hasNext()){
			option += (String)it.next();
		}

		//ダンプファイル名取得
		ArrayList dumpFileList = cp.getDumpFile();
		
		ArrayList chkServerList = cp.getServer();
		// 表示用のサーバリストの作成
		String serverListStr = "";
		int ids[] = new int[chkServerList.size()];
		for (int i=0; i<ids.length;i++) {
			String idStr = (String)chkServerList.get(i);
			ids[i] =  Integer.parseInt(idStr);
			serverListStr += idStr + ",";
		}
		

		// DB選択とダンプファイルの整合性チェック
		// FIXME:
		//validateDumpFile(addDbInfo);
		
		//DB作成処理
		try {
			Database db = null;
			if(option.equals("")){
				db = gsc.createDatabase(ids,dbName);
			}else{
				db = gsc.createDatabase(ids,dbName,option);
			}
			if(db == null){
				//@@指定されたDBはすでに存在している
			}
			
		} catch (Exception e) {
			Logger.error(MessagesCommandLine.getString("cui.error.db.restore",
			new Object[] {dbName, serverListStr}) );
			Logger.error(e.getMessage());
			Logger.trace(e);
			return;
		}

		Logger.println(MessagesCommandLine.getString("cui.message.db.restore", new Object[] {dbName, serverListStr}) );
		**/
		
	}
	
	/**
	 * DBの表示
	 * @param gsc GSCヘの接続情報
	 */	
	public static void show(GSCdata gsc) {

		String[] dbNames;

		//データベース名一覧を取得
		try {
			dbNames = gsc.getDatabaseNames();
		} catch (ForestToolException e) {
			Logger.error( GT.tr("データベース名一覧を取得できません。") );

			Logger.error(e.getDetail().getMessage());
			Logger.trace(e.getDetail());
			return;
		}

		//DB情報表示
		Logger.println(MessagesCommandLine.getString("cui.message.db.show")) ; //$NON-NLS-1$
		Table2StringUtil table2StringUtil = new Table2StringUtil( SHOW_DB_HEADER );

		//IDからインスタンス情報取得し表示
		
		for (int i = 0; i < dbNames.length; i++) {
			Database d = null;

			try {
				d = gsc.getDatabase(dbNames[i]);
			}
			catch (Exception e)
			{
				Logger.error(e.getMessage());
				Logger.trace(e);
				return;
			}

			int[] ids = d.getInstanceIds();

			for (int j=0 ; j<ids.length ; j++)
			{
				Instance instance = gsc.getInstance(ids[j]);

				if (instance == null)
				{
					//インスタンス情報の取得失敗
					Logger.error(MessagesCommandLine.getString("cui.error.gsc.show"));
					return;
				}

				ArrayList row = new ArrayList();
				
				row.add(dbNames[i]);
				row.add(new Integer(instance.getId()));
				row.add(instance.getHostName());
				row.add(instance.getPortNumber());
				
				table2StringUtil.addRow(row);
			}
		}

		table2StringUtil.print();
	}

	/**
	 * DBインスタンスの追加
	 * @param cp コマンド解析結果
	 * @param gsc GSCヘの接続情報
	 */	
	public static void createDbInstance(CommandParser cp, GSCdata gsc) {

		//DB名取得
		String dbName = cp.getDB();
		if (dbName == null || dbName.equals(""))
		{
			Logger.error( GT.tr("データベース名が入力されていません。") );
			return;
		}

		//インスタンスIDの取得
		int instanceId = Integer.parseInt((String)cp.getServer().get(0));

		//コマンドラインから強制フラグを取得
		boolean force = cp.isForce();
		
		Database database = null;
		try {
			database = gsc.getDatabase(dbName);
		}
		catch (ForestToolException e)
		{
			Logger.error( GT.tr("データベース \"{0}\" の情報を取得できません。",
								dbName) );

			Logger.detail(e.getDetail().getMessage());
			Logger.trace(e.getDetail());
			return ;
		}
		if (database==null) {
			Logger.error( GT.tr("データベース \"{0}\" が見つかりません。",
								dbName) );
			return ;
		}
		
		Instance instance = gsc.getInstance(instanceId);
		if (instance == null) {
			Logger.error( GT.tr("サーバ \"{0}\" が見つかりません。",
								Integer.toString(instanceId)) );
			return;
		}

		/*
		 * 既に登録されているかどうかを確認
		 */
		int[] instanceIds = database.getInstanceIds();
		for (int i=0 ; i<instanceIds.length ; i++)
		{
			if ( instanceIds[i]==instanceId && !force )
			{
				Logger.error( GT.tr("インスタンス \"{0}\" は、すでにデータベース \"{1}\" に登録されています。",
									Integer.toString(instanceId), dbName ) );
				return;
			}
		}
		
		/*
		 * 既存のデータベースへインスタンスを追加
		 */
		if ( !database.addInstance(instance,force) )
		{
			Logger.error( GT.tr("データベース \"{0}\" へのサーバ \"{1}\" の追加に失敗しました。",
								dbName, Integer.toString(instanceId) ));
			return;
		}
		
		Logger.notice( GT.tr("データベース \"{0}\" へサーバ \"{1}\" を追加しました。",
							 dbName, Integer.toString(instanceId) ));
	}
	
	/**
	 * DBインスタンスの削除
	 * @param cp コマンド解析結果
	 * @param gsc GSCヘの接続情報
	 */	
	public static void  dropDbInstance(CommandParser cp, GSCdata gsc) {

		//DB名取得
		String dbName = cp.getDB();
		if(dbName == null || dbName.equals("")){
			Logger.error( GT.tr("データベース名が入力されていません。") );
			return;
		}

		int instanceId = Integer.parseInt((String)cp.getServer().get(0));

		
		//コマンドラインから強制フラグを取得
		boolean force = cp.isForce();
		
		try {
			if ( gsc.getDatabase(dbName)==null )
			{
				Logger.error( GT.tr("データベース \"{0}\" が見つかりません。",
									dbName) );
				return;
			}
			if ( gsc.getInstance(instanceId)==null )
			{
				Logger.error( GT.tr("インスタンス \"{0}\" が見つかりません。",
									Integer.toString(instanceId)) );
				return;
			}
		}
		catch (ForestToolException e)
		{
			Logger.error( GT.tr("インスタンスまたはデータベースの情報を取得できません。") );

			Logger.detail(e.getDetail().getMessage());
			Logger.trace(e.getDetail());
			return;
		}

		//データベース削除
		if ( !gsc.dropDatabase(dbName, instanceId, force) )
		{
			Logger.error( GT.tr("データベース \"{0}\" からのサーバ \"{1}\" の削除に失敗しました。",
								dbName, Integer.toString(instanceId) ));
			return;
		}

		Logger.notice( GT.tr("データベース \"{0}\" からサーバ \"{1}\" を削除しました。",
							 dbName, Integer.toString(instanceId) ));
	}

	/**
	 * DDLコマンドの実行
     * 入力情報"ddlform"(DynaValidatorForm)の、DDLをユーザデータベースに対して実行する。
     * DDL文は構文解析を行い、各DDLに対し以下のように処理す津。
     * ・CREATE TABLE
     *   GSCへテーブル情報（forest_tablepart,forest_tablepartdtl）を挿入する
     *   DDLを各ユーザデータベースで実行する。
     * ・DROP TABLE
     *   GSCへテーブル情報（forest_tablepart,forest_tablepartdtl）を削除する
     *   DDLを各ユーザデータベースで実行する。
     * ・ALTER TABLE
     *   指定されたテーブルが、パーティション化テーブルならば
     * 　テーブル名を実テーブルに置換し、実テーブル数分DDLを各ユーザデータベースで実行する。
     * ・CREATE INDEX
     *   指定されたテーブルが、パーティション化テーブルならば
     * 　テーブル名を実テーブル、インデックス名をインデックス名_00（パーティションNo）に置換し、
     * 　実テーブル数分DDLを各ユーザデータベースで実行する。
     * ・DROP INDEX
     *   指定されたインデックスが定義されたテーブルを特定し、
     *   テーブルが、パーティション化テーブルならば
     * 　テーブル名を実テーブル、インデックス名をインデックス名_00（パーティションNo）に置換し、
     * 　実テーブル数分DDLを各ユーザデータベースで実行する。
     * ・その他
     *   DDLを各ユーザデータベースで実行する。
	 * @param gsc GSCヘの接続情報
	 * @param command DDLコマンド
	 * @param selectDb データベース名
	 */	
	public static void executeDdl( String command, Parser parser, Database database) {
		
		
		//ユーザーデータベース名を取得
		

		try {

			Logger.notice( GT.tr("実行コマンド: {0}", command) );

			boolean statusOk = false;
			
			String tableName = null; 
			int type = parser.getType();
			Table table = null;
			String indexName = null;
			
			switch (type) {
				/* ----------------------------------------------
				 * テーブル関連処理
				 * ----------------------------------------------
				 */
				case Parser.CREATE_TABLE :
					if ( parser.getTables().size()==0 )
					{
						Logger.error( GT.tr("テーブル名が指定されていません。") );
						break;
					}

					tableName = (String)parser.getTables().get(0);

					if ( database.getTable(tableName)!=null )
					{
						Logger.error( GT.tr("指定したテーブル \"{0}\" が既に存在します。",
											tableName) );
					}
					else
					{
						table = database.createTable(command);
						if (table != null)
							statusOk = true;
					}

					break;

				case Parser.ALTER_TABLE :
					if ( parser.getTables().size()==0 )
					{
						Logger.error( GT.tr("テーブル名が指定されていません。") );
						break;
					}

					tableName = (String)parser.getTables().get(0);

					table = database.getTable(tableName);
					if ( table==null )
						Logger.error( GT.tr("指定したテーブル \"{0}\" が見つかりません。", tableName) );
					else
						statusOk = table.alter(command) ;

					break;

				case Parser.DROP_TABLE :
					if ( parser.getTables().size()==0 )
					{
						Logger.error( GT.tr("テーブル名が指定されていません。") );
						break;
					}

					tableName = (String)parser.getTables().get(0);

					if ( database.getTable(tableName)==null )
						Logger.error( GT.tr("指定したテーブル \"{0}\" が見つかりません。",
											tableName) );
					else
						statusOk = database.dropTable(tableName);

					break;

				/* ----------------------------------------------
				 * ビュー関連処理
				 *
				 * VIEWは、多重化テーブルと同等のものとして処理
				 * ----------------------------------------------
				 */
				case Parser.CREATE_VIEW:
					if ( parser.getTables().size()==0 )
					{
						Logger.error( GT.tr("ビュー名が指定されていません。") );
						break;
					}

					tableName = (String)parser.getTables().get(0);

					if ( database.getTable(tableName)!=null )
					{
						Logger.error( GT.tr("指定したビューまたはテーブル \"{0}\" が既に存在します。",
											tableName) );
					}
					else
					{
						table = database.createView(command);
						if (table != null)
							statusOk = true;
					}

					break;

				case Parser.DROP_VIEW:
					if ( parser.getTables().size()==0 )
					{
						Logger.error( GT.tr("ビュー名が指定されていません。") );
						break;
					}
					
					tableName = (String)parser.getTables().get(0);

					if ( database.getTable(tableName)==null )
						Logger.error( GT.tr("指定したビュー \"{0}\" が見つかりません。",
											tableName) );
					else
						statusOk = database.dropView(tableName);

					break;

				/* ----------------------------------------------
				 * インデックス関連処理
				 * ----------------------------------------------
				 */
				case Parser.CREATE_INDEX :
					if ( parser.getTables().size()==0 )
					{
						Logger.error( GT.tr("テーブル名が指定されていません。") );
						break;
					}

					tableName = (String)parser.getTables().get(0);
					table = database.getTable(tableName);
					if ( table==null )
						Logger.error( GT.tr("指定したテーブル \"{0}\" が見つかりません。",
											tableName) );
					else
						statusOk = table.createIndex(command);

					break;

				case Parser.ALTER_INDEX :
					indexName = parser.getReplaceWord();
					table = indexGetTable(database, indexName);

					if ( table==null )
						Logger.error( GT.tr("指定したインデックス \"{0}\" を持つテーブルが見つかりません。",
											indexName) );
					else
						statusOk = table.alter(command);

					break;

				case Parser.DROP_INDEX :
					indexName = parser.getReplaceWord();
					table = indexGetTable(database, indexName);

					if ( table==null )
						Logger.error( GT.tr("指定したインデックス \"{0}\" を持つテーブルが見つかりません。",
											indexName) );
					else
						statusOk = table.dropIndex(indexName);

					break;
				    
				    
				default :
					Logger.error( GT.tr("特定のDDLコマンド以外の実行はサポートされていません。") );
					break;
			}

			if ( statusOk )
			    Logger.notice( GT.tr("DDLコマンドの実行に成功しました。") );
			else
			    Logger.error( GT.tr("DDLコマンドの実行に失敗しました。") );

		}
		catch (ForestToolException e)
		{
			Logger.error( GT.tr("DDLコマンドの実行中にエラーが発生しました。") );

			Logger.detail(e.getDetail().getMessage());
			Logger.trace(e.getDetail());
		}				
	}

	private static Table indexGetTable(Database database, String indexName)
	{
		String table = indexGetRelationName(database, indexName);

		Table t = null;
		if ( table!=null )
		{
			t = database.getTable(table);

			if ( t==null )
				Logger.error( GT.tr("指定したテーブル \"{0}\" が見つかりません。",
									table) );
		}
		else
		{
			Logger.error( GT.tr("指定したインデックス \"{0}\" が見つかりません。",
								indexName) );
		}

		return t;
	}

	private static String indexGetRelationName(Database database, String indexName)
	{
		String[] tables = null;

		try {
			tables = database.getTableNames();
			
		} catch (ForestToolException e) {
			Logger.error( GT.tr("テーブル名一覧を取得できません。") );
			
			Logger.detail(e.getDetail().getMessage());
			Logger.trace(e.getDetail());
			return null;
		}

		if (tables.length == 0)
		{
			Logger.error( GT.tr("削除するテーブル名が指定されていません。") );
			return null;
		}
					
		for (int i = 0; i < tables.length; i++)
		{
			Table t = database.getTable(tables[i]);

			String[] indexNames = t.getIndexNames();

			if ( Misc.containsIgnoreCase(indexName,indexNames) )
			{
				return tables[i];
			}
		}

		return null;
	}
}
