package org.postgresforest.tool.cli.action;

import java.util.ArrayList;

import org.postgresforest.tool.Logger;
import org.postgresforest.tool.lib.ForestToolException;
import org.postgresforest.tool.lib.GSCdata;
import org.postgresforest.tool.lib.GT;
import org.postgresforest.tool.lib.Instance;
import org.postgresforest.tool.util.CommandParser;
import org.postgresforest.tool.util.MessagesCommandLine;
import org.postgresforest.tool.util.Table2StringUtil;

/**
 * GSCへのコマンドラインツール用のアクションを提供するクラス
 */
public class GscActions {

	static private String[] SHOW_GSC_HEADER = {"DBNAME", "HOST", "PORT", "SERVERID"}; 
	
	/**
	 * GSCの新規作成
	 * @param loginInfo ログイン情報
	 */
	static public void make(String hostname, String portno, String gscname, String username,	String userpass ) {
		
		try {
			GSCdata.initialize(hostname, portno, gscname, username, userpass );
			Logger.notice( GT.tr("GSC \"{0}\" を作成しました。", gscname) );
		} catch (ForestToolException e) {
			Logger.error( GT.tr("GSC \"{0}\" の作成に失敗しました。", gscname) );

		    Logger.detail( e.getDetail().getMessage() );
			Logger.trace( e.getDetail() );
		}
	}


	/**
	 * GSCの追加作成
	 * GSC名は接続先のものと同一なので、GSCのコピーを行う
	 * @param cp コマンド解析結果
	 * @param gsc GSCヘの接続情報
	 */
	public static void create(CommandParser cp, GSCdata gsc) {

		//コマンドラインから複製先サーバIDを取得
		int id = -1;
		try {
			id = Integer.parseInt((String)cp.getServer().get(0));
		} catch (NumberFormatException e) {
			Logger.error( GT.tr("入力されたサーバID \"{0}\" は不正です。",
								(String)cp.getServer().get(0)));

			Logger.detail(e.getMessage());
			Logger.trace(e);

			return;
		}

		//コマンドラインから強制フラグを取得
		boolean force = cp.isForce();

		//IDからインスタンス情報取得
		Instance instance = gsc.getInstance(id);
		if ( instance==null )
		{
			Logger.error( GT.tr("サーバID \"{0}\" のサーバが見つかりません。",
								(String)cp.getServer().get(0)));
			return;
		}
		String gscName = gsc.getGscName();
		int serverid = instance.getId();

		//GSCコピー実行
		if ( !gsc.copy(instance,force) )
		{
			Logger.error( GT.tr("GSC \"{0}\" のサーバID \"{1}\" への複製に失敗しました。",
								gsc.getGscName(),
								Integer.toString(serverid)) );
			return;
		}

		Logger.notice( GT.tr("GSC \"{0}\" をサーバID \"{1}\" へ複製しました。",
							 gsc.getGscName(),
							 Integer.toString(serverid)) );
	}
	
	/**
	 * GSCの削除
	 * @param cp コマンド解析結果
	 * @param gsc GSCヘの接続情報
	 */
	public static void drop(CommandParser cp, GSCdata gsc) {
		//最後のGSCの削除の場合、確認メッセージを表示する
		
		//コマンドラインからIDを取得
		int id = -1;
		try {
			id = Integer.parseInt((String)cp.getServer().get(0));
		} catch (NumberFormatException e) {
			Logger.error( GT.tr("入力されたサーバID \"{0}\" は不正です。",
								(String)cp.getServer().get(0)));

			Logger.detail(e.getMessage());
			Logger.trace(e);

			return;
		}

		//IDからインスタンス情報取得
		Instance instance = gsc.getInstance(id);
		if ( instance==null )
		{
			Logger.error( GT.tr("サーバID \"{0}\" のサーバが見つかりません。",
								(String)cp.getServer().get(0)));
			return;
		}
		
		//GSC削除
		if( !gsc.drop(instance) ){
			Logger.error( GT.tr("GSC \"{0}\" のサーバID \"{1}\" からの削除に失敗しました。",
								gsc.getGscName(),
								Integer.toString(instance.getId())) );
			return;
		}
		
		//削除しました
		Logger.notice( GT.tr("GSC \"{0}\" をサーバID \"{1}\" から削除しました。",
							 gsc.getGscName(),
							 Integer.toString(instance.getId())) );
	}
	
	/**
	 * GSCの整合性チェック
	 * @param gsc GSCヘの接続情報
	 */
	public static void check(GSCdata gsc) {
		//グローバルシステムカタログ整合性チェック処理

		//GSCの同一性をチェック
		boolean ok = true;

		int[] ids = gsc.getGscInstances();
		for (int i=0 ; i<ids.length ; i++)
		{
			Instance ins =  gsc.getInstance(ids[i]);

			if ( !gsc.equals(ins) )
			{
				Logger.error( GT.tr("GSC \"{0}\" の内容がサーバID \"{1}\" と同一ではありません。",
									gsc.getGscName(),
									Integer.toString(ins.getId())) );
				ok = false;
			}
		}

		/*
		 * ユーザデータの同一性チェック
		 */
		if ( !gsc.validate(GSCdata.VALIDATE_SCHEMA|GSCdata.VALIDATE_RECORD_COUNT) )
		{
			Logger.error( GT.tr("テーブル定義あるいはレコードの内容が同一ではありません。"));
			ok = false;
		}

		if ( ok )
			Logger.notice( GT.tr("GSCおよびテーブルの内容に相違はありません。") );
	}
	
	/**
	 * GSCの表示
	 * @param gsc GSCヘの接続情報
	 */
	public static void show(GSCdata gsc) {

		int[] ids = gsc.getGscInstances();

		if ( ids==null )
		{
			Logger.error( GT.tr("サーバIDを取得できません。") );
			return;
		}
		
		//GSC情報表示
		Logger.println(MessagesCommandLine.getString("cui.message.gsc.show"));

		Table2StringUtil table2StringUtil = new Table2StringUtil( SHOW_GSC_HEADER );

		//IDからインスタンス情報取得し表示
		
		for (int i = 0; i < ids.length; i++) {
			int id = ids[i];
			//IDからインスタンス情報取得
			Instance instance = gsc.getInstance(id);
			if(instance == null){
				//インスタンス情報の取得失敗
				Logger.error(MessagesCommandLine.getString("cui.error.gsc.show"));
				return;
			}

			//インスタンス情報を表示
			ArrayList row = new ArrayList();
			row.add(gsc.getGscName());
			row.add(instance.getHostName());
			row.add(instance.getPortNumber());
			row.add(new Integer(instance.getId()));
			table2StringUtil.addRow(row);
		}
		
		table2StringUtil.print();
	}
}
