package org.postgresforest.tool.cli.action;


import java.util.ArrayList;

import org.postgresforest.tool.Logger;
import org.postgresforest.tool.lib.ForestToolException;
import org.postgresforest.tool.lib.GSCdata;
import org.postgresforest.tool.lib.GT;
import org.postgresforest.tool.lib.Instance;
import org.postgresforest.tool.util.CommandParser;
import org.postgresforest.tool.util.MessagesCommandLine;
import org.postgresforest.tool.util.Table2StringUtil;

/**
 * インスタンスへのコマンドラインツール用のアクションを提供するクラス
 */
public class InstanceActions {

	static private final String[] SHOW_INSTANCE_HEADER = {"SERVERID", "HOST", "PORT", "STATUS"}; 
	static public final int SERVER_RUNNING = 1; 
	static public final int SERVER_TROUBLE = -1; 

	/**
	 * インスタンスの作成
	 * @param cp コマンド解析結果
	 * @param gsc GSCヘの接続情報
	 */
	public static void create(CommandParser cp, GSCdata gsc) {

		//ホスト名、ポートNoを取得
		String host = cp.getHost();	
		if(host == null || host.equals("")){
			Logger.error( GT.tr("ホスト名またはIPアドレスが指定されていません。") );
			return;
		}		
		String port = cp.getPort();	
		if(port == null || port.equals("")){
			Logger.error( GT.tr("ポート番号が指定されていません。") );
			return;
		}		
		
		//インスタンス登録
		Instance instance = gsc.registerInstance(host, port);
		if (instance == null)
		{
			Logger.error( GT.tr("サーバ \"{0}:{1}\" の登録に失敗しました。",
								host, port) );
			return;
		}

		//インスタンス情報にステータスを設定
		if ( !instance.setStatus(SERVER_RUNNING) )
		{
			Logger.error( GT.tr("サーバ \"{0}\" のステータス変更に失敗しました。",
								Integer.toString(instance.getId()) ) );
			return;
		}

		Logger.notice( GT.tr("サーバ \"{0}:{1}\" を登録しました。",
							 host, port) );
	}
	
	/**
	 * インスタンスの削除
	 * @param cp コマンド解析結果
	 * @param gsc GSCヘの接続情報
	 */	
	public static void drop(CommandParser cp, GSCdata gsc) {
		String id = (String)cp.getServer().get(0);
		int instanceId = 0;

		if ( id==null || id.equals("") )
		{
			Logger.error( GT.tr("サーバIDが指定されていません。") );
			return;
		}

		try {
			instanceId = Integer.parseInt(id);
		}
		catch (NumberFormatException e)
		{
			Logger.error( GT.tr("指定されたサーバIDが不正です。") );
			return;
		}

		if ( gsc.getInstance(instanceId)==null )
		{
			Logger.error( GT.tr("サーバID \"{0}\" が見つかりません。",
								id) );
			return;
		}

		//インスタンス削除実行
		if ( gsc.unregisterInstance(instanceId) )
		{
			Logger.notice( GT.tr("サーバID \"{0}\" の削除を行いました。",
								 id) );
			return;
		}

		Logger.error( GT.tr("サーバID \"{0}\" の削除に失敗しました。",
							id) );
	}
	
	/**
	 * インスタンスの表示
	 * @param gsc GSCヘの接続情報
	 */		
	public static void show(GSCdata gsc) {
		int[] ids = null;

		//インスタンスIDのリストを取得
		try {
			ids = gsc.getInstanceIds();
			
		} catch (ForestToolException e) {
			Logger.error( GT.tr("サーバID一覧の取得に失敗しました。") );

			Logger.error(e.getDetail().getMessage());
			Logger.trace(e.getDetail());
			return;
		}
		
		
		//IDからインスタンス情報取得し表示
		System.out.println(MessagesCommandLine.getString("cui.message.instance.show")) ; //$NON-NLS-1$

		Table2StringUtil table2StringUtil = new Table2StringUtil( SHOW_INSTANCE_HEADER );

		if(ids != null ){

			for (int i = 0; i < ids.length; i++) {
				int id = ids[i];
				Instance instance = gsc.getInstance(id);
				if(instance == null){
					//インスタンス情報の取得失敗
					Logger.error( GT.tr("サーバID \"{0}\" が見つかりませんでした。", Integer.toString(id)) );
					return;
				}
				ArrayList row = new ArrayList();
	
				row.add(new Integer(instance.getId())); //$NON-NLS-1$
				row.add(instance.getHostName()); //$NON-NLS-1$
				row.add(instance.getPortNumber()); //$NON-NLS-1$
				row.add(convertStatus(instance.getStatus())); //$NON-NLS-1$
				table2StringUtil.addRow(row);
				
			}
		}
		
		table2StringUtil.print();

	}

	/**
	 * インスタンスの状態設定
	 * @param cp コマンド解析結果
	 * @param gsc GSCヘの接続情報
	 * @param status インスタンスの設定状態
	 * 	 */	
	public static void set(CommandParser cp, GSCdata gsc, int status) {	


		//コマンドラインからIDを取得
		int id;

		try {
			id = Integer.parseInt((String)cp.getServer().get(0));
		}
		catch (NumberFormatException e)
		{
			Logger.error( GT.tr("入力されたサーバIDは不正です。") );
			return;
		}

		//IDからインスタンス情報取得
		Instance instance = gsc.getInstance(id);
		if(instance == null){
			Logger.error( GT.tr("サーバID \"{0}\" が見つかりませんでした。",
								Integer.toString(id)) );
			return;
		}
		
		if( instance.getStatus()==status )
		{
			Logger.notice( GT.tr("サーバID \"{0}\" のステータスは変更されません。",
								 Integer.toString(id)) );
			return;
		}


		//インスタンス情報にステータスを設定
		if( !instance.setStatus(status) )
		{
			Logger.error( GT.tr("サーバID \"{0}\" のステータス変更に失敗しました。",
								Integer.toString(id)) );
			return;
		}
		
		//終了メッセージ表示
		Logger.notice( GT.tr("サーバID \"{0}\" のステータスを \"{1}\" に変更しました。",
							 Integer.toString(id), convertStatus(status) ));
		
	}
	
	private static String convertStatus(int status) {
		String ret = "";
		if (status == SERVER_RUNNING) {
			ret = GT.tr("稼働中");
		} else if (status == SERVER_TROUBLE) {
			ret = GT.tr("障害中");
		}else{
			ret = GT.tr("不明");
		}
		return ret;
	}
}
