package org.postgresforest.tool.cli.action;

import java.text.DecimalFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Iterator;

import org.postgresforest.tool.Logger;
import org.postgresforest.tool.lib.Database;
import org.postgresforest.tool.lib.GT;
import org.postgresforest.tool.lib.ForestToolException;
import org.postgresforest.tool.lib.Table;
import org.postgresforest.tool.util.CommandParser;
import org.postgresforest.tool.util.MessagesCommandLine;
import org.postgresforest.tool.util.Misc;
import org.postgresforest.tool.util.Table2StringUtil;

/**
 * テーブルへのコマンドラインツール用のアクションを提供するクラス
 */
public class TableActions {

	private static final String[] SHOW_TABLE_HEADER = {"TABLENAME","TABLETYPE","PARTITION_FUNC","STATUS" }; //$NON-NLS-1$
	private static final String[] SHOW_COLUMN_HEADER = {"PARTITION_KEY","COLUMNNAME","DATATYPE" }; //$NON-NLS-1$

	private static final int REST_MODE = 1; //$NON-NLS-1$
	private static final int NOREST_MODE = 0; //$NON-NLS-1$
	private static final String DEFAULT_HASH = "default"; //$NON-NLS-1$
	private static final String HASH_OPTION_NAME = "hashname"; //$NON-NLS-1$

	protected static DecimalFormat m_dcFmt = new DecimalFormat("00"); //$NON-NLS-1$
	/**
	 * パーティション化1設定
	 * @param cp コマンド解析結果
	 * @param database データベース情報
	 */
	private static boolean _part(String tableName, int partCount, ArrayList partColumns,
								 ArrayList paramNames, ArrayList paramValues, Database database) {
		
		if (tableName == null || tableName.equals(""))
		{
			Logger.error( GT.tr("テーブル名が入力されていません。") );
			return false;
		}

		// テーブル情報の取得
		Table table = database.getTable(tableName);
		if (table == null){
			//テーブル情報の取得失敗
			Logger.error( GT.tr("テーブル \"{0}\" が見つかりません。",
								tableName) );
			return false;
		}

		String hashName = null;
		for (int i=0; i<paramNames.size(); i++) {
			String paramName = (String)paramNames.get(i);
			if (paramName.equalsIgnoreCase(HASH_OPTION_NAME) ) {
				// パーティション関数の設定
				hashName = (String)paramValues.get(i);
				if (hashName.equals(DEFAULT_HASH)) {
					hashName = null;
				}
			} else {
				/*
				 * Hash関数以外の引数が指定された場合
				 */
				Logger.error( GT.tr("指定されたパラメータ \"{0}\" は無効です。",
									paramName) );
				return false;
			}
		}

		// パーティション数の妥当性チェック
		int serverSize = database.getInstanceIds().length;
		if (partCount < 2 || partCount > serverSize)
		{
			Logger.error( GT.tr("指定したパーティション数 \"{0}\" は無効です。",
								Integer.toString(partCount) ));
			return false;
		}

		
		// パーティション属性の設定
		String[] partColumnsName  = (String[])partColumns.toArray(new String[0]);
		
		//テーブルのパーティション化
		try {
			if (table.part(partColumnsName, partCount, hashName))
			{
				Logger.notice( GT.tr("テーブル \"{0}\" をパーティション化しました。",
									 tableName) );
				return true;
			}
			else
			{
				Logger.error( GT.tr("テーブル \"{0}\" のパーティション化に失敗しました。",
									tableName) );
			}
		}
		catch (ForestToolException e)
		{
			Logger.error( GT.tr("テーブル \"{0}\" のパーティション化に失敗しました。",
								tableName) );

			Logger.detail(e.getDetail().getMessage());
			Logger.trace(e.getDetail());
		}

		return false;
	}
	
	public static void part(CommandParser cp, Database database) {

		_part(cp.getTables(),
			  cp.getPartCnt(),
			  cp.getKeyCol(),
			  cp.getParamName(),
			  cp.getParamValue(),
			  database);

		return;
	}

	/**
	 * パーティション化2設定
	 * @param cp コマンド解析結果
	 * @param database データベース情報
	 */
	public static void part2(CommandParser cp, Database database) {

		if ( !_part(cp.getTables(), cp.getPartCnt(), cp.getKeyCol(),
					cp.getParamName(), cp.getParamValue(), database) )
			return;

		//サーバIDのチェック
		ArrayList setServerList = cp.getServer();
		int[] serverIds = database.getInstanceIds();

		for (Iterator iter = setServerList.iterator(); iter.hasNext();) {
			String severID = (String) iter.next();
			int setid = Integer.parseInt(severID);
			if ( !Misc.contains(setid,serverIds) )
			{
				Logger.error( GT.tr("指定したサーバID \"{0}\" にはデータベース \"{1}\" はありません。",
									severID, database.getDatabaseName() ) );
				return;
			}
		}

		// パーティション2化テーブルのパーティションテーブル配置設定の取得
		HashMap part2NoListMap = cp.getPart2NoListMap();
		
		
		Table table = database.getTable(cp.getTables());

		//指定されていないパーティションを各インスタンスから削除する
		for (int i = 0; i < serverIds.length; i++) {
			
			int serverId = serverIds[i];
			ArrayList partNoList = (ArrayList)part2NoListMap.get(String.valueOf(serverId));
			
			if( partNoList == null){
				
				//サーバIDが指定されていなかったらすべてのパーティションを削除
				for (int k = 0 ; k < cp.getPartCnt() ; k++) {
						
					if ( !table.dropPart( k, serverId) )
					{
						Logger.error( GT.tr("サーバID \"{0}\" において、テーブル \"{1}\" のパーティション \"{2}\" の削除に失敗しました。",
											Integer.toString(serverId),
											cp.getTables(),
											Integer.toString(k) ));
						return;
					}							
				}					
				
			}else{
				
				//指定されていないパーティションを削除

				
				for (int k = 0 ; k < cp.getPartCnt() ; k++) {
					if( ! partNoList.contains( String.valueOf(k) ) ){
						
						if ( ! table.dropPart( k, serverId) ) {
							Logger.error( GT.tr("サーバID \"{0}\" において、テーブル \"{1}\" のパーティション \"{2}\" の削除に失敗しました。",
												Integer.toString(serverId),
												cp.getTables(),
												Integer.toString(k) ));
							return;
						}							
					}
				}					
				
			}
			
		}
		
		Logger.notice( GT.tr("テーブル \"{0}\" をパーティション化しました。",
							  cp.getTables() ));
	}

	/**
	 * パーティション化解除
	 * @param cp コマンド解析結果
	 * @param database データベース情報
	 */
	public static void unpart(CommandParser cp, Database database) {
		// テーブル名設定
		String tableName = cp.getTables();		
		if (tableName == null || tableName.equals(""))
		{
			Logger.error( GT.tr("テーブル名が入力されていません。") );
			return;
		}

		
		Table table = database.getTable(tableName);
		if(table == null){
			//テーブル情報の取得失敗
			Logger.error( GT.tr("テーブル \"{0}\" が見つかりません。",
								tableName) );
			return;
		}


		try {
			if ( table.unpart() )
			{
				Logger.notice( GT.tr("テーブル \"{0}\" を多重化テーブルに変更しました。",
									 tableName ));
			}
			else
			{
				Logger.error( GT.tr("テーブル \"{0}\" の多重化テーブルへの変更に失敗しました。",
									tableName ));
			}
		} catch (ForestToolException e) {
			Logger.error( GT.tr("テーブル \"{0}\" の多重化テーブルへの変更に失敗しました。",
								tableName ));

			Logger.detail(e.getDetail().getMessage());
			Logger.trace(e.getDetail());
		}

		return;
	}


	/**
	 * テーブル一覧表示
	 * @param gsc GSCヘの接続情報
	 * @param database データベース情報
	 */
	public static void showTable(Database database) {

		String[] tables;

		//テーブル名のリストを取得
		try {
			tables = database.getTableNames();
			
		} catch (ForestToolException e) {
			Logger.error( GT.tr("テーブル名一覧を取得できません。") );

			Logger.detail(e.getDetail().getMessage());
			Logger.trace(e.getDetail());
			return;
		}

		//DB情報表示
		Logger.println(MessagesCommandLine.getString("cui.message.table.show"));

		Table2StringUtil table2StringUtil = new Table2StringUtil( SHOW_TABLE_HEADER );

		
		//IDからインスタンス情報取得し表示
		
		for (int i = 0; i < tables.length; i++) {
			String tableName = tables[i];
			//テーブル名からテーブル情報取得
			Table table = database.getTable(tableName);
			if (table == null)
			{
				//テーブル情報の取得失敗
				Logger.error( GT.tr("テーブル \"{0}\" が見つかりません。",
									tableName) );
				return;
			}

			ArrayList row = new ArrayList();
			row.add(tableName );
			int type = table.getTableType();
			row.add(convertTableType(type));
			
			String hashname = table.getHashName();
			if (hashname != null ) {
				row.add(hashname);
			}
			else if (type == Table.TYPE_PARTITION_1 || type == Table.TYPE_PARTITION_2) {
				row.add("DEFAULT");
			}
			else {
				row.add("");
			}

			row.add(convertTableStatus(table.getStatus()));
						
			table2StringUtil.addRow(row);
		}
				
		table2StringUtil.print();
	}

	/**
	 * テーブルカラム情報表示
	 * @param tableName テーブル名
	 * @param database データベース情報
	 */
	public static void showTableColumn(CommandParser cp, Database database) {
		String tableName = cp.getTables();
		if(tableName == null || tableName.equals("")){
			Logger.error( GT.tr("テーブル名が入力されていません。") );
			return;
		}

		showTableColumn(tableName, database);
	}
	/**
	 * テーブルカラム情報表示
	 * @param tableName テーブル名
	 * @param database データベース情報
	 */
	public static void showTableColumn(String tableName, Database database) {
		

		Table table = database.getTable(tableName);
		if(table == null){
			//テーブル情報の取得失敗
			Logger.error( GT.tr("テーブル \"{0}\" が見つかりません。",
								tableName) );
			return;
		}

		
		//Column名リスト表示
		String[] columnNames = null;
		ArrayList partitonKeyList = null;

		try {
			columnNames = table.getColumnNames();
			partitonKeyList = new ArrayList(Arrays.asList(table.getPartitionKeys())) ;
		}
		catch (ForestToolException e)
		{
			Logger.error( GT.tr("テーブル \"{0}\" のカラム名を取得できません。",
								tableName) );

			Logger.detail(e.getDetail().getMessage());
			Logger.trace(e.getDetail());
			return;
		}

		//Column情報表示
		Logger.println(MessagesCommandLine.getString("cui.message.tablecolumn.show", new Object[]{tableName}) );
		
		Table2StringUtil table2StringUtil = new Table2StringUtil( SHOW_COLUMN_HEADER );

		
		String columnName = null;
		try {
			//Column名から情報取得し表示
			for (int i = 0; i < columnNames.length; i++) {
				columnName = columnNames[i];
				//テーブル名からテーブル情報取得
				
				ArrayList row = new ArrayList();
				row.add(convertPartKey(columnName, partitonKeyList ) ); //$NON-NLS-1$
				row.add(columnName); //$NON-NLS-1$
				row.add(table.getColumnType(columnName) ); //$NON-NLS-1$
							
				table2StringUtil.addRow(row);
					
			}
			table2StringUtil.print();
		}
		catch (ForestToolException e)
		{
			Logger.error( GT.tr("カラム \"{0}\" の情報の取得に失敗しました。",
								columnName) );

			Logger.detail(e.getDetail().getMessage());
			Logger.trace(e.getDetail());
		}
	}
	/**
	 * 優先度設定
	 * @param cp コマンド解析結果
	 * @param database データベース情報
	 */
	public static void setPriority(CommandParser cp, Database database) {

		String tableName = cp.getTables();
		if(tableName == null || tableName.equals(""))
		{
			Logger.error( GT.tr("テーブル名が入力されていません。") );
			return;
		}

		Table table = database.getTable(tableName);
		if (table == null) {
			Logger.error( GT.tr("テーブル \"{0}\" が見つかりません。",
								tableName));
			return;
		}

		int partNo = cp.getPartNo();
		ArrayList serverList = cp.getServer();

		for (int pri=0 ; pri<serverList.size() ; pri++)
		{
			int serverid = Integer.parseInt((String)serverList.get(pri));
			if ( !table.setPriority(serverid, partNo, pri) )
			{
				Logger.error( GT.tr("サーバID \"{0}\" 上のテーブル \"{1}\" ／パーティション \"{2}\" の優先度を設定できませんでした。",
									Integer.toString(serverid), tableName, Integer.toString(partNo)));
				return;
			}

			Logger.debug("setPriority: serverid=" + serverid + ",partition=" + partNo + ",priority=" + pri);
		}
		Logger.notice( GT.tr("テーブル \"{0}\" のパーティション \"{1}\" の優先度を設定しました。",
							 tableName, Integer.toString(partNo) ));
	}

	/**
	 * 優先度表示(show priority)
	 * @param cp コマンド解析結果
	 * @param database データベース情報
	 */
	public static void showPriority(CommandParser cp, Database database) {
		String tableName = cp.getTables();
		if (tableName == null || tableName.equals(""))
		{
			Logger.error( GT.tr("テーブル名が入力されていません。") );
			return;
		}

		showPriority(tableName, database);
	}

	/**
	 * 優先度表示(\fp)
	 * @param tableName テーブル名
	 * @param database データベース情報
	 */
	public static void showPriority(String tableName, Database database) {
		
		Table table = database.getTable(tableName);
		if (table == null) {
			Logger.error( GT.tr("テーブル \"{0}\" が見つかりません。",
								tableName));
			return;
		}

		int partCount = table.getPartCount();

		if ( partCount==1 )
		{
			Logger.notice( GT.tr("テーブル \"{0}\" はパーティションテーブルではありません。",
								 tableName) );
			return;
		}

		/*
		 * テーブル表示用ヘッダを作成する（サーバID + パーティション数分）
		 */
		ArrayList columnList = new ArrayList();
		columnList.add( 0,"SERVERID");
		for (int i = 0; i < table.getPartCount(); i++) {
			columnList.add(tableName + "_" + m_dcFmt.format(i));
		}
		
		Logger.println(MessagesCommandLine.getString("cui.message.priority.show", new Object[]{tableName}) );
		
		Table2StringUtil table2StringUtil = new Table2StringUtil( columnList );

		try {
			/*
			 * 各インスタンスの処理
			 */
			int[] ids = table.getInstanceIds();
			
			for (int j=0 ; j<ids.length ; j++)
			{
				ArrayList row = new ArrayList();

				int serverId = ids[j];
				row.add(new Integer(serverId));

				/*
				 * 各パーティションの優先度を取得
				 */
				for (int i = 1; i < columnList.size(); i++ )
				{
					int priority = table.getPriority(serverId, i-1);

					if (priority < 0 )
						row.add("");
					else
						row.add( new Integer(priority) );

					//					Logger.println();
				}

				table2StringUtil.addRow(row);
			}
		
			table2StringUtil.print();

		} catch (ForestToolException e) {
			Logger.error( GT.tr("パーティションテーブルのサーバまたは優先度情報の取得ができませんでした。") );

			Logger.detail(e.getDetail().getMessage());
			Logger.trace(e.getDetail());
		}
	}

	
	/*
	 * FIXME: need to implement
	 *
	private boolean validatePriority(TableInfo tableInfo) {
		int m_partCount = tableInfo.getPartCount();
		ArrayList m_servers = tableInfo.getServer();
		boolean result = true;
		
		//優先順チェック	
		for (int partNo = 0; partNo < m_partCount; partNo++) {
			ArrayList priorityNoList = new ArrayList();
			//優先順配列を作成
			for (Iterator iterator = m_servers.iterator(); iterator.hasNext();) {		
				ServerPriorityInfo svrPriorityInfo =
					(ServerPriorityInfo) iterator.next();

				PriorityInfo priorityInfo =	svrPriorityInfo.getPartition(partNo);
				if(priorityInfo != null){
					priorityNoList.add(priorityInfo.getPriority());
				}
			}

			//優先順配列をチェック
			for (int priority = 0; priority < priorityNoList.size(); priority++) {
				boolean check = false;
				boolean duplicate = false;
				for (int i = 0; i < priorityNoList.size(); i++) {
					Integer priorityNo = (Integer)priorityNoList.get(i);
					if (priority == priorityNo.intValue()) {
						if(check){
							duplicate = true;
							break;
						}
						check = true;
					}
				}
				if (duplicate) {
				Logger.error(MessagesCommandLine.getString("errors.duplicate", m_dcFmt.format(partNo)));
					result = false;
					break;
				}
				if (!check) {
					String[] param = {m_dcFmt.format(partNo),Integer.toString(priority)};
					Logger.error(MessagesCommandLine.getString("errors.priority", param));
					result = false;
					break;
				}
			}
		}
		
		
		return result;
	}
	*/

	
	private static String convertTableType(int type) {
		String ret = "";
		if (type == Table.TYPE_REPLICATED ) {
			ret = GT.tr("多重化");
		} else if (type == Table.TYPE_PARTITION_1){
			ret = GT.tr("パーティション1");
		} else if (type == Table.TYPE_PARTITION_2){
			ret = GT.tr("パーティション2");
		} else {
			ret = GT.tr("不明");
			Logger.warning( GT.tr("テーブルタイプに不明な値が含まれています。") );
		}
		return ret;
	}
	
	
	private static String convertPartKey(String column , ArrayList partitionKeyList ) {
		String ret = " ";
		if ( partitionKeyList.contains(column) ){
			ret = " * ";
		} else {
			ret = "";
		}
		return ret;
	}
	
	
	private static String convertTableStatus(int status) {
		String ret = "";
		if (status == REST_MODE ) {
			ret = GT.tr("メンテナンス中");
		} else if (status == NOREST_MODE){
			ret = GT.tr("運用中");
		} else {
			ret = GT.tr("不明");
			Logger.warning( GT.tr("テーブルステータスに不明な値が含まれています。") );
		}
		return ret;
	}

}
