<?php
/*
* bogoYAML parser with PHP
*
* @author Haruki Setoyama <haruki@planewave.org>
* @copyright Copyright (c) 2005, Haruki SETOYAMA <haruki@planewave.org>
* @license New BSD http://www.opensource.org/licenses/bsd-license.php
* @version 0.1 $Id: BogoYaml.php,v 1.2 2005/03/21 01:37:03 haruki Exp $
* @package BogoYaml
*/
/* require PEAR */
require_once 'PEAR.php';
/* types of array */
define('BOGOYAML_ARRAY', 0);
define('BOGOYAML_ASSOC', 1);
/*
* bogoYAML
* ========
*
* `YAML`_ is good, but it has too many features, at least for some simple use, 
* like configration, etc. bogoYAML deliberately lacks some features:
*
* - All scalars are treated as string.
* - Only one indexed or associative array can exist per a line.
* - "---" is always necessary. No strings except comments can be before "---". 
* - There is no "...".
* - There is no Numeric and Additional Escape code.
* - There is no Block scalar indicator, Folded scalar indicator.
* - There is no Transfer indicator.
* - There is no Alias indicator.
* - There is no Key indicator. 
*
* .. _YAML: http://www.yaml.org/ 
*
* This liblary is based on the java version of the original `bogoYAML` which 
* was written by shinichiro.h. Poor quality in this liblary would be of course 
* blamed for me, not for him. The PHP version may be or will be incompatible to 
* the original. 
* 
* .. _bogoYAML: http://shinh.skr.jp/bogoyaml/index_en.html
*
* Usage
* -----
*
* require_once 'BogoYaml.php';
* $parser = new BogoYaml;
* $data = $parser->load($yaml_text);
* $yaml_dumped = $parser->dump($data);
*/
class BogoYaml extends PEAR {
    
    /**
    * error objects
    * @access private
    */
    var $_error_class = 'BogoYaml_Error';
    
    /**
    * parses YAML text and returns data as array.
    * @param string
    * @return array|BogoYaml_Error
    */
    function load($input) 
    {
        // init
        /* loaded data */
        $this->_stack = null;
        /* indent depth */
        $this->_depth = -1;
        /* references to array for each indent depth */
        $this->_bases = array();
        /* references to the last content for each indent depth */
        $this->_heads = array();
        $this->_heads[$this->_depth][0] =& $this->_stack;
        $this->_heads[$this->_depth][1] =  1;
        /* data type. array or assoc */ 
        $this->_types = array();
        /* charactor for indent. space or tab? */
        $this->_indent_char_use = null;
        $this->_indent_char_not = null;

        /* lines of given string */
        $this->_input = preg_split('/\r?\n/', $input);
        /* pointer in a line */
        $this->_position = 0;
        /* poiner for line */
        $this->_line = 0;
        /* the number of the lines */
        $this->_lines = count($this->_input);
        
        // state machine
        $state = new BogoYaml_ParserFirstLine();
        
        /* // for debug
        for ($i=0; $i<100; $i++) {
            echo $this->_line.' '.$this->_position.' '.get_class($state).'<br>';
            if (!is_a($state, 'BogoYaml_ParserState')) break;
            $state = $state->parse($this);
        }
        exit();*/
        
        while (is_a($state, 'BogoYaml_ParserState')) {
            $state = $state->parse($this);
        }
        if ($state !== true) {
            if (is_string($state)) {
                return $this->raiseError($state);
            } else {
                return $state;
            }
        } 
        return $this->_stack;
    }

    /**
    * returns YAML text for given data.
    * @param array
    * @return string|BogoYaml_Error
    */
    function dump($o) 
    {
        $ret = $this->_dumpLocal($o, 0);
        if (is_string($ret)) {
            return "--- # bogoYAML(PHP):0.1".$ret;
        } else {
            return $ret;
        }
    }

    /**
    * @access private
    */
    function _dumpLocal($o, $depth)
    {
        if (is_array($o)) {
            $ret = '';
            if ($this->_is_assoc($o)) {
                foreach ($o as $key => $value) {
                    $ret .= "\n" . str_repeat('  ', $depth).$key.': ';
                    $cnt = $this->_dumpLocal($value, $depth+1);
                    if (! is_string($cnt)) return $cnt; // error
                    $ret .= $cnt;
                }
            }
            else  {
                foreach ($o as $key => $value) {
                    $ret .= "\n" . str_repeat('  ', $depth).'- ';
                    $cnt = $this->_dumpLocal($value, $depth+1);
                    if (! is_string($cnt)) return $cnt; // error
                    $ret .= $cnt;
                }
            }
            return $ret;
        }
        elseif (is_object($o)) {
            $ret = '';
            foreach ((array)$o as $key => $value) {
                $ret .= "\n" . str_repeat('  ', $depth).$key.': ';
                $cnt = $this->_dumpLocal($value, $depth+1);
                if (! is_string($cnt)) return $cnt; // error
                $ret .= $cnt;
            }
            return $ret;
        }
        elseif (is_numeric($o)) {
            return "$o";
        }
        elseif (is_string($o)) {
            return '"'.$this->_add_slash($o).'"';
        }
        else {
            $o = $this->_dumpLocal_to_string($o);
            if (is_string($o)) {
                return $o;
            } else {
                return parent::raiseError('cannot dumpable Object');
            }
        }
    }
    
    /**
    * @access private
    */
    function _dumpLocal_to_string($o)
    {
        if ($o === true) {
            return '1 # true';
        }
        if ($o === false) {
            return '0 # false';
        }
        if ($o === null) {
            return '# null';
        }
        else {
            return $o;
        }
    }
    
    /**
    * chages given string to mixed data according to YAML specification.
    * @static
    * @param string
    * @param mixed
    */
    function strTodata($str)
    {
        static $t = array('y','Y','yes','Yes','YES','true','True','TRUE', 'on','On','ON');
        static $f = array('n','N','no','No','NO','false','False','FALSE','off','Off','OFF');
        static $n = array('~','null','Null','NULL');
            
        if (in_array($str, $t)) return true;
        if (in_array($str, $f)) return false;
        if (in_array($str, $n)) return null;
        if (is_numeric($str)) return ($str + 0);
        return $str;
    }
    
    /**
    * chages given data, especially bool and null, to YAML string.
    * @static
    * @param mixed
    * @param string
    */
    function dataToStr($o)
    {
        if ($o === true) {
            return 'TRUE';
        }
        if ($o === false) {
            return 'FALSE';
        }
        if ($o === null) {
            return 'NULL';
        }
        else {
            return $o;
        }
    }

/* --- functions below are called by `BogoYaml_ParserState` --- */

    function current() 
    {
        return substr($this->_input[$this->_line], $this->_position, 1);
    }
    
    function substr($offset = null) 
    {
        if (isset($offset)) {
            return substr($this->_input[$this->_line], $this->_position, $offset);
        } else {
            return substr($this->_input[$this->_line], $this->_position);
        }
    }

    function goForward($step =1) 
    { 
        $this->_position += $step; 
    }
    
    function goNextLine()
    {
        $this->_line += 1;
        $this->_position = 0;
    }
    
    function procLineEnd() 
    {
        $s = $this->substr();
        if ($s === false) {
            $this->goNextLine();
            return true;
        }
        $s = ltrim($s);
        if ($s == '' || substr($s, 0, 1) == '#') {    
            $this->goNextLine();
            return true;
        }
        return false;
    }

    function isEnd() 
    {
        return $this->_line >= $this->_lines;
    }

    function addBase($type) 
    {
        if (! isset($this->_bases[$this->_depth])) {
            $d = $this->_depth - 1;
            while (! isset($this->_heads[$d])) {
                $d -= 1;
            }
            
            if ($this->_heads[$d][0] === null) {
                $this->_heads[$d][0] = array();
                $this->_bases[$this->_depth] =& $this->_heads[$d][0];
                $this->_types[$this->_depth] = $type;
            } else {
                return $this->raiseError('invalid indentation');
            }
        } else {
            if ($this->_types[$this->_depth] == BOGOYAML_ASSOC) {
                if ($type != BOGOYAML_ASSOC) {
                    return $this->raiseError('associative array expected');
                }
            } else {
                if ($type == BOGOYAML_ASSOC) {
                    return $this->raiseError('indexed array expected');
                }
            }
        }
        return true;
    }

    function add($holder, $key =null) 
    {
        unset($this->_heads[$this->_depth]);
        $this->_heads[$this->_depth][1] = 1;
        if (isset($key)) {
            $this->_bases[$this->_depth][$key] = $holder;
            $this->_heads[$this->_depth][0] =& $this->_bases[$this->_depth][$key];
        } else {
            $num = count($this->_bases[$this->_depth]);
            $this->_bases[$this->_depth][$num] = $holder;
            $this->_heads[$this->_depth][0] =& $this->_bases[$this->_depth][$num];
        }
    }
    
    function backDepth($d) 
    {
        for ($i = $this->_depth; $i > $d; $i--) {
            unset($this->_bases[$i]);
            unset($this->_heads[$i]);
            unset($this->_types[$i]);
        }
        $this->_depth = $d;
    }
    
    function countIndent()
    {
        if (! isset($this->_indent_char_use)) {
            switch ($this->current()) {
            case ' ':
                $this->_indent_char_use = ' ';
                $this->_indent_char_not = "\t";
                break;
            case "\t":
                $this->_indent_char_use = "\t";
                $this->_indent_char_not = ' ';
                break;
            default:
                return 0;
            }
        }
        
        $d = 0;
        while ($this->current() == $this->_indent_char_use) {
            $this->goForward();
            $d += 1;
        }
        if ($this->current() == $this->_indent_char_not) {
            if ($this->current() == ' ') {
                return $this->raiseError('tab expected for indent');
            } else {
                return $this->raiseError('space expected for indent');
            }
        } else {
            return $d;
        }
    }

    function getScalar() 
    {
        $c = $this->current();
        if ($c === false) {
            $this->goNextLine();
            return null; //'';
        }
        if ($c == '\'' || $c == '"') {
            $ret = $this->_get_quoted($c);
            if (! is_string($ret)) {
                return $ret;
            }
            if ($this->procLineEnd()) {
                return $ret;
            } else {
                return $this->raiseError('garbage after quote');
            }
        } else {
            $ret = $this->substr();
            $pos = strpos($ret, '#');
            if ($pos !== false) {
                $ret = substr($ret, 0, $pos);
            } 
            $this->goNextLine();
            return $this->_getScalar_to_data(trim($ret));
        }
    }
    
    function _getScalar_to_data($str)
    {
        return ($str == '') ? null : $str;
    }
    
    function getScalarToColon() 
    {
        $c = $this->current();
        if ($c === false) {
            return false;
        }
        if ($c == '\'' || $c == '"') {
            $ret = $this->_get_quoted($c);
            if (! is_string($ret)) return $ret;
            
            $s = $this->substr();
            if ($s === false) return $this->raiseError('colon expected');
            $pos = strpos($s, ':');
            if ($pos === false) return $this->raiseError('colon expected');
            if (ltrim(substr($s, 0, $pos)) != '') return $this->raiseError('garbage after quote');
            $this->goForward($pos + 1);
            return $ret;
        } else {
            $s = $this->substr();
            if ($s === false) return $this->raiseError('colon expected'); // this will not happen?
            $pos = strpos($s, ':');
            if ($pos === false) return $this->raiseError('colon expected');
            $this->goForward($pos + 1);
            return trim(substr($s, 0, $pos));
        }
    }
    
    function skipSpaces()
    {
        $c = $this->current(); 
        while ($c == ' ' || $c == "\t") {
            $this->goForward();
            $c = $this->current(); 
        }
    }
    
    function skipEmtpyLines()
    {
        if ($this->isEnd()) return true;
        while ($this->procLineEnd()) {
            if ($this->isEnd()) return true;
        }
        return false;
    }
    
    function _get_quoted($q)
    {
        $s = $this->substr(); 
            
        $pos = strpos($s, $q, 1);
        if ($pos === false) {
            if ($q == '\'') {
                return $this->raiseError('unmatched single quote');
            } else {
                return $this->raiseError('unmatched double quote');
            }
        }

        $ret = substr($s, 1, $pos-1);
        $this->goForward($pos + 1);

        if ($q == '"') {
            $ret = $this->_strip_slash($ret);
        }
        return $ret;
    }
    
    function _strip_slash($str)
    {
        return str_replace(
                array('\0',   '\a',   '\b',   '\t', '\n', '\v',   '\f',   '\r', '\e',   '\ ', '\"', '\\\\'),
                array("\x00", "\x07", "\x08", "\t", "\n", "\x0b", "\x0c", "\r", "\x1b", ' ',  '"' , '\\'),
                $str
               );
    }
    
    function _add_slash($str)
    {
        return str_replace(
                array('\\',   '"',  "\x1b", "\r", "\xc", "\xb", "\n", "\t", "\x8", "\x7", "\x0"),
                array('\\\\', '\"', '\e',   '\r', '\f',  '\v',  '\n', '\t', '\b',  '\a',  '\0'),
                $str
               );
    }

    function _is_assoc($arr)
    {
        $keys = array_keys($arr);
        foreach ($keys as $key) {
            if (! is_int($key)) return true;
        }
        return false;
    }
    
/* --- for error handling --- */

    function raiseError($msg)
    {
        return parent::raiseError($msg.' in line '.($this->_line+1).' pos '.($this->_position+1));
    }
    
}

/**
* BogoYaml_Error
*/
class BogoYaml_Error extends PEAR_Error 
{
    var $error_message_prefix = '<b>BogoYaml</b>: ';
    
    function BogoYaml_Error($message = 'unknown error', $code = null,
                         $mode = null, $options = null, $userinfo = null)
    {
        parent::PEAR_Error($message, $code, $mode, $options, $userinfo);
    }
}


/**
* states abstruction for parsing YAML
* @abstruct
*/
class BogoYaml_ParserState {
    var $context;
    
    /* @param BogoYaml */
    function parse(&$c) {
        $this->context =& $c;
        return $this->parseLocal();
    }

    /* @abstruct */
    function parseLocal() {}
}

/**
* for the header before '---'
*/
class BogoYaml_ParserFirstLine extends BogoYaml_ParserState {
    function parseLocal() 
    {
        if ($this->context->skipEmtpyLines()) return true;
        
        if ($this->context->substr(3) == '---') {
            $this->context->goForward(3);
        } else {
            return 'garbage before document header';
        }

        if (! $this->context->procLineEnd()) {
            return 'garbage after document header';
        }

        return new BogoYaml_ParserLineHead();
    }
}

/**
* for head of each lines
*/
class BogoYaml_ParserLineHead extends BogoYaml_ParserState {
    function parseLocal() 
    {
        if ($this->context->skipEmtpyLines()) return true;

        $d = $this->context->countIndent();
        if (! is_int($d)) return $d;

        $this->context->backDepth($d);
        if ($this->context->current() == '-') {
            return new BogoYaml_ParserArray();
        } else {
            return new BogoYaml_ParserAssoc();
        }
    }
}

/**
* for indexed array
*/
class BogoYaml_ParserArray extends BogoYaml_ParserState {
    function parseLocal() 
    {
        $r = $this->context->addBase(BOGOYAML_ARRAY);
        if ($r !== true) return $r;

        $this->context->goForward();
        $this->context->skipSpaces();

        $str = $this->context->getScalar();
        if (! is_string($str) && $str !== null) return $str;
        
        $this->context->add($str);

        return new BogoYaml_ParserLineHead();
    }
}

/**
* for associative array
*/
class BogoYaml_ParserAssoc extends BogoYaml_ParserState {
    function parseLocal()
    {
       
        $r = $this->context->addBase(BOGOYAML_ASSOC);
        if ($r !== true) return $r;
        $key = $this->context->getScalarToColon();
        if (! is_string($key)) return $key;

        $this->context->skipSpaces();

        $str = $this->context->getScalar();
        if (! is_string($str) && $str !== null) return $str;
        $this->context->add($str, $key);

        return new BogoYaml_ParserLineHead();
    }
}
