/*
 * Copyright (c) 2001-2002 TAKIZAWA Takashi <taki@cyber.email.ne.jp>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "buffer.h"
#include "byte.h"
#include "error.h"
#include "exit.h"
#include "getln.h"
#include "str.h"
#include "seek.h"
#include "stralloc.h"
#include "strerr.h"

#include "auto_users.h"
#include "auto_qmailhome.h"

#include <pwd.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <unistd.h>
#include <fcntl.h>

void stdout_puts(char *x1, char *x2, char *x3, char *x4)
{
  if (x1) buffer_puts(buffer_1,x1);
  if (x2) buffer_puts(buffer_1,x2);
  if (x3) buffer_puts(buffer_1,x3);
  if (x4) buffer_puts(buffer_1,x4);
  buffer_flush(buffer_1);
}

/* splitline
 *   return value:
 *     1: skip
 *     0: success
 *    -1: out of memory
 */
int splitline(stralloc line, stralloc *user, stralloc *password, stralloc *envstr)
{
  char *x;
  char ch;
  int tab;
  int len = line.len;

  x = line.s;

  if (x[0] == '#' || x[0] == '\n')
    return 1;

  while (len) {
    ch = x[len - 1];
    if (ch != '\n') if (ch != ' ') if (ch != '\t') break;
    --len;
  }

  tab = byte_chr(x,len,'\t');
  if (tab == len) return 1;
  if (!stralloc_copyb(user,x,tab)) return -1;
  if (!stralloc_0(user)) return -1;
  x += tab + 1; len -= tab + 1;
  
  tab = byte_chr(x,len,'\t');
  if (!stralloc_copyb(password,x,tab)) return -1;
  if (!stralloc_0(password)) return -1;
  x += tab; len -= tab;

  if (!stralloc_copyb(envstr,x,len)) return -1;
  if (!stralloc_0(envstr)) return -1;

  return 0;
}

/* checkpwduser()
 *   return value;
 *     1: success
 *     0: unknown account
 */
int checkpwduser(char *user) {
  struct passwd *pw;

  pw = getpwnam(user);
  if (pw)
    return 1;
  else
    return 0;
}

/* getpwname()
 *   return:
 *     1: success
 *     0: unknown user
 *    -1: user is root
 */
int getpwname(char **name)
{
  uid_t pwuid;
  struct passwd *pw;

  pwuid = getuid();
  if (!pwuid)
    return -1;
  pw = getpwuid(pwuid);
  if (!pw)
    return 0;
  else
    *name = pw->pw_name;
  return 1;
}

/* chpwdbdir()
 *  return:
 *    1: real user
 *    0: domain owner
 *   -1: unable to chdir authdb
 */
int chpwdbdir(int realuser,char *name,char **dir)
{
  struct stat buf;
  static char *pwdusers = "pwdusers";

  if (chdir(auto_qmailhome) == -1)
    return -1;
  if (chdir("authdb") == -1)
    return -1;

  if (realuser && ! byte_equal(name,str_len(auto_realdomainowner)+1,auto_realdomainowner)) {
    *dir = pwdusers;
    return -1;
  }

  if (stat(name,&buf) == -1) {
    *dir = pwdusers;
    return (chdir(pwdusers) == -1) ? -1 : 1;
  } else if (realuser) {
    *dir = pwdusers;
    return (chdir(pwdusers) == -1) ? -1 : 0;
  } else {
    *dir = name;
    return (chdir(name) == -1) ? -1 : 0;
  }
}
