/*
 * Copyright (c) 2001-2002 TAKIZAWA Takashi <taki@cyber.email.ne.jp> 
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *    
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "byte.h"
#include "error.h"
#include "exit.h"
#include "fmt.h"
#include "getln.h"
#include "seek.h"
#include "str.h"
#include "stralloc.h"
#include "strerr.h"

#include "auto_home.h"
#include "auto_users.h"
#include "auto_qmailhome.h"

#include <pwd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <fcntl.h>
#include <stdio.h>

#include "sgetopt.h"

#define FATAL "vida-assign: fatal: "
#define ADD    1
#define DELETE 2

stralloc userline = {0};
stralloc userextline = {0};
stralloc dotline = {0};
stralloc line = {0};
int flaglock = 0;

static void usage(void)
{
  strerr_die4x(100,"Usage: vida-assign [-a|-d] [-mM] [-u virtualuser]\n",
	       "COMMAND\n  -a  add virtual user to users/assign\n",
	       "  -d  delete virtual user from users/assign\n",
	       "OPTION\n  -m  execute qmail-newu(default)\n  -M  don't execute qmail-newu\n");
}

static void dolock(void)
{
  if (link("users/assign","users/assign.lock") == -1)
     strerr_die2sys(111,FATAL,"unable to link users/assign to users/assign.lock : ");
  flaglock = 1;
}

static void unlock(void)
{
  if (flaglock) {
    unlink("users/assign.lock");
    flaglock = 0;
  }
}

static void nomem(void)
{
  unlock();
  strerr_die2x(111,FATAL,"out of memory");
}

static void die_write(void)
{
  unlock();
  strerr_die2sys(111,FATAL,"unable to write users/assign: ");
}

static void die_open(char *fn)
{
  unlock();
  strerr_die4sys(111,FATAL,"unable to open ",fn," : ");
}

static struct passwd *get_pw(void)
{
  uid_t pwuid;
  struct passwd *pw;

  pwuid = getuid();
  if (!pwuid)
    strerr_die2x(100,FATAL,"you must change user to the owner of domain");
  pw = getpwuid(pwuid);
  if (!pw)
    strerr_die2x(111,FATAL,"unable to read /etc/passwd");
  return pw;
}

static void makeassignline(char *vdomainowner,char *vuser,char *uid,char *gid,char *homedir,int vdomain,int complete)
{
  if (!stralloc_copys(&userline,"=")) nomem();
  if (vdomain) {
    if (!stralloc_cats(&userline,vdomainowner)) nomem();
    if (!stralloc_append(&userline,"-")) nomem();
  }
  if (!stralloc_cats(&userline,vuser)) nomem();
  if (!stralloc_append(&userline,":")) nomem();
  if (!stralloc_cats(&userline,vdomainowner)) nomem();
  if (!stralloc_append(&userline,":")) nomem();
  if (complete) {
    if (!stralloc_cats(&userline,uid)) nomem();
    if (!stralloc_append(&userline,":")) nomem();
    if (!stralloc_cats(&userline,gid)) nomem();
    if (!stralloc_append(&userline,":")) nomem();
    if (!stralloc_cats(&userline,homedir)) nomem();
    if (!stralloc_append(&userline,"/")) nomem();
    if (!stralloc_cats(&userline,vuser)) nomem();
    if (!stralloc_catb(&userline,":::\n",4)) nomem();
  }

  if (!stralloc_copys(&userextline,"+")) nomem();
  if (vdomain) {
    if (!stralloc_cats(&userextline,vdomainowner)) nomem();
    if (!stralloc_append(&userextline,"-")) nomem();
  }
  if (!stralloc_cats(&userextline,vuser)) nomem();
  if (!stralloc_catb(&userextline,"-:",2)) nomem();
  if (!stralloc_cats(&userextline,vdomainowner)) nomem();
  if (!stralloc_append(&userextline,":")) nomem();
  if (complete) {
    if (!stralloc_cats(&userextline,uid)) nomem();
    if (!stralloc_append(&userextline,":")) nomem();
    if (!stralloc_cats(&userextline,gid)) nomem();
    if (!stralloc_append(&userextline,":")) nomem();
    if (!stralloc_cats(&userextline,homedir)) nomem();
    if (!stralloc_append(&userextline,"/")) nomem();
    if (!stralloc_cats(&userextline,vuser)) nomem();
    if (!stralloc_catb(&userextline,":-::\n",5)) nomem();
  }
  if (!stralloc_copyb(&dotline,".\n",2)) nomem();
}

static void addassign()
{
  int fd;
  seek_pos pos;
  char buf[2];

  dolock();

  fd = open("users/assign",O_RDWR | O_NDELAY);
  if (fd == -1) 
    die_open("users/assign");

  seek_end(fd);
  pos = seek_cur(fd);
  while (pos) {
    pos--;
    seek_set(fd,pos);
    if (read(fd,buf,1) == -1)
      die_write();
    if (*buf != '.' && *buf != '\n') {
      pos++;
      seek_set(fd,pos);
      if (write(fd,"\n",1) == -1)
	die_write();
      pos++;
      break;
    }
  }
  seek_set(fd,pos);

  if (write(fd,userline.s,userline.len) == -1)
    die_write();
  if (write(fd,userextline.s,userextline.len) == -1)
    die_write();
  if (write(fd,dotline.s,dotline.len) == -1)
    die_write();
  if (fsync(fd) == -1)
    die_write();
  close(fd);
  unlock();
  return;
} 

static void delassign()
{
  int fd;
  int fdtemp;
  int match = 1;
  char inbuf[BUFFER_INSIZE];
  buffer ssin;

  dolock();

  fd = open("users/assign",O_RDONLY | O_NDELAY);
  if (fd == -1)
    die_open("users/assign");
  buffer_init(&ssin,read,fd,inbuf,sizeof inbuf);

  fdtemp = open("users/assign.tmp",O_WRONLY | O_NDELAY | O_CREAT,0644);
  if (fdtemp == -1)
    die_open("users/assign.tmp");

  while (match) {
    if (getln(&ssin,&line,&match,'\n') == -1)
      strerr_die2sys(111,FATAL,"unable to read users/assign: ");

    if (byte_equal(line.s,userline.len,userline.s))
      continue;
    if (byte_equal(line.s,userextline.len,userextline.s))
      continue;
    if (byte_equal(line.s,dotline.len,dotline.s))
      continue;
    if (write(fdtemp,line.s,line.len) == -1)
      die_write();
  }
  if (write(fdtemp,dotline.s,dotline.len) == -1)
    die_write();
  if (fsync(fdtemp) == -1)
    die_write();
  close(fdtemp);
  close(fd);
  if (rename("users/assign.tmp","users/assign"))
    strerr_die2sys(111,FATAL,"unable to rename users/assign.tmp to users/assign:");

  unlock();
} 

static int exist(int create)
{
  int fd;
  int match = 1;
  int count = 0;
  char inbuf[BUFFER_INSIZE];
  buffer ssin;
  stralloc line = {0};

  if (create)
    fd = open("users/assign",O_RDONLY | O_NDELAY | O_CREAT,0644);
  else  
    fd = open("users/assign",O_RDONLY | O_NDELAY);
  if (fd == -1)
    die_open("users/assign");

  buffer_init(&ssin,read,fd,inbuf,sizeof inbuf);

  while (match) {
    if (getln(&ssin,&line,&match,'\n') == -1)
      strerr_die2sys(111,FATAL,"unable to read users/assign: ");
    if (byte_equal(line.s,userline.len,userline.s))
      { count++; break;}
    if (byte_equal(line.s,userextline.len,userextline.s))
      { count++; break;}
  }
  close(fd);

  return count;
} 

static void qmailnewu()
{
  stralloc path = {0};

  if (!stralloc_copys(&path,auto_home)) nomem();
  if (!stralloc_cats(&path,"/bin/qmail-newu")) nomem();
  if (!stralloc_0(&path)) nomem();
  if (execlp(path.s,"qmail-newu",NULL) == -1)
    strerr_die4sys(111,FATAL,"unable to execute ",path.s,": ");
}

int main(int argc,char **argv)
{
  int opt;
  int act = 0;
  int make = 1;
  int vdomain = 0;
  char *user = NULL;
  char *domainowner;
  char uid[FMT_ULONG];
  char gid[FMT_ULONG];
  struct passwd *pw;

  umask(033);

  while ((opt = getopt(argc,argv,"admMu:")) != opteof)
    switch(opt) {
      case 'a': act = ADD; break;
      case 'd': act = DELETE; break;
      case 'm': make = 1; break;
      case 'M': make = 0; break;
      case 'u': user = optarg; break;
      default: usage();
    }
  argc -= optind;
  argv += optind;

  if (!act || !user)
    usage();

  pw = get_pw();
  domainowner = pw->pw_name;

  if (byte_diff(domainowner,str_len(auto_realdomainowner)+1,auto_realdomainowner))
    vdomain = 1;

  uid[fmt_ulong(uid,(unsigned long) pw->pw_uid)] = 0;
  gid[fmt_ulong(gid,(unsigned long) pw->pw_gid)] = 0;

  if (chdir(auto_qmailhome) == -1)
    strerr_die4sys(111,FATAL,"unable to chdir to ",auto_qmailhome,": ");

  switch (act) {
    case ADD:
      makeassignline(domainowner,user,uid,gid,pw->pw_dir,vdomain,0);
      if (exist(1))
	strerr_die3x(100,FATAL,user," aleady exists in users/assign.");
      makeassignline(domainowner,user,uid,gid,pw->pw_dir,vdomain,1);
      addassign();
      break;
    case DELETE:
      makeassignline(domainowner,user,uid,gid,pw->pw_dir,vdomain,0);
      if (!exist(0))
	strerr_die3x(100,FATAL,user," does not exist in users/assign.");
      delassign();
      break;
  }

  if (make)
    qmailnewu();

  _exit(0);
}
