/*
 * Copyright (c) 2001-2002 TAKIZAWA Takashi <taki@cyber.email.ne.jp>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "auto_home.h"
#include "buffer.h"
#include "byte.h"
#include "error.h"
#include "exit.h"
#include "getln.h"
#include "str.h"
#include "seek.h"
#include "stralloc.h"
#include "strerr.h"

#include "pwdb.h"

#include <pwd.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <unistd.h>
#include <fcntl.h>
#include <stdio.h>

#include "sgetopt.h"


#define FATAL "vida-passwd: fatal: "
#define MESSAGE "vida-passwd: "
#define WARN "vida-passwd: "

#define ADD    1
#define DELETE 2
#define CHANGE 3

#define MINPASSWD 7

stralloc passwdline = {0};
int flaglock = 0;

static void usage(void)
{
  strerr_die6x(100,"Usage: vida-passwd -a [-rRmM] -u user [-p password] [-e envstr]\n",
	       "       vida-passwd -c [-rRmM] -u user [-p password]\n",
	       "       vida-passwd -d [-rRmM] -u user\n",
	       "COMMAND\n  -a  add user\n  -c  change password\n  -d  delete user\n",
	       "OPTION\n  -m  execute vida-pwdbmake(default)\n  -M  don't execute vida-pwdbmake\n",
	       "  -r  treat as the real user\n  -R  don't treat as the real user(default)\n");
}

static void dolock(void)
{
  if (link("passwd","passwd.lock") == -1)
     strerr_die2sys(111,FATAL,"unable to link passwd to passwd.lock : ");
  flaglock = 1;
}

static void unlock(void)
{
  if (flaglock) {
    unlink("passwd.lock");
    flaglock = 0;
  }
}

static void nomem(void)
{
  unlock();
  strerr_die2x(111,FATAL,"out of memory");
}

static void die_open(char *fn)
{
  unlock();
  strerr_die4sys(111,FATAL,"unable to open ",fn,": ");
}

static void die_read(char *fn)
{
  unlock();
  strerr_die4sys(111,FATAL,"unable to read ",fn,": ");
}

static void die_write(char *fn)
{
  unlock();
  strerr_die4sys(111,FATAL,"unable to write ",fn,": ");
}

static int makepasswdline(char *user,char *passwd,char *envstr)
{
  int r = 0;
  if (!stralloc_copys(&passwdline,user)) r = -1;
  if (!stralloc_append(&passwdline,"\t")) r = -1;
  if (!stralloc_cats(&passwdline,passwd)) r = -1;
  if (envstr) {
    if (!stralloc_append(&passwdline,"\t")) r = -1;
    if (!stralloc_cats(&passwdline,envstr)) r = -1;
  }
  if (!stralloc_append(&passwdline,"\n")) r = -1;
  return r;
}

static int user_equal(char *line, char *user)
{
  int len;
  len = str_len(user);
  if (byte_equal(line,len,user) && line[len] == '\t')
    return 1;
  else
    return 0;
}

static int exist(char *user)
{
  int fd;
  int match = 1;
  int count = 0;
  char inbuf[BUFFER_INSIZE];
  buffer ssin;
  stralloc line = {0};

  fd = open("passwd",O_RDONLY | O_NDELAY);
  if (fd == -1)
    die_open("passwd");
  buffer_init(&ssin,read,fd,inbuf,sizeof inbuf);

  while (match) {
    if (getln(&ssin,&line,&match,'\n') == -1)
      die_read("passwd");
    if (user_equal(line.s,user)) {
      count++; break;
    }
  }
  close(fd);

  return count;
} 

static void addpasswd(char *user, char *password, char *envstr)
{
  int fd;
  seek_pos pos;

  if (makepasswdline(user,password,envstr) == -1)
    nomem();

  fd = open("passwd",O_RDWR | O_NDELAY | O_CREAT,0600);
  if (fd == -1)
    die_open("passwd");

  seek_end(fd);
  pos = seek_cur(fd);
  if (pos)
    pos--;

  if (write(fd,passwdline.s,passwdline.len) == -1)
    die_write("passwd");
  if (fsync(fd) == -1)
    die_write("passwd");
  close(fd);
  return;
} 

static void chdelpasswd(int act, char *user, char *oldpassword, char *password)
{
  int fd;
  int fdtemp;
  int match = 1;
  int count = 0;
  int r;
  char inbuf[BUFFER_INSIZE];
  buffer ssin;
  stralloc line = {0};
  stralloc s_user = {0};
  stralloc s_password = {0};
  stralloc s_envstr = {0}; /* \tENV1=env1,ENV2=env2,,,, */

  fd = open("passwd",O_RDONLY | O_NDELAY);
  if (fd == -1)
    die_open("passwd");
  buffer_init(&ssin,read,fd,inbuf,sizeof inbuf);

  fdtemp = open("passwd.tmp",O_WRONLY | O_NDELAY | O_CREAT,0600);
  if (fdtemp == -1)
    die_open("passwd.tmp");

  while (match) {
    if (getln(&ssin,&line,&match,'\n') == -1)
      die_read("passwd");
    if (user_equal(line.s,user)) {
      r = splitline(line,&s_user,&s_password,&s_envstr);
      if (r == 1)
	continue;
      else if (r == -1)
	nomem();

      if (act == CHANGE) {
	if (byte_equal(s_password.s,s_password.len,oldpassword)) {
	  if (makepasswdline(s_user.s,password,s_envstr.s) == -1)
	    nomem();
	  if (write(fdtemp,passwdline.s,passwdline.len) == -1)
	    die_write("passwd.tmp");
	}
	else {
	  unlink("passwd.tmp");
	  unlock();
	  strerr_die4x(1,WARN,"password for ",user," is incorrect.");
	}
      }
      count++;
      continue;
    }

    if (write(fdtemp,line.s,line.len) == -1)
      die_write("passwd.tmp");
  }
  if (fsync(fdtemp) == -1)
    die_write("passwd.tmp");
  close(fdtemp);
  close(fd);

  if (count > 0) {
    if (rename("passwd.tmp","passwd") == -1) {
      unlock();
      strerr_die2sys(111,FATAL,"unable to move passwd.tmp to passwd: ");
    }
  }
  else {
    unlink("passwd.tmp");
    unlock();
    strerr_die3x(4,FATAL,user," does not exist in passwd.");
  }
} 

static char *getpassword(char *output)
{
  stralloc password = {0};
  char *buf;

  buf = getpass (output);
  if (!stralloc_copys(&password,buf)) nomem();
  if (!stralloc_0(&password)) nomem();
  
  return password.s;
}

static char *getnewpassword()
{
  char *password = NULL;
  char *passwordre = NULL;

  password = getpassword("New password: ");
  passwordre = getpassword("Retype new password: ");
  if (byte_diff(password,str_len(password)+1,passwordre)) {
    unlock();
    strerr_die2x(2,WARN,"Sorry, new passwords do not match");
  }
  if (str_len(password) < MINPASSWD) {
    unlock();
    strerr_die2x(3,WARN,"Bad password. It is too short");
  }
  return password;
}

static void pwdbmake(int realuser)
{
  int r;
  struct stralloc path = {0};

  if (!stralloc_copys(&path,auto_home)) nomem();
  if (!stralloc_cats(&path,"/bin/vida-pwdbmake")) nomem();
  if (!stralloc_0(&path)) nomem();
  if (realuser)
    r = execlp(path.s,"vida-pwdbmake","-r",NULL);
  else
    r = execlp(path.s,"vida-pwdbmake",NULL);
  if (r == -1)
    strerr_die4sys(111,FATAL,"unable to execute ",path.s,": ");
}

/* exit code:
 *   0: success
 *   1: password is incorrect
 *   2: passwords do not match
 *   3: password is too short
 *   4: user does not exist
 *   5: user already exists
 * 111: system error
 */

int main(int argc,char **argv)
{
  char *user = NULL;
  char *password = NULL;
  char *oldpassword = NULL;
  char *envstr = NULL;
  char *pwname;
  char *dir;
  int r;
  int opt;
  int act = 0;
  int realuser = 0;
  int make = 1;

  umask(077);

  while ((opt = getopt(argc,argv,"acdmMrRu:p:P:e:")) != opteof)
    switch(opt) {
      case 'a': act = ADD; break;
      case 'c': act = CHANGE; break;
      case 'd': act = DELETE; break;
      case 'm': make = 1; break;
      case 'M': make = 0; break;
      case 'r': realuser = 1; break;
      case 'R': realuser = 0; break;
      case 'u': user = optarg; break;
      case 'p': password = optarg; break;
      case 'P': oldpassword = optarg; break;
      case 'e': envstr = optarg; break;
      default: usage();
    }
  argc -= optind;
  argv += optind;

  r = getpwname(&pwname);
  if (r == 0)
    strerr_die2x(111,FATAL,"unable to read /etc/passwd");
  else if (r == -1)
    strerr_die2x(100,FATAL,"you must change user to the owner of domain");

  if ((r = chpwdbdir(realuser,pwname,&dir)) == 1) {
    user = pwname;
    act = exist(user) ? CHANGE : ADD;
  }
  else if (r == -1)
    strerr_die4sys(111,FATAL,"unable to chdir authdb/",dir,": ");

  if (!user)
    usage();

  if (realuser && !checkpwduser(user))
    strerr_die3x(111,FATAL,"unknown user ",user);

  dolock();
  switch(act) {
    case ADD:
      puts_x3("Adding user ",user,"\n");
      if (exist(user)) {
	unlock();
	strerr_die4x(5,WARN,"user ",user," already exists.");
      }
      if (!password)
	password = getnewpassword();
      addpasswd(user,password,envstr);
      puts_x4(MESSAGE,"user ",user," was added\n");
      break;
    case DELETE:
      puts_x3("Deleting user ",user,"\n");
      chdelpasswd(DELETE,user,NULL,NULL);
      puts_x4(MESSAGE,"user ",user," was deleted\n");
      break;
    case CHANGE:
      puts_x3("Changing password for user ",user,"\n");
      if (!exist(user)) {
	unlock();
	strerr_die3x(4,WARN,user," does not exist.");
      }
      if (!oldpassword)
	oldpassword = getpassword("Current password: ");
      if (!password)
	password = getnewpassword();
      chdelpasswd(CHANGE,user,oldpassword,password);
      puts_x4(MESSAGE,"Password for user ",user," was changed\n");
      break;
  }
  unlock();

  if (make)
    pwdbmake(realuser);

  exit (0);
}
