/*
 * Copyright (c) 2001-2002 TAKIZAWA Takashi <taki@cyber.email.ne.jp>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "cdb_make.h"
#include "env.h"
#include "exit.h"
#include "error.h"
#include "strerr.h"
#include "open.h"
#include "prot.h"

#include "auto_uids.h"
#include "auto_users.h"
#include "auto_qmailhome.h"

#include <pwd.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <fcntl.h>
#include <unistd.h>

#define FATAL "pwdbinit: fatal: "
#define WARN  "pwdbinit: "

static void die_write(char *dir,char *fn)
{
  strerr_die6sys(111,FATAL,"unable to write to authdb/",dir,"/",fn,": ");
}

struct passwd *get_pw(char *name)
{
  struct passwd *pw;
  pw = getpwnam(name);
  if (!pw) {
    if (errno == error_txtbsy)
      strerr_die2sys(111,FATAL,"unable to read /etc/passwd: ");
    else
      strerr_die4x(111,WARN,"user ",name," doesn't exist in /etc/passwd");
  }
  return pw;
}

static void setuidgid(int uid, int gid)
{
  if (prot_gid(gid) == -1)
    strerr_die4sys(111,FATAL,"unable to setgid to ",auto_vida,": ");
  if (prot_uid(uid) == -1)
    strerr_die4sys(111,FATAL,"unable to setuid to ",auto_authdb,": ");
}

static void authdbdirmake(uid_t owner,uid_t group)
{
  if (chdir(auto_qmailhome) == -1)
    strerr_die4sys(111,FATAL,"unable to chdir to ",auto_qmailhome,": ");
  if (chdir("authdb") == 0)
    return;
  if (mkdir("authdb",0700) == -1)
    strerr_die4sys(111,FATAL,"unable to mkdir ",auto_qmailhome,"/authdb: ");
  if (chown("authdb",owner,group) == -1)
    strerr_die6sys(111,FATAL,"unable to chown ",auto_qmailhome,"/authdb to",auto_authdb,": ");
  if (chdir("authdb") == -1)
    strerr_die4sys(111,FATAL,"unable to chdir to ",auto_qmailhome,"/authdb: ");
}

static void pwdbdirmake(char *pwdbdir)
{
  if (chdir(auto_qmailhome) == -1)
    strerr_die4sys(111,FATAL,"unable to chdir to ",auto_qmailhome,": ");
  if (chdir("authdb") == -1)
    strerr_die4sys(111,FATAL,"unable to chdir to ",auto_qmailhome,"/authdb: ");
  if (chdir(pwdbdir) == 0)
    strerr_die4x(111,WARN,"password database entry for ",pwdbdir," already exists");
  if (mkdir(pwdbdir,0700) == -1)
    strerr_die4sys(111,FATAL,"unable to mkdir authdb/",pwdbdir,": ");
  if (chdir(pwdbdir) == -1)
    strerr_die4sys(111,FATAL,"unable to chdir to authdb/",pwdbdir,": ");
}

static void makedb(char *pwdbdir)
{
  struct cdb_make c;
  int fd;

  fd = open_trunc("passwd");
  if (fd == -1)
    strerr_die4sys(111,FATAL,"unable to create authdb/",pwdbdir,"/passwd: ");
  if (write(fd,"#user\tpassword\tenviron\n",23) == -1)
    { close(fd); die_write(pwdbdir,"passwd"); }
  if (fsync(fd) == -1) die_write(pwdbdir,"passwd");
  if (close(fd) == -1) die_write(pwdbdir,"passwd");

  fd = open_trunc("cdb");
  if (fd == -1)
    strerr_die4sys(111,FATAL,"unable to create authdb/",pwdbdir,"/cdb: ");
  if (cdb_make_start(&c,fd) == -1) die_write(pwdbdir,"cdb");
  if (cdb_make_finish(&c) == -1) die_write(pwdbdir,"cdb");
  if (fsync(fd) == -1) die_write(pwdbdir,"cdb");
  if (close(fd) == -1) die_write(pwdbdir,"cdb");
}

int main(int argc,char **argv)
{
  char *domainowner;
  struct passwd *pw;
  uid_t pw_uid;

  pw_uid = getuid();
  if (pw_uid)
    strerr_die2x(111,FATAL,"you must run as root: ");

  umask(077);

  authdbdirmake(auto_uidad,auto_gidvida);
  setuidgid(auto_uidad,auto_gidvida);

  domainowner = argv[1];
  if (domainowner) {
    pw = get_pw(domainowner);
    pwdbdirmake(domainowner);
    makedb(domainowner);
  }
  else {
    pwdbdirmake(auto_realdomainowner);
    makedb(auto_realdomainowner);
    pwdbdirmake("pwdusers");
    makedb("pwdusers");
  }

  _exit(0);
}
