/*
 * Copyright (c) 2001-2002 TAKIZAWA Takashi <taki@cyber.email.ne.jp>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "buffer.h"
#include "byte.h"
#include "cdb_make.h"
#include "error.h"
#include "exit.h"
#include "getln.h"
#include "open.h"
#include "readwrite.h"
#include "str.h"
#include "strerr.h"
#include "stralloc.h"

#include "pwdb.h"

#include <sys/stat.h>
#include <sys/types.h>
#include <unistd.h>
#include <stdio.h>

#define FATAL "pwdbmake: fatal: "
#define MESSAGE "pwdbmake: "

char *pwdbdir;

static void nomem(void)
{
  strerr_die2x(111,FATAL,"out of memory");
}
static void die_bad(stralloc line)
{
  if (!stralloc_0(&line)) nomem();
  strerr_die3x(100,FATAL,"unable to parse this line: ",line.s);
}
static void die_read()
{
  strerr_die4sys(111,FATAL,"unable to read to authdb/",pwdbdir,"/passwd: ");
}
static void die_write()
{
  strerr_die4sys(111,FATAL,"unable to write to authdb/",pwdbdir,"/cdb.tmp: ");
}
static void die_open()
{
  strerr_die4sys(111,FATAL,"unable to open to authdb/",pwdbdir,"/passwd: ");
}

static void pwdbmake()
{
  char *x;
  int len;
  int fd;
  int fdtemp;
  int i;
  int r;
  char ch;
  char inbuf[BUFFER_INSIZE];
  buffer ssin;
  int match = 1;
  stralloc user = {0};
  stralloc password = {0};
  stralloc envstr = {0}; /* \tENV1=env1,ENV2=env2,,,, */
  stralloc line = {0};
  stralloc data = {0};
  stralloc key = {0};
  struct cdb_make c;

  fd = open_read("passwd");
  if (fd == -1)
    die_open();
  buffer_init(&ssin,read,fd,inbuf,sizeof inbuf);

  fdtemp = open_trunc("cdb.tmp");
  if (fdtemp == -1)
    strerr_die4sys(111,FATAL,"unable to create authdb/",pwdbdir,"/cdb.tmp: ");
  if (cdb_make_start(&c,fdtemp) == -1)
    die_write();

  while (match) {
    if (getln(&ssin,&line,&match,'\n') == -1)
      die_read();

    if (!line.len)
      break;
    r = splitline(line,&user,&password,&envstr);
    if (r == 1)
      continue;
    else if (r == -1)
      nomem();

    if (!stralloc_copys(&key,user.s)) nomem();
    if (!stralloc_copy(&data,&password)) nomem();

    x = envstr.s; len = envstr.len - 1;
    while (len)
      switch(*x) {
        case ',':
        case '\t':
          i = byte_chr(x,len,'=');
          if (i == len) die_bad(line);
          if (!stralloc_catb(&data,"+",1)) nomem();
          if (!stralloc_catb(&data,x + 1,i)) nomem();
          x += i + 1; len -= i + 1;
          if (!len) die_bad(line);
          ch = *x;
          x += 1; len -= 1;
          i = byte_chr(x,len,ch);
          if (i == len) die_bad(line);
          if (!stralloc_catb(&data,x,i)) nomem();
          if (!stralloc_0(&data)) nomem();
          x += i + 1; len -= i + 1;
          break;
        default:
          die_bad(line);
      }

    if (cdb_make_add(&c,key.s,key.len,data.s,data.len) == -1) die_write();
  }

  if (cdb_make_finish(&c) == -1) die_write();
  if (fsync(fdtemp) == -1) die_write();
  if (close(fdtemp) == -1) die_write();
  close(fd);
}

static void renamecdb()
{
  if (rename("cdb.tmp","cdb"))
    strerr_die2sys(111,FATAL,"unable to move cdb.tmp to cdb: ");
}

int main(int argc,char **argv)
{
  char *pwname;
  char *dir;
  char realuser = 0;
  int r;

  umask(077);
  if (argv[1])
    realuser = (byte_equal(argv[1],2,"-r")) ? 1 : 0;
  r = getpwname(&pwname);
  if (r == 0)
    strerr_die2x(111,FATAL,"unable to read /etc/passwd");
  else if (r == -1)
    strerr_die2x(100,FATAL,"you must change user to owner of domain");

  r = chpwdbdir(realuser,pwname,&dir);
  if (r != -1) {
    pwdbmake();
    renamecdb();
    puts_x4(MESSAGE,"authdb/",dir,"/cdb was updated\n");
  }
  else
    strerr_die4sys(111,FATAL,"unable to chdir authdb/",dir,": ");

  _exit(0);
}
